import matplotlib.pyplot as plt
import numpy as np

# Data
models = ['ResNet32', 'ResNet44', 'ResNet56', 'MBNv2-x0.5', 'MBNv2-x0.75',
          'MBNv2-x1.0', 'MBNv2-x1.4', 'RepVGG A0', 'RepVGG A1', 'RepVGG A2']
baseline_acc = [70.16, 71.63, 72.63, 70.88, 73.61, 74.20, 75.98, 75.22, 76.12, 77.18]
calibrated_acc = [70.51, 72.63, 73.47, 71.80, 74.92, 75.44, 76.34, 76.22, 76.84, 77.45]
baseline_ted = [8.86, 8.81, 8.83, 8.87, 8.76, 8.73, 8.65, 8.74, 8.72, 8.70]
calibrated_ted = [8.84, 8.77, 8.71, 8.76, 8.63, 8.61, 8.62, 8.66, 8.58, 8.59]
baseline_mcs = [25.39, 25.97, 26.47, 25.50, 26.71, 27.09, 27.41, 26.84, 27.43, 27.99]
calibrated_mcs = [25.76, 26.28, 27.15, 26.36, 27.57, 27.83, 27.69, 27.54, 28.43, 28.17]
baseline_tree_kernel = [48.58, 49.79, 50.17, 49.08, 51.21, 51.47, 52.91, 52.01, 52.81, 53.04]
calibrated_tree_kernel = [49.13, 50.73, 51.43, 50.34, 52.49, 52.82, 53.24, 52.89, 53.81, 53.71]

# Create theta values for each axis
theta = np.linspace(0, 2 * np.pi, len(models), endpoint=False)

# Plot Radar Chart
fig, axs = plt.subplots(2, 2, figsize=(12, 12), subplot_kw={'polar': True})
axs = axs.ravel()

# Set x-axis labels
axs[0].set_xticks(theta)
axs[0].set_xticklabels(models, fontsize=10)

# Adjust spacing between subplots
plt.subplots_adjust(hspace=0.35)

# Customize grid lines
for ax in axs:
    ax.yaxis.grid(color='gray', linestyle='dashed', alpha=0.5)

# Set axis labels and limits
axs[0].set_title('Accuracy', fontsize=12, fontweight='bold', pad=20)
axs[0].set_ylim(70, 78)
axs[1].set_title('TED', fontsize=12, fontweight='bold', pad=20)
axs[1].set_ylim(8.5, 9)
axs[2].set_title('MCS', fontsize=12, fontweight='bold', pad=20)
axs[2].set_ylim(25, 29)
axs[3].set_title('Tree Kernel', fontsize=12, fontweight='bold', pad=20)
axs[3].set_ylim(48, 54)

# Customize plot styles
styles = ['-', '--']
colors = ['steelblue', 'darkorange']

# Plot lines
for i, ax in enumerate(axs):
    ax.plot(theta, baseline_acc, linestyle=styles[0], color=colors[0], linewidth=2, alpha=0.8)
    ax.fill(theta, baseline_acc, color=colors[0], alpha=0.2)
    ax.plot(theta, calibrated_acc, linestyle=styles[1], color=colors[1], linewidth=2, alpha=0.8)
    ax.fill(theta, calibrated_acc, color=colors[1], alpha=0.2)

# Add legend
axs[0].legend(['Baseline', 'Calibrated'], loc=(0.9, 0.95), fontsize=10)

# Add value labels on the plot
offset = 0.1
for ax in axs:
    for angle, label in zip(theta, models):
        x = angle - np.pi / 2
        y = ax.get_ylim()[1] + offset
        ax.text(x, y, label, fontsize=8, ha='center', va='center', rotation_mode='anchor')

plt.tight_layout()
plt.show()
