// SPDX-License-Identifier: GPL-2.0
/*
 * Request reply cache. This is currently a global cache, but this may
 * change in the future and be a per-client cache.
 *
 * This code is heavily inspired by the 44BSD implementation, although
 * it does things a bit differently.
 *
 * Copyright (C) 1995, 1996 Olaf Kirch <okir@monad.swb.de>
 */

#include <linux sunrpc="" svc_xprt.h="">
#include <linux slab.h="">
#include <linux vmalloc.h="">
#include <linux addr.h="" sunrpc="">
#include <linux highmem.h="">
#include <linux log2.h="">
#include <linux hash.h="">
#include <net checksum.h="">

#include "nfsd.h"
#include "cache.h"
#include "trace.h"

/*
 * We use this value to determine the number of hash buckets from the max
 * cache size, the idea being that when the cache is at its maximum number
 * of entries, then this should be the average number of entries per bucket.
 */
#define TARGET_BUCKET_SIZE	64

struct nfsd_drc_bucket {
	struct rb_root rb_head;
	struct list_head lru_head;
	spinlock_t cache_lock;
};

static struct kmem_cache	*drc_slab;

static int	nfsd_cache_append(struct svc_rqst *rqstp, struct kvec *vec);
static unsigned long nfsd_reply_cache_count(struct shrinker *shrink,
					    struct shrink_control *sc);
static unsigned long nfsd_reply_cache_scan(struct shrinker *shrink,
					   struct shrink_control *sc);

/*
 * Put a cap on the size of the DRC based on the amount of available
 * low memory in the machine.
 *
 *  64MB:    8192
 * 128MB:   11585
 * 256MB:   16384
 * 512MB:   23170
 *   1GB:   32768
 *   2GB:   46340
 *   4GB:   65536
 *   8GB:   92681
 *  16GB:  131072
 *
 * ...with a hard cap of 256k entries. In the worst case, each entry will be
 * ~1k, so the above numbers should give a rough max of the amount of memory
 * used in k.
 *
 * XXX: these limits are per-container, so memory used will increase
 * linearly with number of containers.  Maybe that's OK.
 */
static unsigned int
nfsd_cache_size_limit(void)
{
	unsigned int limit;
	unsigned long low_pages = totalram_pages() - totalhigh_pages();

	limit = (16 * int_sqrt(low_pages)) &lt;&lt; (PAGE_SHIFT-10);
	return min_t(unsigned int, limit, 256*1024);
}

/*
 * Compute the number of hash buckets we need. Divide the max cachesize by
 * the "target" max bucket size, and round up to next power of two.
 */
static unsigned int
nfsd_hashsize(unsigned int limit)
{
	return roundup_pow_of_two(limit / TARGET_BUCKET_SIZE);
}

static struct svc_cacherep *
nfsd_reply_cache_alloc(struct svc_rqst *rqstp, __wsum csum,
			struct nfsd_net *nn)
{
	struct svc_cacherep	*rp;

	rp = kmem_cache_alloc(drc_slab, GFP_KERNEL);
	if (rp) {
		rp-&gt;c_state = RC_UNUSED;
		rp-&gt;c_type = RC_NOCACHE;
		RB_CLEAR_NODE(&amp;rp-&gt;c_node);
		INIT_LIST_HEAD(&amp;rp-&gt;c_lru);

		memset(&amp;rp-&gt;c_key, 0, sizeof(rp-&gt;c_key));
		rp-&gt;c_key.k_xid = rqstp-&gt;rq_xid;
		rp-&gt;c_key.k_proc = rqstp-&gt;rq_proc;
		rpc_copy_addr((struct sockaddr *)&amp;rp-&gt;c_key.k_addr, svc_addr(rqstp));
		rpc_set_port((struct sockaddr *)&amp;rp-&gt;c_key.k_addr, rpc_get_port(svc_addr(rqstp)));
		rp-&gt;c_key.k_prot = rqstp-&gt;rq_prot;
		rp-&gt;c_key.k_vers = rqstp-&gt;rq_vers;
		rp-&gt;c_key.k_len = rqstp-&gt;rq_arg.len;
		rp-&gt;c_key.k_csum = csum;
	}
	return rp;
}

static void
nfsd_reply_cache_free_locked(struct nfsd_drc_bucket *b, struct svc_cacherep *rp,
				struct nfsd_net *nn)
{
	if (rp-&gt;c_type == RC_REPLBUFF &amp;&amp; rp-&gt;c_replvec.iov_base) {
		nfsd_stats_drc_mem_usage_sub(nn, rp-&gt;c_replvec.iov_len);
		kfree(rp-&gt;c_replvec.iov_base);
	}
	if (rp-&gt;c_state != RC_UNUSED) {
		rb_erase(&amp;rp-&gt;c_node, &amp;b-&gt;rb_head);
		list_del(&amp;rp-&gt;c_lru);
		atomic_dec(&amp;nn-&gt;num_drc_entries);
		nfsd_stats_drc_mem_usage_sub(nn, sizeof(*rp));
	}
	kmem_cache_free(drc_slab, rp);
}

static void
nfsd_reply_cache_free(struct nfsd_drc_bucket *b, struct svc_cacherep *rp,
			struct nfsd_net *nn)
{
	spin_lock(&amp;b-&gt;cache_lock);
	nfsd_reply_cache_free_locked(b, rp, nn);
	spin_unlock(&amp;b-&gt;cache_lock);
}

int nfsd_drc_slab_create(void)
{
	drc_slab = kmem_cache_create("nfsd_drc",
				sizeof(struct svc_cacherep), 0, 0, NULL);
	return drc_slab ? 0: -ENOMEM;
}

void nfsd_drc_slab_free(void)
{
	kmem_cache_destroy(drc_slab);
}

static int nfsd_reply_cache_stats_init(struct nfsd_net *nn)
{
	return nfsd_percpu_counters_init(nn-&gt;counter, NFSD_NET_COUNTERS_NUM);
}

static void nfsd_reply_cache_stats_destroy(struct nfsd_net *nn)
{
	nfsd_percpu_counters_destroy(nn-&gt;counter, NFSD_NET_COUNTERS_NUM);
}

int nfsd_reply_cache_init(struct nfsd_net *nn)
{
	unsigned int hashsize;
	unsigned int i;
	int status = 0;

	nn-&gt;max_drc_entries = nfsd_cache_size_limit();
	atomic_set(&amp;nn-&gt;num_drc_entries, 0);
	hashsize = nfsd_hashsize(nn-&gt;max_drc_entries);
	nn-&gt;maskbits = ilog2(hashsize);

	status = nfsd_reply_cache_stats_init(nn);
	if (status)
		goto out_nomem;

	nn-&gt;nfsd_reply_cache_shrinker.scan_objects = nfsd_reply_cache_scan;
	nn-&gt;nfsd_reply_cache_shrinker.count_objects = nfsd_reply_cache_count;
	nn-&gt;nfsd_reply_cache_shrinker.seeks = 1;
	status = register_shrinker(&amp;nn-&gt;nfsd_reply_cache_shrinker,
				   "nfsd-reply:%s", nn-&gt;nfsd_name);
	if (status)
		goto out_stats_destroy;

	nn-&gt;drc_hashtbl = kvzalloc(array_size(hashsize,
				sizeof(*nn-&gt;drc_hashtbl)), GFP_KERNEL);
	if (!nn-&gt;drc_hashtbl)
		goto out_shrinker;

	for (i = 0; i &lt; hashsize; i++) {
		INIT_LIST_HEAD(&amp;nn-&gt;drc_hashtbl[i].lru_head);
		spin_lock_init(&amp;nn-&gt;drc_hashtbl[i].cache_lock);
	}
	nn-&gt;drc_hashsize = hashsize;

	return 0;
out_shrinker:
	unregister_shrinker(&amp;nn-&gt;nfsd_reply_cache_shrinker);
out_stats_destroy:
	nfsd_reply_cache_stats_destroy(nn);
out_nomem:
	printk(KERN_ERR "nfsd: failed to allocate reply cache\n");
	return -ENOMEM;
}

void nfsd_reply_cache_shutdown(struct nfsd_net *nn)
{
	struct svc_cacherep	*rp;
	unsigned int i;

	unregister_shrinker(&amp;nn-&gt;nfsd_reply_cache_shrinker);

	for (i = 0; i &lt; nn-&gt;drc_hashsize; i++) {
		struct list_head *head = &amp;nn-&gt;drc_hashtbl[i].lru_head;
		while (!list_empty(head)) {
			rp = list_first_entry(head, struct svc_cacherep, c_lru);
			nfsd_reply_cache_free_locked(&amp;nn-&gt;drc_hashtbl[i],
									rp, nn);
		}
	}
	nfsd_reply_cache_stats_destroy(nn);

	kvfree(nn-&gt;drc_hashtbl);
	nn-&gt;drc_hashtbl = NULL;
	nn-&gt;drc_hashsize = 0;

}

/*
 * Move cache entry to end of LRU list, and queue the cleaner to run if it's
 * not already scheduled.
 */
static void
lru_put_end(struct nfsd_drc_bucket *b, struct svc_cacherep *rp)
{
	rp-&gt;c_timestamp = jiffies;
	list_move_tail(&amp;rp-&gt;c_lru, &amp;b-&gt;lru_head);
}

static noinline struct nfsd_drc_bucket *
nfsd_cache_bucket_find(__be32 xid, struct nfsd_net *nn)
{
	unsigned int hash = hash_32((__force u32)xid, nn-&gt;maskbits);

	return &amp;nn-&gt;drc_hashtbl[hash];
}

static long prune_bucket(struct nfsd_drc_bucket *b, struct nfsd_net *nn,
			 unsigned int max)
{
	struct svc_cacherep *rp, *tmp;
	long freed = 0;

	list_for_each_entry_safe(rp, tmp, &amp;b-&gt;lru_head, c_lru) {
		/*
		 * Don't free entries attached to calls that are still
		 * in-progress, but do keep scanning the list.
		 */
		if (rp-&gt;c_state == RC_INPROG)
			continue;
		if (atomic_read(&amp;nn-&gt;num_drc_entries) &lt;= nn-&gt;max_drc_entries &amp;&amp;
		    time_before(jiffies, rp-&gt;c_timestamp + RC_EXPIRE))
			break;
		nfsd_reply_cache_free_locked(b, rp, nn);
		if (max &amp;&amp; freed++ &gt; max)
			break;
	}
	return freed;
}

static long nfsd_prune_bucket(struct nfsd_drc_bucket *b, struct nfsd_net *nn)
{
	return prune_bucket(b, nn, 3);
}

/*
 * Walk the LRU list and prune off entries that are older than RC_EXPIRE.
 * Also prune the oldest ones when the total exceeds the max number of entries.
 */
static long
prune_cache_entries(struct nfsd_net *nn)
{
	unsigned int i;
	long freed = 0;

	for (i = 0; i &lt; nn-&gt;drc_hashsize; i++) {
		struct nfsd_drc_bucket *b = &amp;nn-&gt;drc_hashtbl[i];

		if (list_empty(&amp;b-&gt;lru_head))
			continue;
		spin_lock(&amp;b-&gt;cache_lock);
		freed += prune_bucket(b, nn, 0);
		spin_unlock(&amp;b-&gt;cache_lock);
	}
	return freed;
}

static unsigned long
nfsd_reply_cache_count(struct shrinker *shrink, struct shrink_control *sc)
{
	struct nfsd_net *nn = container_of(shrink,
				struct nfsd_net, nfsd_reply_cache_shrinker);

	return atomic_read(&amp;nn-&gt;num_drc_entries);
}

static unsigned long
nfsd_reply_cache_scan(struct shrinker *shrink, struct shrink_control *sc)
{
	struct nfsd_net *nn = container_of(shrink,
				struct nfsd_net, nfsd_reply_cache_shrinker);

	return prune_cache_entries(nn);
}
/*
 * Walk an xdr_buf and get a CRC for at most the first RC_CSUMLEN bytes
 */
static __wsum
nfsd_cache_csum(struct svc_rqst *rqstp)
{
	int idx;
	unsigned int base;
	__wsum csum;
	struct xdr_buf *buf = &amp;rqstp-&gt;rq_arg;
	const unsigned char *p = buf-&gt;head[0].iov_base;
	size_t csum_len = min_t(size_t, buf-&gt;head[0].iov_len + buf-&gt;page_len,
				RC_CSUMLEN);
	size_t len = min(buf-&gt;head[0].iov_len, csum_len);

	/* rq_arg.head first */
	csum = csum_partial(p, len, 0);
	csum_len -= len;

	/* Continue into page array */
	idx = buf-&gt;page_base / PAGE_SIZE;
	base = buf-&gt;page_base &amp; ~PAGE_MASK;
	while (csum_len) {
		p = page_address(buf-&gt;pages[idx]) + base;
		len = min_t(size_t, PAGE_SIZE - base, csum_len);
		csum = csum_partial(p, len, csum);
		csum_len -= len;
		base = 0;
		++idx;
	}
	return csum;
}

static int
nfsd_cache_key_cmp(const struct svc_cacherep *key,
			const struct svc_cacherep *rp, struct nfsd_net *nn)
{
	if (key-&gt;c_key.k_xid == rp-&gt;c_key.k_xid &amp;&amp;
	    key-&gt;c_key.k_csum != rp-&gt;c_key.k_csum) {
		nfsd_stats_payload_misses_inc(nn);
		trace_nfsd_drc_mismatch(nn, key, rp);
	}

	return memcmp(&amp;key-&gt;c_key, &amp;rp-&gt;c_key, sizeof(key-&gt;c_key));
}

/*
 * Search the request hash for an entry that matches the given rqstp.
 * Must be called with cache_lock held. Returns the found entry or
 * inserts an empty key on failure.
 */
static struct svc_cacherep *
nfsd_cache_insert(struct nfsd_drc_bucket *b, struct svc_cacherep *key,
			struct nfsd_net *nn)
{
	struct svc_cacherep	*rp, *ret = key;
	struct rb_node		**p = &amp;b-&gt;rb_head.rb_node,
				*parent = NULL;
	unsigned int		entries = 0;
	int cmp;

	while (*p != NULL) {
		++entries;
		parent = *p;
		rp = rb_entry(parent, struct svc_cacherep, c_node);

		cmp = nfsd_cache_key_cmp(key, rp, nn);
		if (cmp &lt; 0)
			p = &amp;parent-&gt;rb_left;
		else if (cmp &gt; 0)
			p = &amp;parent-&gt;rb_right;
		else {
			ret = rp;
			goto out;
		}
	}
	rb_link_node(&amp;key-&gt;c_node, parent, p);
	rb_insert_color(&amp;key-&gt;c_node, &amp;b-&gt;rb_head);
out:
	/* tally hash chain length stats */
	if (entries &gt; nn-&gt;longest_chain) {
		nn-&gt;longest_chain = entries;
		nn-&gt;longest_chain_cachesize = atomic_read(&amp;nn-&gt;num_drc_entries);
	} else if (entries == nn-&gt;longest_chain) {
		/* prefer to keep the smallest cachesize possible here */
		nn-&gt;longest_chain_cachesize = min_t(unsigned int,
				nn-&gt;longest_chain_cachesize,
				atomic_read(&amp;nn-&gt;num_drc_entries));
	}

	lru_put_end(b, ret);
	return ret;
}

/**
 * nfsd_cache_lookup - Find an entry in the duplicate reply cache
 * @rqstp: Incoming Call to find
 *
 * Try to find an entry matching the current call in the cache. When none
 * is found, we try to grab the oldest expired entry off the LRU list. If
 * a suitable one isn't there, then drop the cache_lock and allocate a
 * new one, then search again in case one got inserted while this thread
 * didn't hold the lock.
 *
 * Return values:
 *   %RC_DOIT: Process the request normally
 *   %RC_REPLY: Reply from cache
 *   %RC_DROPIT: Do not process the request further
 */
int nfsd_cache_lookup(struct svc_rqst *rqstp)
{
	struct nfsd_net		*nn;
	struct svc_cacherep	*rp, *found;
	__wsum			csum;
	struct nfsd_drc_bucket	*b;
	int type = rqstp-&gt;rq_cachetype;
	int rtn = RC_DOIT;

	rqstp-&gt;rq_cacherep = NULL;
	if (type == RC_NOCACHE) {
		nfsd_stats_rc_nocache_inc();
		goto out;
	}

	csum = nfsd_cache_csum(rqstp);

	/*
	 * Since the common case is a cache miss followed by an insert,
	 * preallocate an entry.
	 */
	nn = net_generic(SVC_NET(rqstp), nfsd_net_id);
	rp = nfsd_reply_cache_alloc(rqstp, csum, nn);
	if (!rp)
		goto out;

	b = nfsd_cache_bucket_find(rqstp-&gt;rq_xid, nn);
	spin_lock(&amp;b-&gt;cache_lock);
	found = nfsd_cache_insert(b, rp, nn);
	if (found != rp)
		goto found_entry;

	nfsd_stats_rc_misses_inc();
	rqstp-&gt;rq_cacherep = rp;
	rp-&gt;c_state = RC_INPROG;

	atomic_inc(&amp;nn-&gt;num_drc_entries);
	nfsd_stats_drc_mem_usage_add(nn, sizeof(*rp));

	nfsd_prune_bucket(b, nn);

out_unlock:
	spin_unlock(&amp;b-&gt;cache_lock);
out:
	return rtn;

found_entry:
	/* We found a matching entry which is either in progress or done. */
	nfsd_reply_cache_free_locked(NULL, rp, nn);
	nfsd_stats_rc_hits_inc();
	rtn = RC_DROPIT;
	rp = found;

	/* Request being processed */
	if (rp-&gt;c_state == RC_INPROG)
		goto out_trace;

	/* From the hall of fame of impractical attacks:
	 * Is this a user who tries to snoop on the cache? */
	rtn = RC_DOIT;
	if (!test_bit(RQ_SECURE, &amp;rqstp-&gt;rq_flags) &amp;&amp; rp-&gt;c_secure)
		goto out_trace;

	/* Compose RPC reply header */
	switch (rp-&gt;c_type) {
	case RC_NOCACHE:
		break;
	case RC_REPLSTAT:
		svc_putu32(&amp;rqstp-&gt;rq_res.head[0], rp-&gt;c_replstat);
		rtn = RC_REPLY;
		break;
	case RC_REPLBUFF:
		if (!nfsd_cache_append(rqstp, &amp;rp-&gt;c_replvec))
			goto out_unlock; /* should not happen */
		rtn = RC_REPLY;
		break;
	default:
		WARN_ONCE(1, "nfsd: bad repcache type %d\n", rp-&gt;c_type);
	}

out_trace:
	trace_nfsd_drc_found(nn, rqstp, rtn);
	goto out_unlock;
}

/**
 * nfsd_cache_update - Update an entry in the duplicate reply cache.
 * @rqstp: svc_rqst with a finished Reply
 * @cachetype: which cache to update
 * @statp: Reply's status code
 *
 * This is called from nfsd_dispatch when the procedure has been
 * executed and the complete reply is in rqstp-&gt;rq_res.
 *
 * We're copying around data here rather than swapping buffers because
 * the toplevel loop requires max-sized buffers, which would be a waste
 * of memory for a cache with a max reply size of 100 bytes (diropokres).
 *
 * If we should start to use different types of cache entries tailored
 * specifically for attrstat and fh's, we may save even more space.
 *
 * Also note that a cachetype of RC_NOCACHE can legally be passed when
 * nfsd failed to encode a reply that otherwise would have been cached.
 * In this case, nfsd_cache_update is called with statp == NULL.
 */
void nfsd_cache_update(struct svc_rqst *rqstp, int cachetype, __be32 *statp)
{
	struct nfsd_net *nn = net_generic(SVC_NET(rqstp), nfsd_net_id);
	struct svc_cacherep *rp = rqstp-&gt;rq_cacherep;
	struct kvec	*resv = &amp;rqstp-&gt;rq_res.head[0], *cachv;
	struct nfsd_drc_bucket *b;
	int		len;
	size_t		bufsize = 0;

	if (!rp)
		return;

	b = nfsd_cache_bucket_find(rp-&gt;c_key.k_xid, nn);

	len = resv-&gt;iov_len - ((char*)statp - (char*)resv-&gt;iov_base);
	len &gt;&gt;= 2;

	/* Don't cache excessive amounts of data and XDR failures */
	if (!statp || len &gt; (256 &gt;&gt; 2)) {
		nfsd_reply_cache_free(b, rp, nn);
		return;
	}

	switch (cachetype) {
	case RC_REPLSTAT:
		if (len != 1)
			printk("nfsd: RC_REPLSTAT/reply len %d!\n",len);
		rp-&gt;c_replstat = *statp;
		break;
	case RC_REPLBUFF:
		cachv = &amp;rp-&gt;c_replvec;
		bufsize = len &lt;&lt; 2;
		cachv-&gt;iov_base = kmalloc(bufsize, GFP_KERNEL);
		if (!cachv-&gt;iov_base) {
			nfsd_reply_cache_free(b, rp, nn);
			return;
		}
		cachv-&gt;iov_len = bufsize;
		memcpy(cachv-&gt;iov_base, statp, bufsize);
		break;
	case RC_NOCACHE:
		nfsd_reply_cache_free(b, rp, nn);
		return;
	}
	spin_lock(&amp;b-&gt;cache_lock);
	nfsd_stats_drc_mem_usage_add(nn, bufsize);
	lru_put_end(b, rp);
	rp-&gt;c_secure = test_bit(RQ_SECURE, &amp;rqstp-&gt;rq_flags);
	rp-&gt;c_type = cachetype;
	rp-&gt;c_state = RC_DONE;
	spin_unlock(&amp;b-&gt;cache_lock);
	return;
}

/*
 * Copy cached reply to current reply buffer. Should always fit.
 * FIXME as reply is in a page, we should just attach the page, and
 * keep a refcount....
 */
static int
nfsd_cache_append(struct svc_rqst *rqstp, struct kvec *data)
{
	struct kvec	*vec = &amp;rqstp-&gt;rq_res.head[0];

	if (vec-&gt;iov_len + data-&gt;iov_len &gt; PAGE_SIZE) {
		printk(KERN_WARNING "nfsd: cached reply too large (%zd).\n",
				data-&gt;iov_len);
		return 0;
	}
	memcpy((char*)vec-&gt;iov_base + vec-&gt;iov_len, data-&gt;iov_base, data-&gt;iov_len);
	vec-&gt;iov_len += data-&gt;iov_len;
	return 1;
}

/*
 * Note that fields may be added, removed or reordered in the future. Programs
 * scraping this file for info should test the labels to ensure they're
 * getting the correct field.
 */
int nfsd_reply_cache_stats_show(struct seq_file *m, void *v)
{
	struct nfsd_net *nn = net_generic(file_inode(m-&gt;file)-&gt;i_sb-&gt;s_fs_info,
					  nfsd_net_id);

	seq_printf(m, "max entries:           %u\n", nn-&gt;max_drc_entries);
	seq_printf(m, "num entries:           %u\n",
		   atomic_read(&amp;nn-&gt;num_drc_entries));
	seq_printf(m, "hash buckets:          %u\n", 1 &lt;&lt; nn-&gt;maskbits);
	seq_printf(m, "mem usage:             %lld\n",
		   percpu_counter_sum_positive(&amp;nn-&gt;counter[NFSD_NET_DRC_MEM_USAGE]));
	seq_printf(m, "cache hits:            %lld\n",
		   percpu_counter_sum_positive(&amp;nfsdstats.counter[NFSD_STATS_RC_HITS]));
	seq_printf(m, "cache misses:          %lld\n",
		   percpu_counter_sum_positive(&amp;nfsdstats.counter[NFSD_STATS_RC_MISSES]));
	seq_printf(m, "not cached:            %lld\n",
		   percpu_counter_sum_positive(&amp;nfsdstats.counter[NFSD_STATS_RC_NOCACHE]));
	seq_printf(m, "payload misses:        %lld\n",
		   percpu_counter_sum_positive(&amp;nn-&gt;counter[NFSD_NET_PAYLOAD_MISSES]));
	seq_printf(m, "longest chain len:     %u\n", nn-&gt;longest_chain);
	seq_printf(m, "cachesize at longest:  %u\n", nn-&gt;longest_chain_cachesize);
	return 0;
}
</net></linux></linux></linux></linux></linux></linux></linux></okir@monad.swb.de>