#!/usr/bin/env bash

# Bash settings
set -o errexit    # abort on nonzero exitstatus
set -o pipefail   # don't hide errors within pipes
set -o nounset    # abort on unbound variable

# Variables
readonly script_name=$(basename "${0}")                                   # Name of the script
readonly script_dir=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )     # Directory of the script
IFS=$'\t\n'                                                               # Split on newlines and tabs (but not on spaces)
NCPU="${1:-4}"                                                            # Number of CPUs used by Julia

# Description of the script
echo -n "Running the script ${script_name} in ${script_dir}."
printf "\n"
echo -n "Each Julia job will use ${NCPU} CPUs."
printf "\n\n"

# Characteristic times simulations T^⋆_{1/2}/T^⋆
echo -n "Running: julia -O3 -p${NCPU} adatt_simulations.jl --seed 42 --Nruns 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" adatt_simulations.jl --seed 42 --Nruns 1000
printf "\n\n"

# Random Gaussian Instances
echo -n "Running: julia -O3 -p${NCPU} adatt_randoms.jl --expe 'random' --inst 'uniform' --mu1 0.6 --gapmin 0.1 --gapmax 0.4 --K 10 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" adatt_randoms.jl --expe "random" --inst "uniform" --mu1 0.6 --gapmin 0.1 --gapmax 0.4 --K 10 --seed 42 --Nruns 5000
printf "\n\n"

# Random Equal Gap Gaussian Instances
echo -n "Running: julia -O3 -p${NCPU} adatt_experiments.jl --expe 'fastAdapt' --inst 'eqgapE' --K 35 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" adatt_experiments.jl --expe "fastAdapt" --inst "eqgapE" --K 35 --seed 42 --Nruns 5000
printf "\n\n"

# Random Gaussian Instances with Multiple Close Competitors
echo -n "Running: julia -O3 -p${NCPU} adatt_randoms.jl --expe 'random' --inst 'unif2G' --K 10 --seed 42 --Nruns 5000"
printf "\n\n"
julia -O3 -p"${NCPU}" adatt_randoms.jl --expe "random" --inst "unif2G" --K 10 --seed 42 --Nruns 5000
printf "\n\n"

# Sparse benchmark
for K in 10 50 100 250 500 750 1000
do
  echo -n "Running: julia -O3 -p${NCPU} adatt_experiments.jl --expe 'fast' --inst 'sparse' --K ${K} --seed 42 --Nruns 100"
  printf "\n\n"
  julia -O3 -p"${NCPU}" adatt_experiments.jl --expe "fast" --inst "sparse" --K "${K}" --seed 42 --Nruns 100
  printf "\n\n"
done

# Alpha 0.3 benchmark
for K in 10 50 100 250 500 750 1000
do
  echo -n "Running: julia -O3 -p${NCPU} adatt_experiments.jl --expe 'fast' --inst 'alpha3' --K ${K} --seed 42 --Nruns 100"
  printf "\n\n"
  julia -O3 -p"${NCPU}" adatt_experiments.jl --expe "fast" --inst "alpha3" --K "${K}" --seed 42 --Nruns 100
  printf "\n\n"
done

# Alpha 0.6 benchmark
for K in 10 50 100 250 500
do
  echo -n "Running: julia -O3 -p${NCPU} adatt_experiments.jl --expe 'fast' --inst 'alpha6' --K ${K} --seed 42 --Nruns 100"
  printf "\n\n"
  julia -O3 -p"${NCPU}" adatt_experiments.jl --expe "fast" --inst "alpha6" --K "${K}" --seed 42 --Nruns 100
  printf "\n\n"
done
