import time
import argparse
import numpy as np
import matplotlib.pyplot as plt

from Algorithms.DataDropOja import DataDropOja
from Algorithms.ExperienceReplayOja import ExperienceReplayOja
from Algorithms.Offline import Offline
from Algorithms.VanillaOja import VanillaOja
from datetime import datetime
from MarkovChains.Cyclic import generate_cyclic_mc
from MarkovChains.FaultyServer import generate_faulty_server_mc
from MarkovChains.WorstCase1 import generate_worstcase1_mc
from MarkovChains.WorstCase2 import generate_worstcase2_mc
from MarkovChains.ErdosRenyi import generate_erdosrenyi_mc
from plot_experiments import plot_and_save
from numpy import linalg


def generate_markov_chain_data(mc, num_repetitions, is_faulty_server=False, offset=0):
    # Generate Markov Chain data
    repetitions = {}
    timesteps = args.timesteps
    if not is_faulty_server:
        for r in range(num_repetitions):
            data = []
            state = mc.get_initial_state()
            for i in range(timesteps):
                if i >= offset:
                    sample = mc.get_sample(state)
                    data.append([i, state, sample])
                state = mc.get_next_state(state)
            repetitions[r] = data
    else:
        for r in range(num_repetitions):
            data = []
            state = np.array([0])
            prev_value = -1
            for i in range(timesteps):
                if int(state) == 0:
                    sample = mc.get_sample(state)
                    if i >= offset:
                        data.append((i, state, sample))
                    prev_value = sample
                else:
                    if i >= offset:
                        data.append((i, state, prev_value))
                state = mc.get_next_state(state)
            repetitions[r] = data
    return repetitions


def generate_markov_chain_data_drop_data(mc, num_repetitions, is_faulty_server=False, offset=0):
    # Generate Markov Chain data
    repetitions = {}
    timesteps = args.timesteps
    if not is_faulty_server:
        for r in range(num_repetitions):
            data = []
            state = mc.get_initial_state()
            for i in range(timesteps * args.drop_number):
                if i % args.drop_number == 0:
                    sample = mc.get_sample(state)
                    if i >= offset:
                        data.append((i, state, sample))
                state = mc.get_next_state(state)
            repetitions[r] = data
    else:
        for r in range(num_repetitions):
            data = []
            state = np.array([0])
            prev_value = -1
            for i in range(timesteps * args.drop_number):
                if int(state) == 0:
                    sample = mc.get_sample(state)
                    prev_value = sample
                    if i % args.drop_number == 0:
                        if i >= offset:
                            data.append((i, state, sample))
                else:
                    if i % args.drop_number == 0:
                        if i >= offset:
                            data.append((i, state, prev_value))
                state = mc.get_next_state(state)
            repetitions[r] = data
    return repetitions


def generate_iid_data(mc, num_repetitions, offset=0):
    # Generate data for IID simulations
    iid_repetitions = {}
    timesteps = args.timesteps
    for r in range(num_repetitions):
        data = []
        state = mc.get_initial_state()
        for i in range(timesteps):
            sample = mc.get_sample(state)
            state = np.random.choice(np.arange(mc.num_states), 1, p=mc.get_stationary_distribution())
            if i >= offset:
                data.append((i, state, sample))
        iid_repetitions[r] = data
    return iid_repetitions


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Arguments for Markov-Oja simulations.')
    parser.add_argument('--seed', type=int, default=5,
                        help='The random seed for all data generation and simulations.')
    parser.add_argument('--timesteps', type=int, default=5000, help='Number of timesteps for simulation.')
    parser.add_argument('--num_repetitions', type=int, default=1,
                        help='Number of repetitions over which to average the results.')
    parser.add_argument('--num_states', type=int, default=50, help='Number of states of Markov chain.')
    parser.add_argument('--num_dimensions', default=1000, type=int,
                        help='Number of dimensions of the data being generated at each state.')
    parser.add_argument('--markov_chain', default='erdosrenyi', help='Type of Markov Chain to use.')
    parser.add_argument('--experiment_identifier', default='standard', help='Identifying name of the experiment.')
    parser.add_argument('--output_dir', default='results', help='Directory for saving experiment results.')
    parser.add_argument('--cov_eigengap_threshold', type=int, default=60,
                        help='Threshold for eigengap of True Covariance '
                             'Matrix, Used for generating Cyclic Markov Chains.')
    parser.add_argument('--drop_number', type=int, default=10, help='Drop Number for Data Drop Algorithm.')
    parser.add_argument('--buffer_size', type=int, default=1000, help='Buffer size for experience replay.')
    parser.add_argument('--buffer_drop_number', type=int, default=100,
                        help='Drop Number for dropping data at beginning of each buffer.')
    parser.add_argument('--algorithms', nargs='+', help='Algorithms to use.', default="data_drop vanilla offline iid")
    parser.add_argument('--lr_multiplier', type=float, default=0.005, help='Multiplier for the learning rate.')
    parser.add_argument('--lr_decay', default=False, action='store_true',
                        help='Use this flag to enable decay of learning rate.')
    parser.add_argument('--offset', type=int, default=0, help='Offset after which to consider data.')
    parser.add_argument('--data_drop_optimize', default=False, action='store_true',
                        help='Use this flag to optimize data generation for data-drop over large number of timesteps.')
    args = parser.parse_args()
    print("Arguments : ", args)

    # Set random seed
    np.random.seed(args.seed)

    # Set experiment name
    current_timestamp = datetime.today().strftime('%Y-%m-%d %H:%M:%S')
    experiment_name = args.experiment_identifier + "-" + current_timestamp + "-" + str(args.seed)

    # Initialise Markov Chain
    markov_chain_generator_fn = {'cyclic': generate_cyclic_mc,
                                 'faulty_server': generate_faulty_server_mc,
                                 'worstcase1': generate_worstcase1_mc,
                                 'worstcase2': generate_worstcase2_mc,
                                 'erdosrenyi': generate_erdosrenyi_mc}
    markov_chain_generator_args = {'cyclic': {'num_states': args.num_states, 'num_dimensions': args.num_dimensions,
                                              'seed': args.seed, 'cov_eigengap_threshold': args.cov_eigengap_threshold},
                                   'faulty_server': {'num_states': args.num_states,
                                                     'num_dimensions': args.num_dimensions,
                                                     'seed': args.seed},
                                   'worstcase1': {'num_states': args.num_states, 'num_dimensions': args.num_dimensions,
                                                  'seed': args.seed,
                                                  'cov_eigengap_threshold': args.cov_eigengap_threshold},
                                   'worstcase2': {'eps': 0.2,
                                                  'beta': 1.0,
                                                  'num_states': args.num_states, 'num_dimensions': args.num_dimensions,
                                                  'seed': args.seed,
                                                  'cov_eigengap_threshold': args.cov_eigengap_threshold},
                                   'erdosrenyi': {'num_states': args.num_states, 'num_dimensions': args.num_dimensions,
                                                  'seed': args.seed,
                                                  'cov_eigengap_threshold': args.cov_eigengap_threshold,
                                                  'p': 2 * np.log(args.num_states) / args.num_states}}
    markov_chain = markov_chain_generator_fn[args.markov_chain](**markov_chain_generator_args[args.markov_chain])

    # # Generate Markov Chain data for mean and covariance estimation
    # start = time.time()
    # markov_chain_data = generate_markov_chain_data(markov_chain, args.num_repetitions,
    #                                                (args.markov_chain == 'faulty_server'),
    #                                                args.offset)
    # end = time.time()
    # print("Time required to generate Markov Chain data : ", end - start)
    # sample_mean = np.zeros(markov_chain.num_dimensions)
    # sample_sum = np.zeros(markov_chain.num_dimensions)
    # counter = 0
    # for r in range(args.num_repetitions):
    #     itr = 0
    #     for t in range(len(markov_chain_data[r])):
    #         At = markov_chain_data[r][t][2]
    #         sample_sum += At
    #         sample_mean = sample_sum / (counter + 1)
    #         itr += 1
    #         counter += 1
    #     print("Sample Mean calculated")
    #
    # sample_covariance_matrix = np.zeros((markov_chain.num_dimensions, markov_chain.num_dimensions))
    # sample_covariance_matrix_sum = np.zeros((markov_chain.num_dimensions, markov_chain.num_dimensions))
    # counter = 0
    # for r in range(args.num_repetitions):
    #     itr = 0
    #     for t in range(len(markov_chain_data[r])):
    #         At = markov_chain_data[r][t][2]
    #         At = At - sample_mean
    #         markov_chain_data[r][t][2] = At
    #         sample_covariance_matrix_sum += np.outer(At, At)
    #         sample_covariance_matrix = sample_covariance_matrix_sum / (counter + 1)
    #         itr += 1
    #         counter += 1
    #     print("Sample Covariance calculated")
    # sample_covariance_matrix = (sample_covariance_matrix + np.transpose(sample_covariance_matrix)) / 2
    # markov_chain.true_covariance_matrix = sample_covariance_matrix
    #
    # eigenvalues, eigenvectors = linalg.eig(markov_chain.true_covariance_matrix)
    # sorted_indices = np.real(eigenvalues).argsort()[::-1]
    # markov_chain.principal_components = np.real(eigenvectors[:, sorted_indices])
    # markov_chain.eigenvalues = np.real(eigenvalues)[sorted_indices]
    #
    # markov_chain.largest_eigenvector = markov_chain.principal_components[:, 0]
    # markov_chain.largest_eigenvector /= linalg.norm(markov_chain.largest_eigenvector)
    #
    # markov_chain.lambda1 = markov_chain.eigenvalues[0]
    # if len(markov_chain.eigenvalues) > 1:
    #     markov_chain.lambda2 = markov_chain.eigenvalues[1]
    #     markov_chain.cov_eigengap = markov_chain.lambda1 - markov_chain.lambda2
    # markov_chain.print()

    # Generate Markov Chain data
    start = time.time()
    markov_chain_data = generate_markov_chain_data(markov_chain, args.num_repetitions,
                                                   (args.markov_chain == 'faulty_server'),
                                                   args.offset)

    # Generate data for IID simulations
    start = time.time()
    iid_data = generate_iid_data(markov_chain, args.num_repetitions, args.offset)
    end = time.time()
    print("Time required to generate IID data : ", end - start)

    # # Plot eigenvalues of Covariance Matrix
    # plt.figure(1)
    # plt.title("Eigenvalues of Covariance Matrix")
    # plt.xlabel("Index")
    # plt.ylabel("Eigenvalue")
    # plt.plot(np.flip(markov_chain.eigenvalues))
    # plt.show()

    # Bt = np.eye(args.num_dimensions)
    # I = np.eye(args.num_dimensions)
    # a = 1
    # b = args.timesteps
    # k = 40
    # m = 10
    # v_k = np.real(markov_chain.principal_components[:, args.num_dimensions - k])
    # v_m = np.real(markov_chain.principal_components[:, args.num_dimensions - m])
    # lambda_k = markov_chain.eigenvalues[args.num_dimensions - k]
    # lambda_m = markov_chain.eigenvalues[args.num_dimensions - m]
    # v_k /= linalg.norm(v_k)
    # v_k = np.reshape(v_k, (args.num_dimensions, 1))
    # v_m /= linalg.norm(v_m)
    # v_m = np.reshape(v_m, (args.num_dimensions, 1))
    # mu = markov_chain.eigenvalues[0]
    # print(markov_chain.principal_components.shape)
    # print("lambda_k : ", lambda_k)
    # print("mu : ", mu)
    # print("check1 : ", np.transpose(v_k) @ v_k)
    # print("check : ", np.transpose(v_k) @ markov_chain.get_true_covariance_matrix() @ v_k)
    # alpha = []
    # alpha_true = []
    # beta = []
    # alpha_1 = []
    # alpha_2 = []
    # sum_eta = 0
    # for t in range(len(markov_chain_data[0])):
    #     At = markov_chain_data[0][t][2]
    #     eta = a / (b + t)
    #     sum_eta += eta
    #     Bt = np.matmul((I - eta * np.outer(At, At)), Bt)
    #     # alpha_t = np.transpose(v_k) @ Bt @ np.transpose(Bt) @ v_k
    #     alpha_t = abs(np.transpose(v_k) @ Bt @ np.transpose(Bt) @ v_m)
    #     alpha_t = alpha_t[0]
    #     alpha.append(alpha_t)
    #     # alpha_true.append(np.exp(-2 * lambda_k * sum_eta))
    #     alpha_true.append(np.exp(-2 * (lambda_k + lambda_m) * sum_eta))
    #     alpha_1.append(np.exp(-2 * lambda_m * sum_eta))
    #     alpha_2.append(np.exp(-2 * lambda_k * sum_eta))
    #     beta.append(np.exp(-2 * mu * sum_eta))
    #
    # plt.figure(2)
    # plt.title("Rate comparison")
    # plt.xlabel("Timesteps")
    # plt.ylabel("Value")
    # plt.text(1, 1, "Num Dimensions : " + str(args.num_dimensions))
    # plt.text(0.95, 0.95, "K = "+str(k))
    # plt.plot(np.arange(args.timesteps), alpha, label="estimated")
    # # plt.plot(np.arange(args.timesteps), alpha_true, label="true")
    # # plt.plot(np.arange(args.timesteps), beta, label="mu")
    # # plt.plot(np.arange(args.timesteps), alpha_1, label="lambda_m")
    # # plt.plot(np.arange(args.timesteps), alpha_2, label="lambda_k")
    # plt.legend()
    # plt.show()

    # num_runs = 1000
    # markov_sum_avg = 0
    # iid_sum_avg = 0
    # mu = 0
    # for i in range(len(markov_chain.stationary_distribution)):
    #     mu += markov_chain.stationary_distribution[i] * markov_chain.means[i]
    # print("Mu : ", mu)
    # Nu = 0
    # for i in range(markov_chain.num_states):
    #     s_i = markov_chain.stationary_distribution[i]
    #     Nu_i = markov_chain.means[i] - 2 * markov_chain.means[i] * mu + mu * mu
    #     Nu += s_i * Nu_i
    # print("Nu : ", Nu)
    # print("Upper Bound : ", Nu / markov_chain.transition_eigengap)
    #
    # for x in range(num_runs):
    #     print("Run : ", x)
    #     # Generate Markov Chain data
    #     start = time.time()
    #     markov_chain_data = generate_markov_chain_data(markov_chain, args.num_repetitions,
    #                                                    (args.markov_chain == 'faulty_server'),
    #                                                    args.offset)
    #     end = time.time()
    #     # print("Time required to generate Markov Chain data : ", end - start)
    #
    #     # Generate data for IID simulations
    #     start = time.time()
    #     iid_data = generate_iid_data(markov_chain, args.num_repetitions, args.offset)
    #     end = time.time()
    #     # print("Time required to generate IID data : ", end - start)
    #
    #     # mean_iid = 0
    #     # for t in range(args.timesteps):
    #     #     At = iid_data[0][t][2]
    #     #     mean_iid += At
    #     # print("IID mean : ", mean_iid/args.timesteps)
    #     #
    #     # mean_markov = 0
    #     # for t in range(args.timesteps):
    #     #     At = markov_chain_data[0][t][2]
    #     #     mean_markov += At
    #     # print("Markov mean : ", mean_markov/args.timesteps)
    #
    #     markov_sum = 0
    #     A0 = markov_chain_data[0][0][2]-mu
    #     for t in range(1, 50):
    #         At = markov_chain_data[0][t][2]
    #         markov_sum += (A0*At)
    #     markov_sum_avg += markov_sum
    #
    #     iid_sum = 0
    #     A0 = iid_data[0][0][2]-mu
    #     for t in range(1, 50):
    #         At = iid_data[0][t][2]
    #         iid_sum += (A0*At)
    #     iid_sum_avg += iid_sum
    #
    # print("Markov Sum : ", abs(markov_sum_avg/num_runs))
    # print("IID Sum : ", abs(iid_sum_avg/num_runs))

    w_init = []
    for i in range(args.num_repetitions):
        w = np.random.randn(markov_chain.num_dimensions)
        w /= linalg.norm(w)
        w_init.append(w)

    data_drop_algorithm = DataDropOja()
    # drop_number = int(np.log(args.timesteps)/markov_chain.transition_eigengap)
    drop_number = 5
    print("Drop Number : ", drop_number)
    data_drop_parameters = {'data': markov_chain_data, 'markov_chain': markov_chain,
                            'lr_multiplier': args.lr_multiplier, 'drop_number': drop_number,
                            'lr_decay': args.lr_decay, 'w_init': w_init}
    if args.data_drop_optimize:
        # Generate data for data drop simulations
        start = time.time()
        markov_chain_data_drop_data = generate_markov_chain_data_drop_data(markov_chain, args.num_repetitions,
                                                                           (args.markov_chain == 'faulty_server'),
                                                                           args.offset)
        end = time.time()
        print("Time required to generate Markov Chain Data Drop data : ", end - start)
        data_drop_algorithm = VanillaOja()
        data_drop_parameters = {'data': markov_chain_data_drop_data, 'markov_chain': markov_chain,
                                'lr_multiplier': args.lr_multiplier, 'lr_decay': args.lr_decay, 'w_init': w_init}
    print("=================================================")

    # Apply Algorithms
    algorithm_fn = {'data_drop': data_drop_algorithm,
                    'experience_replay': ExperienceReplayOja(args.seed),
                    'offline-iid': Offline(),
                    'offline': Offline(),
                    'vanilla': VanillaOja(),
                    'iid': VanillaOja()}

    algorithm_args = {'iid': {'data': iid_data, 'markov_chain': markov_chain,
                              'lr_multiplier': args.lr_multiplier, 'lr_decay': args.lr_decay,
                              'w_init': w_init, 'is_iid': True},
                      'vanilla': {'data': markov_chain_data, 'markov_chain': markov_chain,
                                  'lr_multiplier': args.lr_multiplier, 'lr_decay': args.lr_decay,
                                  'w_init': w_init, 'is_iid': False},
                      'offline': {'data': markov_chain_data, 'markov_chain': markov_chain},
                      'offline-iid': {'data': iid_data, 'markov_chain': markov_chain},
                      'data_drop': data_drop_parameters,
                      'experience_replay': {'data': markov_chain_data, 'markov_chain': markov_chain,
                                            'buffer_size': args.buffer_size,
                                            'buffer_drop_number': args.buffer_drop_number,
                                            'lr_multiplier': args.lr_multiplier,
                                            'lr_decay': args.lr_decay}}
    results = {}
    valid_algorithms = args.algorithms
    for algorithm in algorithm_fn:
        if algorithm in valid_algorithms:
            print("Simulation for " + algorithm + " started.")
            start = time.time()
            results[algorithm] = algorithm_fn[algorithm].run_simulation(**algorithm_args[algorithm])
            end = time.time()
            print("Simulation for " + algorithm + " completed.")
            print("Time required : ", end - start)
            print("=================================================")

    # Plot and Save results
    plot_and_save(experiment_name, args.output_dir, results,
                  markov_chain.transition_eigengap,
                  markov_chain.cov_eigengap,
                  markov_chain.eigenvalues,
                  markov_chain.num_states)
