import argparse
from networks.network import Actor, Critic
from utils.utils import ReplayBuffer, make_mini_batch, convert_to_tensor

import torch
import torch.nn as nn
import torch.optim as optim
import numpy as np

class PPO_Var(nn.Module):
    def __init__(self, writer, device, state_dim, action_dim, args):
        super(PPO_Var,self).__init__()
        self.args = args
        
        self.data = ReplayBuffer(action_prob_exist = True, max_size = self.args.traj_length, state_dim = state_dim, num_action = action_dim)
        self.actor = Actor(self.args.layer_num, state_dim, action_dim, self.args.hidden_dim, \
                           self.args.activation_function, self.args.last_activation, self.args.trainable_std)
        self.critic = Critic(self.args.layer_num, state_dim, 1, \
                             self.args.hidden_dim, self.args.activation_function, last_activation=None)
        
        self.actor_optimizer = optim.Adam(self.actor.parameters(), lr=self.args.actor_lr)
        self.critic_optimizer = optim.Adam(self.critic.parameters(), lr=self.args.critic_lr)

        self.writer = writer
        self.device = device
        self.episodes = None
    def get_action(self,x):
        mu,sigma = self.actor(x)
        return mu,sigma
    
    def v(self,x):
        return self.critic(x)
    
    def put_data(self,transition):
        self.data.put_data(transition)

    def put_episodes(self, episodes):
        self.episodes = episodes
        
    def get_gae(self, states, rewards, next_states, dones):
        values = self.v(states).detach()
        td_target = rewards + self.args.gamma * self.v(next_states) * (1 - dones)
        delta = td_target - values
        delta = delta.detach().cpu().numpy()
        advantage_lst = []
        advantage = 0.0
        for idx in reversed(range(len(delta))):
            if dones[idx] == 1:
                advantage = 0.0
            advantage = self.args.gamma * self.args.lambda_ * advantage + delta[idx][0]
            advantage_lst.append([advantage])
        advantage_lst.reverse()
        advantages = torch.tensor(advantage_lst, dtype=torch.float).to(self.device)
        return values, advantages
    
    def train_net(self, n_epi):
        data = self.data.sample(shuffle = False)
        states, actions, rewards, next_states, dones, old_log_probs = convert_to_tensor(self.device, data['state'], data['action'], data['reward'], data['next_state'], data['done'], data['log_prob'])
        
        old_values, advantages = self.get_gae(states, rewards, next_states, dones)
        returns = advantages + old_values
        advantages = (advantages - advantages.mean())/(advantages.std()+1e-3)
        
        for i in range(self.args.train_epoch):

            actor_loss_lst = []

            for state,action,old_log_prob,advantage,return_,old_value \
                in make_mini_batch(self.args.batch_size, states, actions, \
                                           old_log_probs,advantages,returns,old_values): 

                curr_mu,curr_sigma = self.get_action(state)
                value = self.v(state).float()
                curr_dist = torch.distributions.Normal(curr_mu,curr_sigma)
                entropy = curr_dist.entropy() * self.args.entropy_coef
                curr_log_prob = curr_dist.log_prob(action).sum(1,keepdim = True)

                #policy clipping
                ratio = torch.exp(curr_log_prob - old_log_prob.detach())
                surr1 = ratio * advantage
                surr2 = torch.clamp(ratio, 1-self.args.max_clip, 1+self.args.max_clip) * advantage
                actor_loss = (-torch.min(surr1, surr2) - entropy).mean()
                actor_loss_lst.append(actor_loss)
                
                #value clipping (PPO2 technic)
                old_value_clipped = old_value + (value - old_value).clamp(-self.args.max_clip,self.args.max_clip)
                value_loss = (value - return_.detach().float()).pow(2)
                value_loss_clipped = (old_value_clipped - return_.detach().float()).pow(2)
                critic_loss = 0.5 * self.args.critic_coef * torch.max(value_loss,value_loss_clipped).mean()
                
                # self.actor_optimizer.zero_grad()
                # actor_loss.backward()
                # nn.utils.clip_grad_norm_(self.actor.parameters(), self.args.max_grad_norm)
                # self.actor_optimizer.step()
                
                self.critic_optimizer.zero_grad()
                critic_loss.backward()
                nn.utils.clip_grad_norm_(self.critic.parameters(), self.args.max_grad_norm)
                self.critic_optimizer.step()
                
                if self.writer != None:
                    self.writer.add_scalar("loss/actor_loss", actor_loss.item(), n_epi)
                    self.writer.add_scalar("loss/critic_loss", critic_loss.item(), n_epi)


            ppo_loss = torch.stack(actor_loss_lst).mean()
            variance_loss = self.prepare_var()
            policy_loss = ppo_loss + self.args.lam * variance_loss
            self.actor_optimizer.zero_grad()
            policy_loss.backward()
            nn.utils.clip_grad_norm_(self.actor.parameters(), self.args.max_grad_norm)
            self.actor_optimizer.step()

    def prepare_var(self):
        episodes_num = len(self.episodes['state'])

        npg_lst = []
        sum_log_prob_lst = []
        ret_lst = []
        ratio_lst = []
        ep_len_lst = []

        for i_ep in range(episodes_num):
            reward_lst = self.episodes['reward'][i_ep]
            state_lst = self.episodes['state'][i_ep]
            action_lst = self.episodes['action'][i_ep]
            old_log_prob_lst = self.episodes['log_prob'][i_ep]

            # compute rewards-to-go
            traj_len = len(reward_lst)
            ep_len_lst.append(traj_len)
            running_g = 0
            gs = []
            for R in reward_lst[::-1]:
                running_g = R + self.args.gamma * running_g
                gs.insert(0, running_g)
            ret_lst.append(running_g)

            # compute log_prob
            state = torch.tensor(np.array(state_lst)).float()
            action = torch.tensor(np.array(action_lst)).float()
            curr_mu, curr_sigma = self.get_action(state)
            curr_dist = torch.distributions.Normal(curr_mu,curr_sigma)
            curr_log_prob = curr_dist.log_prob(action).sum(1,keepdim = True)
            gs_t = torch.tensor(np.array(gs)).float()
            #print('curr_log_prob:', curr_log_prob.shape)
            #print('gs_t:', gs_t.shape)
            npg = curr_log_prob.squeeze(-1) * gs_t
            #print('npg:', npg.shape)
            npg_lst.append(npg.sum())
            sum_curr_log_prob = curr_log_prob.sum()
            sum_log_prob_lst.append(sum_curr_log_prob)
            
            # compute ratio
            old_log_prob = torch.tensor(np.array(old_log_prob_lst))
            log_ratio = curr_log_prob.detach() - old_log_prob
            ratio = torch.exp(log_ratio.sum()).item()
            ratio_lst.append(ratio)

        '''clip IS ratio'''
        is_ratio = np.array(ratio_lst)
        is_ratio = np.minimum(is_ratio, 1.6)
        ep_len = np.array(ep_len_lst)

        # nabla_theta M(theta)
        ret_t = torch.tensor(ret_lst).float()
        ret2_t = torch.square(ret_t)
        is_ratio_t = torch.tensor(is_ratio).float()
        sum_log_prob_t = torch.stack(sum_log_prob_lst)
        M = ret2_t * sum_log_prob_t * is_ratio_t
        #print("M:", M.shape)
        n_trans = ep_len.sum()
        M = M.sum() / n_trans

        # J nabla J
        npg_t = torch.stack(npg_lst)
        #print('npg_t:', npg_t.shape)
        half_idx= int(episodes_num / 2)
        J = (ret_t[0: half_idx] * is_ratio_t[0: half_idx]).mean()
        nabla_J = npg_t[half_idx:] * is_ratio_t[half_idx:]
        half_n_trans = ep_len[half_idx :].sum()
        nabla_J = nabla_J.sum() / half_n_trans

        variance_loss = M - 2 * J * nabla_J

        return variance_loss
        



