import numpy as np
from math import comb
from collections import deque

# Iterate over binary numbers of a given Hamming weight, stopping at end

def fixed_weight_iterator(weight, end=None):
    x = (1<<weight) - 1
    while (end is None) or (x<end):
        yield(x)
        xlsb = x & -x
        y = x + xlsb
        ylsb = y & -y
        x = y + ylsb//(2*xlsb) - 1


# Iterate over bits set to 1, from least significant to most significant

def bit_iterator(x):
    while x>0:
        lsb = x & -x
        yield lsb
        x ^= lsb


# Iterate over powers of two up to a given number, starting at 1

def power_2_iterator(end = None):
    x = 1
    while end is None or x <= end:
        yield x
        x <<= 1



# Output noise term that makes the prefix sums of a sequence of 0-1
# variables differentially private, using rho-zCDP

def continual_observation_noise(length, rho = 1, dimensions = 1, noise_generator = np.random.normal, neutral_element = 0., negation = lambda x: -x):
    # Initialization
    depth = 0
    while comb(depth, depth//2) <= length:
        depth += 2
    variance = depth/(4*rho) # rho-zCDP
    noise = {}
    leaves = fixed_weight_iterator(depth//2, 1<<depth)
    # Iteration
    n = neutral_element
    l1, l2 = 0, 0 # Two adjacent leaves
    for _ in range(length):
        l1, l2 = l2, next(leaves)
        for b in power_2_iterator(l1 ^ l2):
            if b & l1 > 0: # Remove nodes from path to previous leaf
                n += negation(noise[b])
                del noise[b]
            if b & l2 > 0: # Add nodes from path to next leaf
                if b not in noise:
                    noise[b] = noise_generator(0, variance, size=dimensions)
                n += noise[b]
        yield n



def binary_mechanism_noise(length, rho = 1, dimensions = 1, noise_generator = np.random.normal, neutral_element = 0., negation = lambda x: -x):
    # Initialization
    depth = 0
    while 2**depth <= length:
        depth += 1
    variance = depth/(2*rho) # rho-zCDP
    noise = {}
    leaves = iter(range(1, length+1))
    # Iteration
    n = neutral_element
    l1, l2 = 0, 0 # Two adjacent leaves
    for _ in range(length): # Invariant: 1s in l2 have stored noise, summing to n
        l1, l2 = l2, next(leaves)
        for b in power_2_iterator(l1 ^ l2):
            if b & l1 > 0: # Remove nodes from path to previous leaf
                n += negation(noise[b])
                del noise[b]
            if b & l2 > 0: # Add nodes from path to next leaf
                if b not in noise:
                    noise[b] = noise_generator(0, variance, size=dimensions)
                n += noise[b]
        yield n
