from .anderson import anderson_solver
from .broyden import broyden_solver
from .naive import naive_solver, speedy_naive_solver

from .utils import produce_final_info


_solvers = {
        'anderson': anderson_solver,
        'broyden': broyden_solver,
        'naive_solver': naive_solver,
        'speedy_naive': speedy_naive_solver,
        }


def register_solver(solver_type, solver):
    """
    Registers a user-defined fixed point solver. 
    This solver can be designated using args.f_solver and args.b_solver.

    This method adds a new entry to the solver dict with the key as
    the specified ``solver_type`` and the value as the ``solver``.

    Args:
        solver_type (str): The type of solver to register. This will be used as the key in the solver dict.
        solver_class (callable): The solver function. This will be used as the value in the solver dict.

    Example:
        >>> register_solver('newton', newton_solver)
    """
    _solvers[solver_type] = solver


def get_solver(key):
    assert key in _solvers

    return _solvers[key]
