# =================================================================================================#
# Description: Plots the experimental results for the stability analysis
# Author: Ryan Thompson
# =================================================================================================#

import Cairo, ColorSchemes, CSV, DataFrames, Fontconfig, Gadfly, Pipe, Statistics

# Load data
result = CSV.read("Results/stability.csv", DataFrames.DataFrame)

# Summarise results for plotting
result = Pipe.@pipe result |>
        DataFrames.transform(_, [:hamming_dist, :p] => DataFrames.ByRow((x, p) -> x / p) => 
            :hamming_dist) |>
        DataFrames.stack(_, [:hamming_dist]) |>
        DataFrames.groupby(_, [:n, :m, :p, :estimator, :loss, :variable]) |>
        DataFrames.combine(
        _, 
        :value => Statistics.mean => :mean, 
        :value => (x -> Statistics.mean(x) - Statistics.std(x) / sqrt(size(x, 1))) => :low,
        :value => (x -> Statistics.mean(x) + Statistics.std(x) / sqrt(size(x, 1))) => :high
        ) |>
        DataFrames.transform(_, [:p, :m] => DataFrames.ByRow((p, m) -> "p = $p, m = $m") => :pm)

# Plot results
Gadfly.plot(
    result,
    x = :n,
    y = :mean,
    ymin = :low,
    ymax = :high,
    xgroup = :pm,
    Gadfly.Geom.subplot_grid(Gadfly.Geom.point, Gadfly.Geom.line, Gadfly.Geom.yerrorbar),
    Gadfly.Guide.xlabel("Sample size"), 
    Gadfly.Guide.ylabel("Selection instability"),
    Gadfly.Scale.x_log10,
    Gadfly.Theme(default_color = "black", plot_padding = [0Gadfly.mm])
    ) |> 
Gadfly.PDF("Figures/stability.pdf", 9Gadfly.inch, 2.5Gadfly.inch)