# python3.8
"""Contains the class to evaluate consistency of face images output by
3D-aware GANs."""


import os.path
import time
import numpy as np

import torch
import torch.nn.functional as F

from .base_gan_metric import BaseGANMetric
from models.rendering.utils import LookAtPoseSampler
from models.rendering.utils import FOV_to_intrinsics
from models.rendering.point_sampler import sample_camera_extrinsics


class ReprojectionError(BaseGANMetric):
    """Defines the class for evaluation of reprojection error."""

    def __init__(self,
                 name='ReprojectionError',
                 work_dir=None,
                 logger=None,
                 tb_writer=None,
                 batch_size=1,
                 latent_dim=512,
                 latent_codes=None,
                 label_dim=0,
                 labels=None,
                 seed=0,
                 fake_num=1024,
                 fov=18.837,
                 random_pose=False,
                 eg3d_mode=True):
        super().__init__(name=name,
                         work_dir=work_dir,
                         logger=logger,
                         tb_writer=tb_writer,
                         batch_size=batch_size,
                         latent_num=fake_num,
                         latent_dim=latent_dim,
                         latent_codes=latent_codes,
                         label_dim=label_dim,
                         labels=labels,
                         seed=seed)
        self.fake_num = fake_num
        self.num_views = 5
        self.fov = fov
        self.random_pose = random_pose
        self.eg3d_mode = eg3d_mode
        self.cam_intrinsics = FOV_to_intrinsics(fov, device=self.device)

    def synthesize(self, data_loader, generator, generator_kwargs):
        """Synthesizes multi-view images and corresponding depth maps with the
        generator."""
        training_set = data_loader.dataset
        fake_num = self.fake_num
        # Note: Here, only a batch size of `1` is supported due to unknown bugs.
        batch_size = 1
        g1 = torch.Generator(device=self.device)
        g1.manual_seed(self.seed)

        G = generator
        G_kwargs = generator_kwargs
        G_mode = G.training  # save model training mode.
        G.eval()

        # When consistency of multi-view images, ensure that the mapping
        # condition remains fixed.
        label_swapped = torch.tensor([
            1, 0, 0, 0, 0, -1, 0, 0, 0, 0, -1, 2, 0, 0, 0, 1, 4, 0, 0, 0, 4, 0,
            0, 0, 1]).type(torch.float32)
        label_swapped = label_swapped.unsqueeze(0).to(self.device)

        self.logger.info(f'Synthesizing {self.fake_num} images and '
                         f'corresponding depth maps {self.log_tail}.',
                         is_verbose=True)
        self.logger.init_pbar()
        pbar_task = self.logger.add_pbar_task('Synthesis', total=fake_num)
        all_images = []
        all_depths = []
        all_poses = []
        for start in range(0, self.replica_latent_num, batch_size):
            end = min(start + batch_size, self.replica_latent_num)
            with torch.no_grad():
                batch_codes = torch.randn((end - start, *self.latent_dim),
                                          generator=g1,
                                          device=self.device)
                if self.random_pose:
                    batch_labels = sample_camera_extrinsics(
                        batch_size=(end - start),
                        radius_strategy='fix',
                        radius_fix=1.0,
                        polar_strategy='normal',
                        polar_mean=np.pi / 2,
                        polar_stddev=0.155,
                        azimuthal_strategy='normal',
                        azimuthal_mean=np.pi / 2,
                        azimuthal_stddev=0.3)['cam2world_matrix']
                else:
                    cam_pivot = torch.tensor(G.eval_rendering_options.get(
                        'avg_camera_pivot', [0, 0, 0]), device=self.device)
                    cam_radius = G.eval_rendering_options.get(
                        'avg_camera_radius', 2.7)
                    batch_labels = sample_cam_pose(
                        batch_size=(end - start),
                        cam_pivot=cam_pivot,
                        cam_radius=cam_radius,
                        device=self.device)  # [N * 5, 16]
                    cam_intrinsics = self.cam_intrinsics.reshape(-1, 9).repeat(
                        batch_labels.shape[0], 1)  # [N * 5, 9]
                    batch_labels = torch.cat([batch_labels, cam_intrinsics],
                                             dim=1)
                multi_view_images = []
                multi_view_depths = []
                multi_view_labels = []
                G_kwargs.update(
                    label_swapped=label_swapped.repeat(end - start, 1))
                for i in range(self.num_views):
                    labels = batch_labels[i * (end - start):(i + 1) *
                                          (end - start)]  # [N, 25]
                    if self.eg3d_mode:
                        batch_results = G(batch_codes, labels, **G_kwargs)
                    else:
                        batch_results = G(batch_codes,
                                          cam2world_matrix=labels,
                                          **G_kwargs)
                    batch_images = batch_results['image']  # [N, 3, H, W]
                    batch_depths = batch_results['image_depth']  # [N, 1, h, w]
                    _, _, H, W = batch_images.shape
                    batch_depths = F.interpolate(batch_depths, size=(H, W))
                    multi_view_images.append(batch_images)
                    multi_view_depths.append(batch_depths)
                    multi_view_labels.append(labels.unsqueeze(1))
                multi_view_images = torch.cat(
                    multi_view_images, dim=1)  # [N, 3 * num_views, H, W]
                multi_view_depths = torch.cat(multi_view_depths,
                                              dim=1)  # [N, num_views, H, W]
                multi_view_labels = torch.cat(multi_view_labels,
                                              dim=1)  # [N, num_views, 25]
                gathered_images = self.gather_batch_results(multi_view_images)
                self.append_batch_results(gathered_images, all_images)
                gathered_depths = self.gather_batch_results(multi_view_depths)
                self.append_batch_results(gathered_depths, all_depths)
                gathered_labels = self.gather_batch_results(multi_view_labels)
                self.append_batch_results(gathered_labels, all_poses)
            self.logger.update_pbar(pbar_task, (end - start) * self.world_size)
        self.logger.close_pbar()
        all_depths = self.gather_all_results(all_depths)[:fake_num]
        all_images = self.gather_all_results(all_images)[:fake_num]
        all_poses = self.gather_all_results(all_poses)[:fake_num]

        if self.is_chief:
            assert all_images.shape[:2] == (fake_num, self.num_views * 3)
            assert all_depths.shape[:2] == (fake_num, self.num_views)
            assert all_poses.shape[:2] == (fake_num, self.num_views)
        else:
            assert len(all_images) == 0
            assert len(all_depths) == 0
            assert len(all_poses) == 0
            all_images = None
            all_depths = None
            all_poses = None

        if G_mode:
            G.train()  # restore model training mode.

        self.sync()
        return all_images, all_depths, all_poses

    def evaluate(self, data_loader, generator, generator_kwargs):
        images, depths, poses = self.synthesize(data_loader, generator,
                                                 generator_kwargs)
        if self.is_chief:
            assert images.shape[:2] == (self.fake_num, self.num_views * 3)
            assert depths.shape[:2] == (self.fake_num, self.num_views)
            assert poses.shape[:2] == (self.fake_num, self.num_views)
            errors = []
            for i in range(self.fake_num):
                image = images[i]  # [3 * num_views, H, W]
                depth = depths[i]  # [num_views, H, W]
                pose = poses[i]  # [num_views, 25]
                error_sym = []
                for j in range(self.num_views - 1):
                    img0 = convert_image(image[3 * j:3 * (j + 1)])
                    img1 = convert_image(image[3 * (j + 1):3 * (j + 2)])
                    depth0 = depth[j]
                    depth1 = depth[j + 1]
                    cam2world0 = pose[j][:16].reshape(4, 4)
                    cam2world1 = pose[j + 1][:16].reshape(4, 4)
                    error_0 = calculate_reprojection_error(
                        img0, cam2world0, depth0, img1, cam2world1, self.fov)
                    error_1 = calculate_reprojection_error(
                        img1, cam2world1, depth1, img0, cam2world0, self.fov)
                    error_sym.append(error_0)
                    error_sym.append(error_1)
                error_sym = np.array(error_sym)
                errors.append(error_sym.mean())
            errors = np.array(errors)
            reprjection_error = errors.mean()
            result = {self.name: float(reprjection_error)}
        else:
            assert images is None and depths is None and poses is None
            result = None
        self.sync()
        return result

    def _is_better_than(self, metric_name, new, ref):
        """Lower depth error is better."""
        if metric_name == self.name:
            return ref is None or new < ref
        return None

    def save(self, result, target_filename=None, log_suffix=None, tag=None):
        if not self.is_chief:
            assert result is None
            self.sync()
            return

        assert isinstance(result, dict)
        reprjection_error = result[self.name]
        assert isinstance(reprjection_error, float)
        prefix = f'Evaluating `{self.name}`: '
        if log_suffix is None:
            msg = f'{prefix}{reprjection_error:.3f}.'
        else:
            msg = f'{prefix}{reprjection_error:.3f}, {log_suffix}.'
        self.logger.info(msg)

        with open(os.path.join(self.work_dir, f'{self.name}.txt'), 'a+') as f:
            date = time.strftime('%Y-%m-%d %H:%M:%S')
            f.write(f'[{date}] {msg}\n')

        # Save to TensorBoard if needed.
        if self.tb_writer is not None:
            if tag is None:
                self.logger.warning('`Tag` is missing when writing data to '
                                    'TensorBoard, hence, the data may be mixed '
                                    'up!')
            self.tb_writer.add_scalar(f'Metrics/{self.name}', reprjection_error,
                                      tag)
            self.tb_writer.flush()
        self.sync()

    def info(self):
        metric_info = super().info()
        metric_info['Num samples'] = self.fake_num
        return metric_info


def sample_cam_pose(batch_size, cam_pivot=None, cam_radius=None, device=None):
    batch_cam_pose = []
    angle_p = 0.0
    for _ in range(batch_size):
        for angle_y, angle_p in [(.4, angle_p), (.2, angle_p), (0, angle_p),
                                (-.2, angle_p), (-.4, angle_p)]:
            cam2world_pose = LookAtPoseSampler.sample(np.pi / 2 + angle_y,
                                                    np.pi / 2 + angle_p,
                                                    cam_pivot,
                                                    radius=cam_radius,
                                                    device=device)  # [1, 4, 4]
            cam2world_pose = cam2world_pose.reshape(1, 16)
            batch_cam_pose.append(cam2world_pose)
    batch_cam_pose = torch.cat(batch_cam_pose, dim=0)  # [N * 5, 16]
    return  batch_cam_pose


def get_K(H, W, fov):
    """Gets camera intrinsic matrix from field of view, image height and width.

    Args:
        H: Image height.
        W: Image width.
        fov: Field of view of camera.

    Returns:
        K: Intrinsic matrix of camera.
    """
    fx = W / (2 * np.tan(np.pi / 360 * fov))
    fy = -H / (2 * np.tan(np.pi / 360 * fov))
    cx = W / 2
    cy = H / 2
    K = np.array([[fx, 0, cx], [0, fy, cy], [0, 0, 1]])
    return K


def project(xyz, K, RT):
    """Projects points in 3D space onto 2D planes.

    Args:
        xyz: Input 3D points, with shape [N, 3].
        K: Intrinsic matrix of camera, with shape [3, 3].
        RT: Extrinsic matrix of camera, i.e. rotation and translation, with
            shape [3, 4].

    Returns:
        xy: Projected coordinates in 2D planes.
    """
    xyz = np.dot(xyz, RT[:, :3].T) + RT[:, 3:].T
    xyz[:, 2] = xyz[:, 2] * -1
    xyz = np.dot(xyz, K.T)
    xy = xyz[:, :2] / xyz[:, 2:]
    return xy


def convert_image(image):
    """Converts an image from PyTorch format to CV2 format."""
    image = image.transpose(1, 2, 0)[:, :, ::-1]
    image = (image + 1) * 127.5
    image = image.clip(0, 255)
    return image


def normalize(image):
    """"Transforms an image from [0, 255] to [-1, 1] by normalization."""
    image = (image - 127.5) / 127.5
    return image


def unproject(depth_map, K):
    """Unprojects pixels in 2D planes to 3D space with depth maps and camera
    intrinsic matrix.

    Args:
        depth_map: Depth map, with shape [H, W].
        K: Intrinsic matrix of camera, with shape [3, 3].

    Returns:
        xyz_map: Point coordinates in 3D space, with shape [H, W, 3].
    """
    H, W = depth_map.shape
    y, x = np.mgrid[:H, :W]
    xy_map = np.stack([x, y], axis=2) * depth_map[..., None]
    xyz_map = np.concatenate([xy_map, depth_map[..., None]], axis=-1)
    xyz = xyz_map.reshape(-1, 3)
    xyz = np.dot(xyz, np.linalg.inv(K).T)
    xyz_map = xyz.reshape(H, W, 3)
    return xyz_map


def transform_from_img0_to_img1(img0, depth0, K0, cam2world0, K1, world2cam1):
    """Transforms an image into another perspective."""
    H, W = img0.shape[:2]

    # Obtain the point cloud in the canonical space.
    points = unproject(depth0, K0).reshape(-1, 3)
    points[:, 2] = points[:, 2] * -1

    points = points @ cam2world0[:3, :3].T + cam2world0[:3, 3]

    # Obtain the xy in the target image space.
    xy1 = project(points, K1, world2cam1[:3])
    xy1 = np.round(xy1).astype(np.int32)

    xy1[:, 0] = np.clip(xy1[:, 0], 0, W - 1)
    xy1[:, 1] = np.clip(xy1[:, 1], 0, H - 1)

    # Create the target image.
    img1 = np.zeros_like(img0)
    rgb0 = img0.reshape(H * W, 3)
    img1[xy1[:, 1], xy1[:, 0]] = rgb0

    # Create the target mask.
    mask1 = np.zeros_like(img0)[..., 0]
    mask1[xy1[:, 1], xy1[:, 0]] = 1

    return img1, mask1


def calculate_reprojection_error(img0,
                                 cam2world0,
                                 depth0,
                                 img1,
                                 cam2world1,
                                 fov):
    """"Calculates the reprojection error between images from two views."""
    H, W = img0.shape[:2]
    K = get_K(H, W, fov)
    c2w0 = cam2world0.copy()
    c2w1 = cam2world1.copy()
    c2w0[:, 1:3] = -c2w0[:, 1:3]
    c2w1[:, 1:3] = -c2w1[:, 1:3]
    c2w1_inv = np.linalg.inv(c2w1)
    warp_img, mask = transform_from_img0_to_img1(img0, depth0, K, c2w0, K,
                                                 c2w1_inv)
    warp_img = normalize(warp_img)
    img1_copy = img1.copy()
    img1_copy[mask == 0] = 0
    gt_img = normalize(img1_copy)
    error = np.square((warp_img - gt_img)[mask != 0]).mean()

    return error
