# coding=utf-8
# Copyright 2023 The Soar Neurips2023 Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Plot correlation and distortion with respect to SOAR lambda."""
from absl import app
from absl import flags
import h5py
import matplotlib.pyplot as plt
import numpy as np
from scipy.stats import pearsonr
import utils

_HDF5 = flags.DEFINE_string("hdf5", None, "Path to hdf5 of dataset.")
_NUM_CENTERS = flags.DEFINE_integer("num_centers", 2000, "# k-means centers.")
_ETA = flags.DEFINE_float("eta", 2.5, "AVQ eta.")


def compute():
  """Compute correlation and distortion for various lambda."""
  eta = _ETA.value
  num_centers = _NUM_CENTERS.value

  hdf5 = h5py.File(_HDF5.value, "r")
  ds = utils.normalize(hdf5["train"][()])
  qs = utils.normalize(hdf5["test"][()])
  gt = hdf5["neighbors"][()]
  # Use random elements because we want to look at the general inner product
  # distribution, not just distribution of the nearest neighbors. This saves
  # compute over looking at every inner product.
  gt = np.random.randint(len(ds), size=(gt.shape))
  print("Dataset shape:", ds.shape)
  print("Query shape:", qs.shape)
  print("Ground truth shape:", gt.shape, flush=True)

  print("Training k-means...", flush=True)
  orig_centers, toke = utils.train_kmeans(ds, num_centers)
  print("Updating centers...", flush=True)
  centers = utils.compute_avq_centers(ds, orig_centers, toke, eta)

  primary_diffs = utils.compute_score_diffs(ds, centers, toke, qs, gt).flatten()

  res = []
  print("lambda,r,mse", flush=True)
  for soar_lambda in np.linspace(0, 4, 17):
    soar_toke = utils.soar_assign(ds, centers, toke, soar_lambda, False)
    mse = np.sum(np.square(ds - centers[soar_toke])) / len(ds)
    cur_diffs = utils.compute_score_diffs(ds, centers, soar_toke, qs, gt)
    r = pearsonr(primary_diffs, cur_diffs.flat).statistic
    res.append((soar_lambda, r, mse))
    print(f"{soar_lambda},{r},{mse}", flush=True)
  with open("corr_table.out", "w") as f:
    for soar_lambda, r, mse in res:
      f.write(f"{soar_lambda},{r},{mse}\n")


def plot():
  """Generate plot."""
  with open("corr_table.out", "r") as f:
    ls = f.readlines()
  res = [list(map(float, l.split(","))) for l in ls]
  xs = [r[0] for r in res]
  corrs = [r[1] for r in res]
  mses = [r[2] for r in res]

  _, ax1 = plt.subplots(figsize=(2.5, 1.5))
  ax1.set_xlabel("SOAR λ")
  ax1.set_ylabel("Correlation", color="blue")
  ax1.plot(xs, corrs, color="blue")
  ax2 = ax1.twinx()
  ax2.set_ylim([0, 1.2])
  ax2.set_ylabel("Distortion", color="red")
  ax2.plot(xs, mses, color="red")
  plt.savefig("out/lambda_effect.pdf", bbox_inches="tight", pad_inches=0.02)


def main(argv):
  del argv  # Unused.
  # compute()
  plot()


if __name__ == "__main__":
  app.run(main)
