;; gorilla-repl.fileformat = 1

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [[ a] [ 0 ]
          [ a]
          (let [
                a (sample (poisson 5.140000000000001))
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 2.326)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 23.26)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 5)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 4.208)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 42.08)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 11)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 3.008)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 30.080000000000002)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 7)
                a (sample (binomial a 0.2636))
                [ a] (cond
                  (= (sample (bernoulli 0.1)) 1)
                  (let [
                        a (+ a (sample (poisson 0.8559999999999999)))
                       ]
                    [ a ]
                  )
                  :else
                  (let [
                        a (+ a (sample (poisson 8.559999999999999)))
                       ]
                    [ a ]
                  )
                )
                _unused (observe (binomial a 0.2) 5)
               ]
            [ a ]
          )
         ]
    a
    )
  )

)


(def model_name "population_modified200")
(def outfile "population_modified200_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:smc []]
    [:smc [:number-of-particles 100]]
    [:pgibbs []]
    [:ipmcmc []]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with %s samples and options %s" method num_samples options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)




;; @@
;; ->
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 158.764494 ms
;;; Empirical mean: 20.551882530060674
;;; Empirical variance: 17.2031858118311
;;; Empirical std: 4.147672336604122
;;; Empirical skewness: 0.08236890001911726
;;; Empirical kurtosis: 2.8616126780572295
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 151.10505 ms
;;; Empirical mean: 20.20438166890006
;;; Empirical variance: 15.311410731484298
;;; Empirical std: 3.9129797765238066
;;; Empirical skewness: 0.07192172768541749
;;; Empirical kurtosis: 2.898952505461781
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 145.900191 ms
;;; Empirical mean: 20.42448322018606
;;; Empirical variance: 17.019530645166345
;;; Empirical std: 4.1254733843725555
;;; Empirical skewness: 0.11302073499836313
;;; Empirical kurtosis: 3.1634381515735472
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 127.452642 ms
;;; Empirical mean: 20.258898345243608
;;; Empirical variance: 16.221953015101473
;;; Empirical std: 4.027648571449783
;;; Empirical skewness: 0.22818436620067065
;;; Empirical kurtosis: 3.109404211896436
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 142.738595 ms
;;; Empirical mean: 20.289986700789203
;;; Empirical variance: 17.092156751418813
;;; Empirical std: 4.134266168429267
;;; Empirical skewness: 0.12515176782162787
;;; Empirical kurtosis: 3.0352891461225218
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 148.327219 ms
;;; Empirical mean: 20.34508412791621
;;; Empirical variance: 16.28179117832157
;;; Empirical std: 4.035070157794232
;;; Empirical skewness: 0.1708033783395698
;;; Empirical kurtosis: 3.007374143449535
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 143.819228 ms
;;; Empirical mean: 20.184995465606743
;;; Empirical variance: 16.083830619902038
;;; Empirical std: 4.010465137599632
;;; Empirical skewness: 0.18930045093420936
;;; Empirical kurtosis: 3.295503148687536
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 132.48759 ms
;;; Empirical mean: 20.024526164537612
;;; Empirical variance: 17.18716219612103
;;; Empirical std: 4.145740247063367
;;; Empirical skewness: 0.1622918257494518
;;; Empirical kurtosis: 2.965098242295249
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 127.644177 ms
;;; Empirical mean: 20.290284981219394
;;; Empirical variance: 16.32015976171048
;;; Empirical std: 4.0398217487545764
;;; Empirical skewness: 0.2501962570297991
;;; Empirical kurtosis: 3.0615794962206917
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 132.615887 ms
;;; Empirical mean: 20.21851156674218
;;; Empirical variance: 16.63295089778678
;;; Empirical std: 4.07835149267284
;;; Empirical skewness: 0.11531817609773592
;;; Empirical kurtosis: 3.2687545334842585
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 127.949388 ms
;;; Empirical mean: 20.427946320508738
;;; Empirical variance: 17.73180004445822
;;; Empirical std: 4.2109143952897234
;;; Empirical skewness: 0.1549170159725045
;;; Empirical kurtosis: 2.805656877654023
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 126.123365 ms
;;; Empirical mean: 20.448038673172796
;;; Empirical variance: 16.072468203467647
;;; Empirical std: 4.009048291486104
;;; Empirical skewness: 0.15148094170235346
;;; Empirical kurtosis: 2.987951885133404
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 140.561405 ms
;;; Empirical mean: 20.26208319359586
;;; Empirical variance: 18.32312693791372
;;; Empirical std: 4.280552176754036
;;; Empirical skewness: 0.32028344863068847
;;; Empirical kurtosis: 3.116379906213886
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 151.291476 ms
;;; Empirical mean: 20.154988911738528
;;; Empirical variance: 16.598096177710193
;;; Empirical std: 4.074076113391868
;;; Empirical skewness: 0.28473315935794863
;;; Empirical kurtosis: 3.1753199960522545
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 130.633418 ms
;;; Empirical mean: 20.332600211822665
;;; Empirical variance: 18.178855941546782
;;; Empirical std: 4.2636669595017365
;;; Empirical skewness: 0.21287854666977943
;;; Empirical kurtosis: 3.2362002087743607
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 136.070602 ms
;;; Empirical mean: 20.05908764532153
;;; Empirical variance: 14.704857289302208
;;; Empirical std: 3.834691289960928
;;; Empirical skewness: -0.006951003000911592
;;; Empirical kurtosis: 2.9312524611976847
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 146.767454 ms
;;; Empirical mean: 20.254676416367253
;;; Empirical variance: 15.925756739185385
;;; Empirical std: 3.9907088016022145
;;; Empirical skewness: -0.008813064987790598
;;; Empirical kurtosis: 2.668488006051679
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 123.970942 ms
;;; Empirical mean: 20.157058769179688
;;; Empirical variance: 18.268878357399785
;;; Empirical std: 4.274210846156257
;;; Empirical skewness: 0.27585243876494425
;;; Empirical kurtosis: 3.030199767927778
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 128.919134 ms
;;; Empirical mean: 20.32360659162411
;;; Empirical variance: 16.19726707732707
;;; Empirical std: 4.024582845131539
;;; Empirical skewness: 0.07504844886679476
;;; Empirical kurtosis: 3.3228637538230035
;;; 
;;; Method :importance with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 134.773009 ms
;;; Empirical mean: 20.51599486880821
;;; Empirical variance: 17.25143773148006
;;; Empirical std: 4.153485010383457
;;; Empirical skewness: 0.17253436828947077
;;; Empirical kurtosis: 3.0435511400169903
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 167.08014 ms
;;; Empirical mean: 19.110999999999997
;;; Empirical variance: 12.312679000000003
;;; Empirical std: 3.508942718255743
;;; Empirical skewness: 0.5568752893179097
;;; Empirical kurtosis: 3.897415261554904
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 165.292129 ms
;;; Empirical mean: 20.229000000000003
;;; Empirical variance: 10.306559000000004
;;; Empirical std: 3.2103829989582247
;;; Empirical skewness: -0.15756814373001798
;;; Empirical kurtosis: 2.8801934333436376
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 185.138814 ms
;;; Empirical mean: 20.207000000000004
;;; Empirical variance: 18.418151000000012
;;; Empirical std: 4.291637333233088
;;; Empirical skewness: 0.262391226063561
;;; Empirical kurtosis: 3.282482367564379
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 170.393979 ms
;;; Empirical mean: 20.174000000000003
;;; Empirical variance: 23.933724000000005
;;; Empirical std: 4.892210543302486
;;; Empirical skewness: -0.020298026874472314
;;; Empirical kurtosis: 3.1246599576772587
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 179.987182 ms
;;; Empirical mean: 18.848000000000003
;;; Empirical variance: 23.514896000000007
;;; Empirical std: 4.8492160191107185
;;; Empirical skewness: 0.24648423755867735
;;; Empirical kurtosis: 2.1818494291130572
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 161.023942 ms
;;; Empirical mean: 19.96
;;; Empirical variance: 17.1524
;;; Empirical std: 4.141545605205863
;;; Empirical skewness: 0.3303659273295262
;;; Empirical kurtosis: 2.527287813173225
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 176.932273 ms
;;; Empirical mean: 20.587
;;; Empirical variance: 15.488431000000004
;;; Empirical std: 3.935534398274268
;;; Empirical skewness: -0.14742599784743854
;;; Empirical kurtosis: 2.806941175296293
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 184.38748 ms
;;; Empirical mean: 19.094999999999995
;;; Empirical variance: 14.379975000000007
;;; Empirical std: 3.7920937488411344
;;; Empirical skewness: 0.1371689068595354
;;; Empirical kurtosis: 2.744277968416713
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 182.735493 ms
;;; Empirical mean: 19.144000000000005
;;; Empirical variance: 12.947264000000004
;;; Empirical std: 3.5982306763185714
;;; Empirical skewness: 0.1369429832235729
;;; Empirical kurtosis: 2.1630400426932987
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 181.477958 ms
;;; Empirical mean: 19.391000000000002
;;; Empirical variance: 18.144119
;;; Empirical std: 4.25959141233053
;;; Empirical skewness: 0.3230339209032523
;;; Empirical kurtosis: 2.7076873388988933
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 175.913268 ms
;;; Empirical mean: 20.723999999999997
;;; Empirical variance: 13.009824000000009
;;; Empirical std: 3.606913361864963
;;; Empirical skewness: -0.11010777275399723
;;; Empirical kurtosis: 3.3915926921263715
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 176.823984 ms
;;; Empirical mean: 20.522999999999996
;;; Empirical variance: 16.237471000000003
;;; Empirical std: 4.029574543298585
;;; Empirical skewness: 0.17489431272268954
;;; Empirical kurtosis: 2.2657758099575296
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 166.118775 ms
;;; Empirical mean: 21.302
;;; Empirical variance: 11.690796000000004
;;; Empirical std: 3.4191806035949615
;;; Empirical skewness: -0.05081949527392025
;;; Empirical kurtosis: 2.71858730581312
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 168.554451 ms
;;; Empirical mean: 20.093000000000004
;;; Empirical variance: 11.338351000000003
;;; Empirical std: 3.36724679820176
;;; Empirical skewness: 0.42245173229570737
;;; Empirical kurtosis: 2.9403539471425604
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 166.488166 ms
;;; Empirical mean: 20.47
;;; Empirical variance: 22.2751
;;; Empirical std: 4.719650410782561
;;; Empirical skewness: 0.30090440332006924
;;; Empirical kurtosis: 2.450455745715336
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 164.405309 ms
;;; Empirical mean: 20.23
;;; Empirical variance: 14.751100000000005
;;; Empirical std: 3.840716079066507
;;; Empirical skewness: 0.03515279303088567
;;; Empirical kurtosis: 2.728330556700511
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 166.399751 ms
;;; Empirical mean: 20.052
;;; Empirical variance: 20.357296
;;; Empirical std: 4.511906027390198
;;; Empirical skewness: 0.23714830962697842
;;; Empirical kurtosis: 3.196430267701018
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 167.90899 ms
;;; Empirical mean: 19.626
;;; Empirical variance: 14.620124
;;; Empirical std: 3.82362707386586
;;; Empirical skewness: 0.09845984769811274
;;; Empirical kurtosis: 3.196046036896643
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 167.682201 ms
;;; Empirical mean: 18.716
;;; Empirical variance: 18.783344
;;; Empirical std: 4.333975542155263
;;; Empirical skewness: 0.46335723319750793
;;; Empirical kurtosis: 2.6675749970126574
;;; 
;;; Method :lmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 176.476426 ms
;;; Empirical mean: 20.883000000000006
;;; Empirical variance: 13.427311000000007
;;; Empirical std: 3.6643295430405827
;;; Empirical skewness: 0.3922107138169885
;;; Empirical kurtosis: 2.816138974597694
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 242.790434 ms
;;; Empirical mean: 19.334999999999997
;;; Empirical variance: 12.284775000000002
;;; Empirical std: 3.504964336480473
;;; Empirical skewness: 0.06923746207304812
;;; Empirical kurtosis: 2.533398743162027
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 238.663177 ms
;;; Empirical mean: 21.901999999999994
;;; Empirical variance: 14.650396000000004
;;; Empirical std: 3.827583571915838
;;; Empirical skewness: -0.13720477924159233
;;; Empirical kurtosis: 3.142872907131973
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 234.543421 ms
;;; Empirical mean: 21.347
;;; Empirical variance: 16.978591000000005
;;; Empirical std: 4.120508585114222
;;; Empirical skewness: 0.6509466116760886
;;; Empirical kurtosis: 4.753709705677391
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 236.3649 ms
;;; Empirical mean: 20.662
;;; Empirical variance: 11.453756000000002
;;; Empirical std: 3.3843398174533244
;;; Empirical skewness: 0.08014577814178936
;;; Empirical kurtosis: 2.513696199327682
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 236.360798 ms
;;; Empirical mean: 20.077999999999996
;;; Empirical variance: 16.341916
;;; Empirical std: 4.042513574497926
;;; Empirical skewness: 0.31584549836916437
;;; Empirical kurtosis: 2.531408245043726
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 239.77444 ms
;;; Empirical mean: 19.787999999999993
;;; Empirical variance: 16.389056000000004
;;; Empirical std: 4.048339906677798
;;; Empirical skewness: -0.17349356139751831
;;; Empirical kurtosis: 2.5197027999667236
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 246.235142 ms
;;; Empirical mean: 20.836
;;; Empirical variance: 12.689104000000002
;;; Empirical std: 3.56217686253785
;;; Empirical skewness: -0.5285864988310817
;;; Empirical kurtosis: 3.196865440231497
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 237.314992 ms
;;; Empirical mean: 18.785999999999998
;;; Empirical variance: 20.650204000000002
;;; Empirical std: 4.544249553006525
;;; Empirical skewness: 0.24984346960606826
;;; Empirical kurtosis: 3.430447356617724
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 235.183559 ms
;;; Empirical mean: 20.442999999999998
;;; Empirical variance: 10.328751000000002
;;; Empirical std: 3.2138374258820255
;;; Empirical skewness: -0.5600508972720702
;;; Empirical kurtosis: 2.8273216207623144
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 223.624417 ms
;;; Empirical mean: 18.649000000000004
;;; Empirical variance: 14.157799000000008
;;; Empirical std: 3.762685078504446
;;; Empirical skewness: -0.35722011956712446
;;; Empirical kurtosis: 2.742834591067999
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 223.219031 ms
;;; Empirical mean: 21.143999999999995
;;; Empirical variance: 12.991264000000001
;;; Empirical std: 3.6043396066408615
;;; Empirical skewness: 0.023850287242171905
;;; Empirical kurtosis: 3.11097251762379
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 248.099403 ms
;;; Empirical mean: 19.886
;;; Empirical variance: 12.619004000000006
;;; Empirical std: 3.5523237465073487
;;; Empirical skewness: 0.26830343116078914
;;; Empirical kurtosis: 2.901270958434649
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 223.43943 ms
;;; Empirical mean: 21.755
;;; Empirical variance: 12.034975000000003
;;; Empirical std: 3.4691461485501014
;;; Empirical skewness: -0.2566780406599645
;;; Empirical kurtosis: 2.423455087187429
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 241.633812 ms
;;; Empirical mean: 20.798000000000002
;;; Empirical variance: 17.19719600000001
;;; Empirical std: 4.146950204668487
;;; Empirical skewness: -0.22323567902970637
;;; Empirical kurtosis: 2.559577311292429
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 231.679605 ms
;;; Empirical mean: 20.581
;;; Empirical variance: 13.739439000000004
;;; Empirical std: 3.706674925050753
;;; Empirical skewness: 0.15325736275284452
;;; Empirical kurtosis: 2.8137448407024883
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 228.649794 ms
;;; Empirical mean: 20.272000000000002
;;; Empirical variance: 12.232016000000002
;;; Empirical std: 3.497429913522214
;;; Empirical skewness: 0.01562743403260132
;;; Empirical kurtosis: 2.8278374339499304
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 227.670379 ms
;;; Empirical mean: 21.029000000000007
;;; Empirical variance: 14.362159000000007
;;; Empirical std: 3.7897439227472884
;;; Empirical skewness: 0.07303964306282056
;;; Empirical kurtosis: 2.8331837345076143
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 230.755226 ms
;;; Empirical mean: 20.204
;;; Empirical variance: 16.394384000000006
;;; Empirical std: 4.048997900715682
;;; Empirical skewness: -0.0038543158601270777
;;; Empirical kurtosis: 2.714743511844695
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 226.765767 ms
;;; Empirical mean: 20.168999999999997
;;; Empirical variance: 20.084439000000007
;;; Empirical std: 4.481566578775775
;;; Empirical skewness: -0.5806552850954841
;;; Empirical kurtosis: 2.6539165115873447
;;; 
;;; Method :rmh with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 226.463776 ms
;;; Empirical mean: 19.293
;;; Empirical variance: 9.927151000000004
;;; Empirical std: 3.150738167477584
;;; Empirical skewness: 0.1621556449163015
;;; Empirical kurtosis: 2.6792942931619113
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 176.371654 ms
;;; Empirical mean: 20.371054950887835
;;; Empirical variance: 17.322076441298787
;;; Empirical std: 4.161979870362035
;;; Empirical skewness: -0.04351601804308625
;;; Empirical kurtosis: 2.578275216328658
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 170.475366 ms
;;; Empirical mean: 20.30715647878284
;;; Empirical variance: 17.81033355042083
;;; Empirical std: 4.220229087433623
;;; Empirical skewness: 0.1012215255469589
;;; Empirical kurtosis: 2.7431011621618926
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 177.230469 ms
;;; Empirical mean: 20.35855645724719
;;; Empirical variance: 18.24443855204427
;;; Empirical std: 4.271350904812699
;;; Empirical skewness: 0.2862958843887747
;;; Empirical kurtosis: 3.156264952230588
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 172.549593 ms
;;; Empirical mean: 20.243373348174167
;;; Empirical variance: 16.252905470821965
;;; Empirical std: 4.03148923734418
;;; Empirical skewness: 0.18090187279816064
;;; Empirical kurtosis: 3.0365790545606384
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 176.412171 ms
;;; Empirical mean: 20.14109730036532
;;; Empirical variance: 16.103846468297792
;;; Empirical std: 4.0129598139400535
;;; Empirical skewness: 0.21862833543158697
;;; Empirical kurtosis: 3.2174142993236408
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 175.558956 ms
;;; Empirical mean: 20.325193913805276
;;; Empirical variance: 18.338357094311164
;;; Empirical std: 4.282330801597555
;;; Empirical skewness: 0.09444150550983942
;;; Empirical kurtosis: 2.9146742055627173
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 170.911932 ms
;;; Empirical mean: 20.212504046235132
;;; Empirical variance: 15.804146693518037
;;; Empirical std: 3.9754429556362694
;;; Empirical skewness: 0.19823496709150534
;;; Empirical kurtosis: 3.242214058101087
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 174.385292 ms
;;; Empirical mean: 20.148025794062224
;;; Empirical variance: 16.47358436251266
;;; Empirical std: 4.058766359685251
;;; Empirical skewness: 0.005646806234758473
;;; Empirical kurtosis: 2.823482175894652
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 176.448454 ms
;;; Empirical mean: 19.885805281866443
;;; Empirical variance: 15.281936051503218
;;; Empirical std: 3.909211691825248
;;; Empirical skewness: 0.31184248253331776
;;; Empirical kurtosis: 3.1501665704454718
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 175.28488 ms
;;; Empirical mean: 20.248761163539374
;;; Empirical variance: 16.275896822451966
;;; Empirical std: 4.034339700924051
;;; Empirical skewness: 0.1123854952266472
;;; Empirical kurtosis: 2.9494846743726417
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 173.865476 ms
;;; Empirical mean: 20.3134119017982
;;; Empirical variance: 16.483184643435173
;;; Empirical std: 4.059948847391452
;;; Empirical skewness: 0.17650971605499244
;;; Empirical kurtosis: 2.9300177025887044
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 174.046509 ms
;;; Empirical mean: 20.25832602035693
;;; Empirical variance: 17.952419193888506
;;; Empirical std: 4.237029524783667
;;; Empirical skewness: 0.21681041715791902
;;; Empirical kurtosis: 2.95304536463914
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 194.496727 ms
;;; Empirical mean: 20.221638724482506
;;; Empirical variance: 16.158766273059882
;;; Empirical std: 4.019796794995972
;;; Empirical skewness: 0.13151323799676087
;;; Empirical kurtosis: 2.9043584233120447
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 179.892898 ms
;;; Empirical mean: 20.311977827159215
;;; Empirical variance: 16.06089203096292
;;; Empirical std: 4.007604275744166
;;; Empirical skewness: 0.16175309274675956
;;; Empirical kurtosis: 3.1765802549484023
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 179.370148 ms
;;; Empirical mean: 20.43350648113064
;;; Empirical variance: 16.88876053518058
;;; Empirical std: 4.109593718992254
;;; Empirical skewness: 0.050968497339642
;;; Empirical kurtosis: 2.637491716028799
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 178.849956 ms
;;; Empirical mean: 20.217734482486094
;;; Empirical variance: 17.64938027984717
;;; Empirical std: 4.201116551566639
;;; Empirical skewness: 0.24643289880045077
;;; Empirical kurtosis: 3.1787704905922647
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 177.770667 ms
;;; Empirical mean: 20.333847214405047
;;; Empirical variance: 16.50537214577995
;;; Empirical std: 4.062680413936094
;;; Empirical skewness: 0.19455160760683457
;;; Empirical kurtosis: 3.245345562073868
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 174.551262 ms
;;; Empirical mean: 20.26728122619634
;;; Empirical variance: 15.422100382726375
;;; Empirical std: 3.927098214041301
;;; Empirical skewness: 0.03726700812912546
;;; Empirical kurtosis: 2.8056855805957004
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 189.059468 ms
;;; Empirical mean: 20.27088957385174
;;; Empirical variance: 16.75799543676356
;;; Empirical std: 4.093653067464751
;;; Empirical skewness: 0.339600262488369
;;; Empirical kurtosis: 3.441580533863961
;;; 
;;; Method :smc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 182.232931 ms
;;; Empirical mean: 20.5160867249158
;;; Empirical variance: 17.36928300624814
;;; Empirical std: 4.1676471787146445
;;; Empirical skewness: 0.05904702257813818
;;; Empirical kurtosis: 2.837857725407759
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 133.531186 ms
;;; Empirical mean: 20.068060699807067
;;; Empirical variance: 15.758306793782566
;;; Empirical std: 3.9696733862853963
;;; Empirical skewness: 0.10591749283232982
;;; Empirical kurtosis: 3.074713254475979
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 135.746479 ms
;;; Empirical mean: 20.15322578142363
;;; Empirical variance: 16.043058647100274
;;; Empirical std: 4.005378714566236
;;; Empirical skewness: 0.06827832763581004
;;; Empirical kurtosis: 2.898186650704582
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 139.15409 ms
;;; Empirical mean: 20.201499763752185
;;; Empirical variance: 14.802533342342372
;;; Empirical std: 3.8474060537383328
;;; Empirical skewness: 0.20899140476576136
;;; Empirical kurtosis: 2.9748593934926397
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 135.819903 ms
;;; Empirical mean: 20.329141226701072
;;; Empirical variance: 16.369403167857314
;;; Empirical std: 4.0459119080693435
;;; Empirical skewness: 0.1652452056904286
;;; Empirical kurtosis: 2.96347389439218
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 130.800839 ms
;;; Empirical mean: 20.214640327913038
;;; Empirical variance: 15.750744567994543
;;; Empirical std: 3.9687207722381457
;;; Empirical skewness: 0.2233721742001262
;;; Empirical kurtosis: 3.344638015885764
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 131.953511 ms
;;; Empirical mean: 20.201122038567984
;;; Empirical variance: 15.966524167521554
;;; Empirical std: 3.9958133299143936
;;; Empirical skewness: 0.16955641757233053
;;; Empirical kurtosis: 3.150194460732332
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 143.073029 ms
;;; Empirical mean: 20.445656526102034
;;; Empirical variance: 15.895149128046771
;;; Empirical std: 3.9868720982804016
;;; Empirical skewness: 0.11215264575697242
;;; Empirical kurtosis: 2.869267044795003
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 141.776107 ms
;;; Empirical mean: 20.389541693998442
;;; Empirical variance: 16.897536962511687
;;; Empirical std: 4.1106613777483165
;;; Empirical skewness: 0.1602636665836969
;;; Empirical kurtosis: 3.481175793429214
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 135.211155 ms
;;; Empirical mean: 20.454868435140806
;;; Empirical variance: 16.66324917470923
;;; Empirical std: 4.082064327605486
;;; Empirical skewness: 0.06374256488037491
;;; Empirical kurtosis: 3.0631096787878613
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 140.342767 ms
;;; Empirical mean: 20.217618009599008
;;; Empirical variance: 17.60255715243886
;;; Empirical std: 4.195540150259423
;;; Empirical skewness: 0.36669655572150445
;;; Empirical kurtosis: 3.165853082188017
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 133.766892 ms
;;; Empirical mean: 20.27405441833598
;;; Empirical variance: 16.322223898541168
;;; Empirical std: 4.0400772144281065
;;; Empirical skewness: 0.018492090329107473
;;; Empirical kurtosis: 2.9426611871814776
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 139.922506 ms
;;; Empirical mean: 20.25332017015035
;;; Empirical variance: 16.829417485288456
;;; Empirical std: 4.102367302581334
;;; Empirical skewness: 0.08702727166603393
;;; Empirical kurtosis: 3.0748563594120646
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 140.433107 ms
;;; Empirical mean: 20.0874938936489
;;; Empirical variance: 15.658138583847421
;;; Empirical std: 3.957036591168626
;;; Empirical skewness: 0.21276909198824967
;;; Empirical kurtosis: 3.099824953364727
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 132.864541 ms
;;; Empirical mean: 20.422820991645292
;;; Empirical variance: 16.29763820846533
;;; Empirical std: 4.037033342501066
;;; Empirical skewness: 0.17991386185970332
;;; Empirical kurtosis: 2.847684834569994
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 133.825632 ms
;;; Empirical mean: 20.25319136500231
;;; Empirical variance: 17.330688181461
;;; Empirical std: 4.1630143143473575
;;; Empirical skewness: 0.20852922098805557
;;; Empirical kurtosis: 3.3058223337183903
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 129.160784 ms
;;; Empirical mean: 20.361991816477754
;;; Empirical variance: 15.983456165164279
;;; Empirical std: 3.9979314858016615
;;; Empirical skewness: 0.17867042626388688
;;; Empirical kurtosis: 3.1500759457557455
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 143.284324 ms
;;; Empirical mean: 20.43287878372858
;;; Empirical variance: 17.91201354295019
;;; Empirical std: 4.232258681005947
;;; Empirical skewness: 0.15935222156260895
;;; Empirical kurtosis: 2.9750970259809604
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 134.602851 ms
;;; Empirical mean: 20.59191731411986
;;; Empirical variance: 15.812947018306014
;;; Empirical std: 3.976549637349698
;;; Empirical skewness: -0.04655503771162419
;;; Empirical kurtosis: 2.8583166908102693
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 137.5052 ms
;;; Empirical mean: 20.372583889228746
;;; Empirical variance: 17.066784325716974
;;; Empirical std: 4.131196476290733
;;; Empirical skewness: 0.3129462568472892
;;; Empirical kurtosis: 3.090000272963893
;;; 
;;; Method :smc with 1000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 159.855629 ms
;;; Empirical mean: 20.42395827415871
;;; Empirical variance: 15.641730234278572
;;; Empirical std: 3.95496273487862
;;; Empirical skewness: 0.051330821469426635
;;; Empirical kurtosis: 2.969190128378916
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 158.227895 ms
;;; Empirical mean: 20.356
;;; Empirical variance: 16.009264000000005
;;; Empirical std: 4.001157832428009
;;; Empirical skewness: 0.1183189131508556
;;; Empirical kurtosis: 3.235524672026266
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 152.45963 ms
;;; Empirical mean: 20.175
;;; Empirical variance: 17.888375
;;; Empirical std: 4.229465096203064
;;; Empirical skewness: 0.4692447967190715
;;; Empirical kurtosis: 3.841358449509279
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 148.449844 ms
;;; Empirical mean: 19.817999999999998
;;; Empirical variance: 14.456876000000006
;;; Empirical std: 3.8022198779134286
;;; Empirical skewness: 0.0638820536843052
;;; Empirical kurtosis: 3.1229669786263594
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 140.978211 ms
;;; Empirical mean: 20.426999999999992
;;; Empirical variance: 16.652671
;;; Empirical std: 4.080768432538166
;;; Empirical skewness: 0.023030799492808156
;;; Empirical kurtosis: 2.864443630398148
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 140.635519 ms
;;; Empirical mean: 19.909
;;; Empirical variance: 15.310719000000004
;;; Empirical std: 3.9128913861746795
;;; Empirical skewness: 0.24182570583014748
;;; Empirical kurtosis: 3.0744879853744496
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 146.651623 ms
;;; Empirical mean: 20.427999999999994
;;; Empirical variance: 15.104816000000003
;;; Empirical std: 3.886491476897898
;;; Empirical skewness: 0.21349236524124326
;;; Empirical kurtosis: 3.0401733159487025
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 152.603986 ms
;;; Empirical mean: 20.326
;;; Empirical variance: 17.807724
;;; Empirical std: 4.219919904453164
;;; Empirical skewness: 0.2217169346823137
;;; Empirical kurtosis: 3.129315646482918
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 157.984178 ms
;;; Empirical mean: 19.906
;;; Empirical variance: 16.11916400000001
;;; Empirical std: 4.014867868311485
;;; Empirical skewness: 0.054614279382913175
;;; Empirical kurtosis: 2.9525067561354064
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 151.820837 ms
;;; Empirical mean: 20.797
;;; Empirical variance: 15.331791000000003
;;; Empirical std: 3.9155830983392503
;;; Empirical skewness: -0.004223914666630593
;;; Empirical kurtosis: 3.0378752781825447
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 142.605373 ms
;;; Empirical mean: 20.983
;;; Empirical variance: 18.666711000000006
;;; Empirical std: 4.320498929521914
;;; Empirical skewness: 0.1060019796193606
;;; Empirical kurtosis: 2.926397987012357
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 143.179236 ms
;;; Empirical mean: 20.585000000000004
;;; Empirical variance: 17.710775000000005
;;; Empirical std: 4.208417160881274
;;; Empirical skewness: 0.09110417039758312
;;; Empirical kurtosis: 2.7387882346478203
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 146.714216 ms
;;; Empirical mean: 20.267000000000003
;;; Empirical variance: 15.855711000000008
;;; Empirical std: 3.981923027884895
;;; Empirical skewness: -0.03662649817124864
;;; Empirical kurtosis: 3.14599189229818
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 141.406755 ms
;;; Empirical mean: 20.52699999999999
;;; Empirical variance: 19.183270999999994
;;; Empirical std: 4.379871116825242
;;; Empirical skewness: 0.06195988564992474
;;; Empirical kurtosis: 2.517664885630598
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 147.748945 ms
;;; Empirical mean: 19.865000000000002
;;; Empirical variance: 18.168775000000007
;;; Empirical std: 4.262484604077768
;;; Empirical skewness: 0.47138262401347
;;; Empirical kurtosis: 3.5990344448041043
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 145.055451 ms
;;; Empirical mean: 20.42499999999999
;;; Empirical variance: 14.790375000000001
;;; Empirical std: 3.8458256590750444
;;; Empirical skewness: 0.14599737193287365
;;; Empirical kurtosis: 3.195113389725881
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 152.818383 ms
;;; Empirical mean: 20.164000000000005
;;; Empirical variance: 17.239104000000008
;;; Empirical std: 4.152000000000001
;;; Empirical skewness: 0.25407618250368347
;;; Empirical kurtosis: 2.839377727007761
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 151.22895 ms
;;; Empirical mean: 19.968999999999994
;;; Empirical variance: 14.650039000000005
;;; Empirical std: 3.827536936464494
;;; Empirical skewness: -0.07833489008952337
;;; Empirical kurtosis: 2.6957601111079086
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 171.192856 ms
;;; Empirical mean: 20.369000000000003
;;; Empirical variance: 17.156839
;;; Empirical std: 4.14208148157421
;;; Empirical skewness: 0.056203959854239864
;;; Empirical kurtosis: 2.9146723155534775
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 147.712707 ms
;;; Empirical mean: 20.172
;;; Empirical variance: 15.752416000000006
;;; Empirical std: 3.968931342313697
;;; Empirical skewness: -0.04384714986996631
;;; Empirical kurtosis: 2.973721207567746
;;; 
;;; Method :pgibbs with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 148.243171 ms
;;; Empirical mean: 19.956
;;; Empirical variance: 15.010064000000007
;;; Empirical std: 3.8742823851650265
;;; Empirical skewness: -0.10143179867037451
;;; Empirical kurtosis: 2.6872705566377
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 85.662225 ms
;;; Empirical mean: 20.879542843705135
;;; Empirical variance: 16.13148839817346
;;; Empirical std: 4.016402419849567
;;; Empirical skewness: 0.27291157642936764
;;; Empirical kurtosis: 3.0752709255394812
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 101.643202 ms
;;; Empirical mean: 20.67028098595266
;;; Empirical variance: 17.251964949039035
;;; Empirical std: 4.1535484767893385
;;; Empirical skewness: 0.19946805066319742
;;; Empirical kurtosis: 2.739983391873732
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 69.982709 ms
;;; Empirical mean: 20.439051494916306
;;; Empirical variance: 13.849238643731194
;;; Empirical std: 3.7214565218112106
;;; Empirical skewness: -0.019943750011987917
;;; Empirical kurtosis: 2.944897102291791
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 82.5422 ms
;;; Empirical mean: 20.60581295423096
;;; Empirical variance: 16.58770100226243
;;; Empirical std: 4.072800142685917
;;; Empirical skewness: 0.14836298354930755
;;; Empirical kurtosis: 3.076628103285989
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 79.880902 ms
;;; Empirical mean: 20.425635908463125
;;; Empirical variance: 15.052951103961545
;;; Empirical std: 3.8798132821002542
;;; Empirical skewness: 0.168886149536477
;;; Empirical kurtosis: 2.8803161027457396
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 82.34447 ms
;;; Empirical mean: 20.58867814912788
;;; Empirical variance: 14.331811344026505
;;; Empirical std: 3.785737886334249
;;; Empirical skewness: -0.011526139125356563
;;; Empirical kurtosis: 3.1924639948332247
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 83.076077 ms
;;; Empirical mean: 20.843347924086167
;;; Empirical variance: 14.453820273347343
;;; Empirical std: 3.8018180221240656
;;; Empirical skewness: -3.755329481323796E-4
;;; Empirical kurtosis: 2.866658925230498
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 81.883337 ms
;;; Empirical mean: 20.590612230319454
;;; Empirical variance: 15.509911747530769
;;; Empirical std: 3.9382625290260638
;;; Empirical skewness: 0.1978396131233136
;;; Empirical kurtosis: 3.056925020583143
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 81.27308 ms
;;; Empirical mean: 21.099714601487154
;;; Empirical variance: 15.657398121426201
;;; Empirical std: 3.9569430273161883
;;; Empirical skewness: 0.03465120358165374
;;; Empirical kurtosis: 2.765146821319933
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 76.14604 ms
;;; Empirical mean: 20.284924159757907
;;; Empirical variance: 15.673679911711332
;;; Empirical std: 3.958999862555104
;;; Empirical skewness: 0.12700255785361994
;;; Empirical kurtosis: 2.821795227807111
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 76.525927 ms
;;; Empirical mean: 20.586961172075927
;;; Empirical variance: 17.204086753022153
;;; Empirical std: 4.147780943230025
;;; Empirical skewness: -0.15357264129822315
;;; Empirical kurtosis: 2.755482223774121
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 78.912556 ms
;;; Empirical mean: 20.704509231954976
;;; Empirical variance: 17.923724900331454
;;; Empirical std: 4.23364203733989
;;; Empirical skewness: -0.2142396532458577
;;; Empirical kurtosis: 2.7065058513828855
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 78.980396 ms
;;; Empirical mean: 20.54059305677612
;;; Empirical variance: 17.400583717548184
;;; Empirical std: 4.171400690121747
;;; Empirical skewness: 0.3442500561967052
;;; Empirical kurtosis: 3.61583305012919
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 75.339276 ms
;;; Empirical mean: 19.95812596578914
;;; Empirical variance: 15.385482143072558
;;; Empirical std: 3.9224331916646533
;;; Empirical skewness: -0.0960306646328985
;;; Empirical kurtosis: 2.8759544573910527
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 78.498591 ms
;;; Empirical mean: 20.238044687745187
;;; Empirical variance: 16.267625932463314
;;; Empirical std: 4.0333145094900935
;;; Empirical skewness: 0.24710031277989833
;;; Empirical kurtosis: 2.8319807145703293
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 82.020519 ms
;;; Empirical mean: 20.13423222980618
;;; Empirical variance: 15.914957592787925
;;; Empirical std: 3.989355536021818
;;; Empirical skewness: 0.06812293096202815
;;; Empirical kurtosis: 2.8381197997379504
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 74.765612 ms
;;; Empirical mean: 21.010607503084916
;;; Empirical variance: 16.61787061779902
;;; Empirical std: 4.076502252887765
;;; Empirical skewness: 0.5024828985346123
;;; Empirical kurtosis: 4.447214663202028
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 80.720268 ms
;;; Empirical mean: 20.39594826594001
;;; Empirical variance: 18.98023075615992
;;; Empirical std: 4.356630665567133
;;; Empirical skewness: 0.42537941682856395
;;; Empirical kurtosis: 3.3232765072674053
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 73.568779 ms
;;; Empirical mean: 20.857690593333142
;;; Empirical variance: 17.920208652979646
;;; Empirical std: 4.233226742448323
;;; Empirical skewness: -0.05624040894644164
;;; Empirical kurtosis: 2.6046473956055767
;;; 
;;; Method :ipmcmc with 1000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 81.999304 ms
;;; Empirical mean: 20.640558851151113
;;; Empirical variance: 16.883492194793735
;;; Empirical std: 4.1089526883128915
;;; Empirical skewness: 0.04032094451319161
;;; Empirical kurtosis: 2.512010122741725
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1315.619774 ms
;;; Empirical mean: 20.300949092679534
;;; Empirical variance: 17.119111759159132
;;; Empirical std: 4.137524834869168
;;; Empirical skewness: 0.156320229333996
;;; Empirical kurtosis: 2.9917996901078716
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1256.700851 ms
;;; Empirical mean: 20.33684969936588
;;; Empirical variance: 16.58633538366392
;;; Empirical std: 4.072632488165844
;;; Empirical skewness: 0.13534021895531664
;;; Empirical kurtosis: 3.080266110724953
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1263.598404 ms
;;; Empirical mean: 20.28852417115419
;;; Empirical variance: 16.522163252100924
;;; Empirical std: 4.064746394561526
;;; Empirical skewness: 0.10634484377740555
;;; Empirical kurtosis: 2.9619303416928275
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1248.527635 ms
;;; Empirical mean: 20.23039636369964
;;; Empirical variance: 16.783975163060674
;;; Empirical std: 4.096825010060922
;;; Empirical skewness: 0.1472082668184053
;;; Empirical kurtosis: 3.0290909875685723
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1266.2955 ms
;;; Empirical mean: 20.3908014823224
;;; Empirical variance: 16.91904349272684
;;; Empirical std: 4.113276491159674
;;; Empirical skewness: 0.10106861382017017
;;; Empirical kurtosis: 2.995344380130304
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1302.472161 ms
;;; Empirical mean: 20.258002901473436
;;; Empirical variance: 16.372306293194434
;;; Empirical std: 4.0462706648461415
;;; Empirical skewness: 0.1809681869688679
;;; Empirical kurtosis: 3.0821208298661453
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1281.86073 ms
;;; Empirical mean: 20.34694102719338
;;; Empirical variance: 17.040755260618756
;;; Empirical std: 4.128044968337767
;;; Empirical skewness: 0.16161188023472797
;;; Empirical kurtosis: 3.025603572177013
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1317.341653 ms
;;; Empirical mean: 20.24790703853765
;;; Empirical variance: 16.646946428635065
;;; Empirical std: 4.0800669637439855
;;; Empirical skewness: 0.1333392403715399
;;; Empirical kurtosis: 3.056642163676445
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1277.731784 ms
;;; Empirical mean: 20.347752998192412
;;; Empirical variance: 16.967798027463317
;;; Empirical std: 4.119198711820458
;;; Empirical skewness: 0.11189547176091433
;;; Empirical kurtosis: 3.0602980336061725
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1259.051854 ms
;;; Empirical mean: 20.276308102375726
;;; Empirical variance: 16.775139136769265
;;; Empirical std: 4.095746468809961
;;; Empirical skewness: 0.13079019915875537
;;; Empirical kurtosis: 2.9836688645809697
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1340.987738 ms
;;; Empirical mean: 20.26537065675575
;;; Empirical variance: 16.80512094489351
;;; Empirical std: 4.099404950098673
;;; Empirical skewness: 0.13248688362295216
;;; Empirical kurtosis: 3.0648660666564083
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1319.491394 ms
;;; Empirical mean: 20.299589820745588
;;; Empirical variance: 17.283757642957177
;;; Empirical std: 4.157373887799506
;;; Empirical skewness: 0.18785629246908775
;;; Empirical kurtosis: 3.0419642377057103
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1324.671489 ms
;;; Empirical mean: 20.315771119885326
;;; Empirical variance: 16.789038310777972
;;; Empirical std: 4.097442899025925
;;; Empirical skewness: 0.1450853983397923
;;; Empirical kurtosis: 3.025495933719997
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1282.761703 ms
;;; Empirical mean: 20.280923424974333
;;; Empirical variance: 16.454650818086446
;;; Empirical std: 4.056433263112614
;;; Empirical skewness: 0.12185218272076649
;;; Empirical kurtosis: 2.9997877840669993
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1245.08268 ms
;;; Empirical mean: 20.357326432055327
;;; Empirical variance: 16.330083446523904
;;; Empirical std: 4.04104979510571
;;; Empirical skewness: 0.1386800441237657
;;; Empirical kurtosis: 3.141421221692747
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1278.106542 ms
;;; Empirical mean: 20.28382953011444
;;; Empirical variance: 16.874175033042928
;;; Empirical std: 4.107818768281158
;;; Empirical skewness: 0.16539420669417437
;;; Empirical kurtosis: 3.0383779155714143
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1290.432526 ms
;;; Empirical mean: 20.30442855485448
;;; Empirical variance: 16.575328935398115
;;; Empirical std: 4.071280994404355
;;; Empirical skewness: 0.13335946425930942
;;; Empirical kurtosis: 3.020696678228207
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1285.913924 ms
;;; Empirical mean: 20.29365729722867
;;; Empirical variance: 16.351554188569903
;;; Empirical std: 4.043705502205855
;;; Empirical skewness: 0.17653353312663625
;;; Empirical kurtosis: 3.10993147451141
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1258.898038 ms
;;; Empirical mean: 20.26510541619784
;;; Empirical variance: 16.594875713763482
;;; Empirical std: 4.073680855659103
;;; Empirical skewness: 0.12521889495026126
;;; Empirical kurtosis: 2.943475545422639
;;; 
;;; Method :importance with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1273.48019 ms
;;; Empirical mean: 20.282370136266408
;;; Empirical variance: 16.66080756471671
;;; Empirical std: 4.081765251054586
;;; Empirical skewness: 0.16781033449549745
;;; Empirical kurtosis: 3.103522246543991
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1760.09909 ms
;;; Empirical mean: 20.865299999999998
;;; Empirical variance: 17.229955909999983
;;; Empirical std: 4.150898205208119
;;; Empirical skewness: 0.03189563468334432
;;; Empirical kurtosis: 3.105378266767012
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1705.444528 ms
;;; Empirical mean: 19.791600000000006
;;; Empirical variance: 17.521569439999993
;;; Empirical std: 4.185877379952737
;;; Empirical skewness: -0.033490028077397815
;;; Empirical kurtosis: 3.164574162516263
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1745.51953 ms
;;; Empirical mean: 20.3733
;;; Empirical variance: 16.372747109999978
;;; Empirical std: 4.046325136466418
;;; Empirical skewness: 0.05654748250556101
;;; Empirical kurtosis: 3.1745262941073764
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1705.621962 ms
;;; Empirical mean: 20.209999999999994
;;; Empirical variance: 16.19549999999999
;;; Empirical std: 4.024363303679228
;;; Empirical skewness: 0.16355222494123958
;;; Empirical kurtosis: 2.8900984360519884
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1746.452948 ms
;;; Empirical mean: 20.514699999999998
;;; Empirical variance: 17.244383909999986
;;; Empirical std: 4.152635778635057
;;; Empirical skewness: 0.07871138650961285
;;; Empirical kurtosis: 2.98128682732093
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1691.062271 ms
;;; Empirical mean: 20.344599999999996
;;; Empirical variance: 16.032650839999988
;;; Empirical std: 4.004079274939494
;;; Empirical skewness: 0.032729834242536206
;;; Empirical kurtosis: 3.219827852706676
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1704.995322 ms
;;; Empirical mean: 20.203999999999997
;;; Empirical variance: 17.657383999999986
;;; Empirical std: 4.202069014188128
;;; Empirical skewness: 0.10884714851112576
;;; Empirical kurtosis: 3.169613009969878
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1679.466258 ms
;;; Empirical mean: 19.730099999999993
;;; Empirical variance: 14.69705398999999
;;; Empirical std: 3.833673693730335
;;; Empirical skewness: 0.26074544873933525
;;; Empirical kurtosis: 3.055713353326113
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1746.222001 ms
;;; Empirical mean: 20.1098
;;; Empirical variance: 15.534143959999986
;;; Empirical std: 3.9413378388562412
;;; Empirical skewness: 0.1182113949153919
;;; Empirical kurtosis: 3.079285541374175
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1708.674373 ms
;;; Empirical mean: 20.479999999999997
;;; Empirical variance: 14.12879999999998
;;; Empirical std: 3.7588296050765564
;;; Empirical skewness: 0.12939986681582535
;;; Empirical kurtosis: 3.0223711371747726
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1699.498449 ms
;;; Empirical mean: 20.281699999999997
;;; Empirical variance: 16.885945109999987
;;; Empirical std: 4.109251161708175
;;; Empirical skewness: 0.08055839656760994
;;; Empirical kurtosis: 3.015091292109166
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1714.348475 ms
;;; Empirical mean: 20.232299999999995
;;; Empirical variance: 16.352336709999985
;;; Empirical std: 4.043802259013166
;;; Empirical skewness: 0.1712561869004935
;;; Empirical kurtosis: 2.974860904535725
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1676.96439 ms
;;; Empirical mean: 20.291900000000002
;;; Empirical variance: 16.63649438999998
;;; Empirical std: 4.078785896562846
;;; Empirical skewness: 0.2886443155111419
;;; Empirical kurtosis: 3.1795021106320736
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1646.561192 ms
;;; Empirical mean: 20.019900000000007
;;; Empirical variance: 17.144103989999987
;;; Empirical std: 4.1405439244137945
;;; Empirical skewness: 0.353227663457634
;;; Empirical kurtosis: 3.397904307616107
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1679.892318 ms
;;; Empirical mean: 20.018900000000002
;;; Empirical variance: 17.91334278999999
;;; Empirical std: 4.232415715640418
;;; Empirical skewness: 0.12395956344213489
;;; Empirical kurtosis: 2.990611214344528
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1727.398447 ms
;;; Empirical mean: 20.0829
;;; Empirical variance: 18.082827589999972
;;; Empirical std: 4.252390808709845
;;; Empirical skewness: 0.2679480344841059
;;; Empirical kurtosis: 3.1180010138619383
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1689.851084 ms
;;; Empirical mean: 19.754600000000003
;;; Empirical variance: 16.88297883999999
;;; Empirical std: 4.108890219998581
;;; Empirical skewness: 0.07338914214525796
;;; Empirical kurtosis: 3.152586632300663
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1696.629192 ms
;;; Empirical mean: 20.674
;;; Empirical variance: 17.694923999999986
;;; Empirical std: 4.206533489703842
;;; Empirical skewness: 0.13361693372839753
;;; Empirical kurtosis: 3.1353299107033448
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1655.532792 ms
;;; Empirical mean: 20.292
;;; Empirical variance: 16.53093599999999
;;; Empirical std: 4.065825377460275
;;; Empirical skewness: 0.08172664437983143
;;; Empirical kurtosis: 3.1721985120611635
;;; 
;;; Method :lmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1686.603927 ms
;;; Empirical mean: 20.1038
;;; Empirical variance: 13.631025559999989
;;; Empirical std: 3.6920218796751447
;;; Empirical skewness: 0.1732975504302398
;;; Empirical kurtosis: 2.975929549545818
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 2310.148419 ms
;;; Empirical mean: 20.73820000000001
;;; Empirical variance: 17.21886075999999
;;; Empirical std: 4.149561514184359
;;; Empirical skewness: 0.4342130628895222
;;; Empirical kurtosis: 3.3161957294816657
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 2322.404391 ms
;;; Empirical mean: 20.326500000000006
;;; Empirical variance: 15.849497749999987
;;; Empirical std: 3.981142769356556
;;; Empirical skewness: 0.18873975693278716
;;; Empirical kurtosis: 2.806555567369903
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 2309.223866 ms
;;; Empirical mean: 20.74750000000001
;;; Empirical variance: 16.49614374999999
;;; Empirical std: 4.061544503018524
;;; Empirical skewness: 0.1767441613391321
;;; Empirical kurtosis: 3.159844812416628
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 2275.693307 ms
;;; Empirical mean: 20.692
;;; Empirical variance: 16.392935999999985
;;; Empirical std: 4.048819087091937
;;; Empirical skewness: 0.31457664159558857
;;; Empirical kurtosis: 2.836661044634023
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 2288.036714 ms
;;; Empirical mean: 20.0389
;;; Empirical variance: 14.15818678999999
;;; Empirical std: 3.7627366091715735
;;; Empirical skewness: -0.23966911523425563
;;; Empirical kurtosis: 3.2241403980904972
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 2533.226257 ms
;;; Empirical mean: 20.030600000000007
;;; Empirical variance: 15.58506363999999
;;; Empirical std: 3.947792248839849
;;; Empirical skewness: 0.16285092259671832
;;; Empirical kurtosis: 3.0682214992038
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 2530.339636 ms
;;; Empirical mean: 19.889100000000003
;;; Empirical variance: 15.132001189999983
;;; Empirical std: 3.8899872994651257
;;; Empirical skewness: 0.30735420977295924
;;; Empirical kurtosis: 3.5551616012212315
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 2571.171152 ms
;;; Empirical mean: 20.294700000000002
;;; Empirical variance: 15.896451909999987
;;; Empirical std: 3.9870354788990765
;;; Empirical skewness: 0.16717418615537183
;;; Empirical kurtosis: 3.0692038929682828
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 2825.211534 ms
;;; Empirical mean: 20.191399999999998
;;; Empirical variance: 16.384766039999988
;;; Empirical std: 4.0478100301274
;;; Empirical skewness: 0.011273044564767097
;;; Empirical kurtosis: 2.968231255381178
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 3054.814378 ms
;;; Empirical mean: 20.4304
;;; Empirical variance: 16.460955839999993
;;; Empirical std: 4.057210351953667
;;; Empirical skewness: 0.04264222257459788
;;; Empirical kurtosis: 2.9209545009120186
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 2688.448145 ms
;;; Empirical mean: 20.1761
;;; Empirical variance: 20.259888789999977
;;; Empirical std: 4.501098620337037
;;; Empirical skewness: -0.026353920444429302
;;; Empirical kurtosis: 2.735610362752046
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 2787.397417 ms
;;; Empirical mean: 20.181399999999996
;;; Empirical variance: 15.95229403999998
;;; Empirical std: 3.994032303324546
;;; Empirical skewness: 0.146102829821205
;;; Empirical kurtosis: 3.064275880343437
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 2341.822961 ms
;;; Empirical mean: 20.4382
;;; Empirical variance: 19.577780759999982
;;; Empirical std: 4.424678605277448
;;; Empirical skewness: 0.17904804814789926
;;; Empirical kurtosis: 2.7115779370219535
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 2374.653695 ms
;;; Empirical mean: 20.599
;;; Empirical variance: 15.765998999999985
;;; Empirical std: 3.9706421395033806
;;; Empirical skewness: 0.1703352047242129
;;; Empirical kurtosis: 2.9278607250170614
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 2361.010099 ms
;;; Empirical mean: 20.297099999999997
;;; Empirical variance: 18.45023158999998
;;; Empirical std: 4.295373277143673
;;; Empirical skewness: 0.33968163347068986
;;; Empirical kurtosis: 3.0837188663532724
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 2476.430817 ms
;;; Empirical mean: 20.110100000000003
;;; Empirical variance: 17.14097798999999
;;; Empirical std: 4.140166420568138
;;; Empirical skewness: -0.14344091155243838
;;; Empirical kurtosis: 2.88772974955158
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 2635.504135 ms
;;; Empirical mean: 20.3504
;;; Empirical variance: 15.88461983999999
;;; Empirical std: 3.9855513846894497
;;; Empirical skewness: 0.23958162313939185
;;; Empirical kurtosis: 3.2422705602250073
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 2450.263149 ms
;;; Empirical mean: 20.6206
;;; Empirical variance: 16.429055639999984
;;; Empirical std: 4.053277148185155
;;; Empirical skewness: 0.038785175392375394
;;; Empirical kurtosis: 2.9957981226508945
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 2352.628457 ms
;;; Empirical mean: 20.527199999999997
;;; Empirical variance: 15.799260159999987
;;; Empirical std: 3.9748283183050797
;;; Empirical skewness: 0.2559351505050436
;;; Empirical kurtosis: 3.035087433214836
;;; 
;;; Method :rmh with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 2346.987088 ms
;;; Empirical mean: 20.269700000000004
;;; Empirical variance: 17.223561909999983
;;; Empirical std: 4.1501279389917585
;;; Empirical skewness: 0.09213905893301642
;;; Empirical kurtosis: 2.7885941236196525
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1758.628322 ms
;;; Empirical mean: 20.302995541151347
;;; Empirical variance: 17.018765849142866
;;; Empirical std: 4.125380691420231
;;; Empirical skewness: 0.19164724683350157
;;; Empirical kurtosis: 3.089016266056877
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1751.573012 ms
;;; Empirical mean: 20.233684061474126
;;; Empirical variance: 16.66866852829605
;;; Empirical std: 4.082728074253299
;;; Empirical skewness: 0.1606028420985357
;;; Empirical kurtosis: 3.0217603883933317
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1730.289799 ms
;;; Empirical mean: 20.23756358082249
;;; Empirical variance: 16.68482810927683
;;; Empirical std: 4.084706612386847
;;; Empirical skewness: 0.14322541982290934
;;; Empirical kurtosis: 2.983713716456029
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1716.479606 ms
;;; Empirical mean: 20.288395118518313
;;; Empirical variance: 16.815195008871147
;;; Empirical std: 4.100633488727217
;;; Empirical skewness: 0.12906824241236844
;;; Empirical kurtosis: 3.0325867522628105
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1754.941786 ms
;;; Empirical mean: 20.237464913178748
;;; Empirical variance: 16.413580512778406
;;; Empirical std: 4.0513677335905225
;;; Empirical skewness: 0.16632763654044116
;;; Empirical kurtosis: 3.0991800924700414
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1741.721698 ms
;;; Empirical mean: 20.255494497839482
;;; Empirical variance: 16.603576350874885
;;; Empirical std: 4.074748624255843
;;; Empirical skewness: 0.18584928688125016
;;; Empirical kurtosis: 3.1325430303168305
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1776.385188 ms
;;; Empirical mean: 20.292108670239134
;;; Empirical variance: 16.7232740925258
;;; Empirical std: 4.089409993205107
;;; Empirical skewness: 0.14095657241623008
;;; Empirical kurtosis: 3.0209680240016152
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1727.191311 ms
;;; Empirical mean: 20.32869526946304
;;; Empirical variance: 16.532837190499887
;;; Empirical std: 4.066059172036222
;;; Empirical skewness: 0.12920413407483472
;;; Empirical kurtosis: 3.0562510191229055
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1738.074147 ms
;;; Empirical mean: 20.348188878184207
;;; Empirical variance: 17.02621288492671
;;; Empirical std: 4.12628318040906
;;; Empirical skewness: 0.13181759932591103
;;; Empirical kurtosis: 3.0246197021533843
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1720.007154 ms
;;; Empirical mean: 20.28722399967613
;;; Empirical variance: 16.677995169387096
;;; Empirical std: 4.08387012151306
;;; Empirical skewness: 0.14740934042451567
;;; Empirical kurtosis: 3.075179071287314
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1727.587809 ms
;;; Empirical mean: 20.22489667482509
;;; Empirical variance: 17.260861403570072
;;; Empirical std: 4.1546192850332355
;;; Empirical skewness: 0.17816587493056296
;;; Empirical kurtosis: 3.013060181413364
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1724.883231 ms
;;; Empirical mean: 20.243022319972646
;;; Empirical variance: 16.373402613203623
;;; Empirical std: 4.046406135474246
;;; Empirical skewness: 0.11556779283024919
;;; Empirical kurtosis: 2.9905633432814414
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1734.163589 ms
;;; Empirical mean: 20.381778729864234
;;; Empirical variance: 16.856037792079192
;;; Empirical std: 4.105610526106829
;;; Empirical skewness: 0.10637803783807555
;;; Empirical kurtosis: 3.0147914839506784
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1766.633489 ms
;;; Empirical mean: 20.262917462135636
;;; Empirical variance: 16.491372727705684
;;; Empirical std: 4.0609571196585765
;;; Empirical skewness: 0.13189961168910608
;;; Empirical kurtosis: 3.0961637904503454
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1784.138501 ms
;;; Empirical mean: 20.260353625959304
;;; Empirical variance: 16.40575070358064
;;; Empirical std: 4.050401301547865
;;; Empirical skewness: 0.1828946791943931
;;; Empirical kurtosis: 3.1231967860272043
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1823.053067 ms
;;; Empirical mean: 20.25486328181863
;;; Empirical variance: 16.98624272577466
;;; Empirical std: 4.12143697340802
;;; Empirical skewness: 0.1649491500519644
;;; Empirical kurtosis: 3.069475696251274
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1801.938977 ms
;;; Empirical mean: 20.304716958676885
;;; Empirical variance: 16.730693678496003
;;; Empirical std: 4.090317063321131
;;; Empirical skewness: 0.1508177776188954
;;; Empirical kurtosis: 3.0919598332974547
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1761.772574 ms
;;; Empirical mean: 20.319170266676522
;;; Empirical variance: 16.84237062388054
;;; Empirical std: 4.1039457384181555
;;; Empirical skewness: 0.14137071007143087
;;; Empirical kurtosis: 3.0015527179343726
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1747.702188 ms
;;; Empirical mean: 20.306574894848087
;;; Empirical variance: 17.116154418357755
;;; Empirical std: 4.137167439004343
;;; Empirical skewness: 0.17774610674004215
;;; Empirical kurtosis: 3.0128435938501665
;;; 
;;; Method :smc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1812.647564 ms
;;; Empirical mean: 20.300911375997615
;;; Empirical variance: 17.02313588694331
;;; Empirical std: 4.1259103100944055
;;; Empirical skewness: 0.13777152239575663
;;; Empirical kurtosis: 2.9565722058862947
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 0
;;; Elapsed time: 1294.069717 ms
;;; Empirical mean: 20.31146279059244
;;; Empirical variance: 16.681762436531518
;;; Empirical std: 4.084331332853827
;;; Empirical skewness: 0.18801739139270135
;;; Empirical kurtosis: 3.213620730575008
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 1
;;; Elapsed time: 1390.948314 ms
;;; Empirical mean: 20.293273951393125
;;; Empirical variance: 16.463104826243
;;; Empirical std: 4.057475178758707
;;; Empirical skewness: 0.1590843177067797
;;; Empirical kurtosis: 3.119469737155567
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 2
;;; Elapsed time: 1310.848812 ms
;;; Empirical mean: 20.30926003853298
;;; Empirical variance: 16.696313340684952
;;; Empirical std: 4.086112252580067
;;; Empirical skewness: 0.1603970160169159
;;; Empirical kurtosis: 3.1026818539808465
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 3
;;; Elapsed time: 1347.267412 ms
;;; Empirical mean: 20.267938371322966
;;; Empirical variance: 16.565139072910657
;;; Empirical std: 4.070029370030474
;;; Empirical skewness: 0.15551767822106255
;;; Empirical kurtosis: 3.0586066055422507
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 4
;;; Elapsed time: 1344.76697 ms
;;; Empirical mean: 20.26774487779398
;;; Empirical variance: 16.251424430045812
;;; Empirical std: 4.031305549080324
;;; Empirical skewness: 0.11277653215873545
;;; Empirical kurtosis: 3.137481232286516
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 5
;;; Elapsed time: 1344.269499 ms
;;; Empirical mean: 20.298430714024757
;;; Empirical variance: 16.532246092534358
;;; Empirical std: 4.065986484548905
;;; Empirical skewness: 0.14977614660631758
;;; Empirical kurtosis: 2.969859950275484
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 6
;;; Elapsed time: 1399.711253 ms
;;; Empirical mean: 20.31710457443268
;;; Empirical variance: 17.16663611897366
;;; Empirical std: 4.143263945125106
;;; Empirical skewness: 0.08779188593537314
;;; Empirical kurtosis: 3.012741294137741
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 7
;;; Elapsed time: 1309.83527 ms
;;; Empirical mean: 20.337783018134953
;;; Empirical variance: 16.649717741523165
;;; Empirical std: 4.080406565714153
;;; Empirical skewness: 0.1632965228938127
;;; Empirical kurtosis: 3.099531009854892
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 8
;;; Elapsed time: 1290.776372 ms
;;; Empirical mean: 20.393631521046796
;;; Empirical variance: 16.985702311879127
;;; Empirical std: 4.121371411542416
;;; Empirical skewness: 0.11307659649116616
;;; Empirical kurtosis: 3.095626751997766
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 9
;;; Elapsed time: 1284.300459 ms
;;; Empirical mean: 20.324656783787393
;;; Empirical variance: 16.287016729074796
;;; Empirical std: 4.035717622564145
;;; Empirical skewness: 0.1714149586693414
;;; Empirical kurtosis: 3.0886071933684254
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 10
;;; Elapsed time: 1408.536527 ms
;;; Empirical mean: 20.26573440528244
;;; Empirical variance: 16.538479567230414
;;; Empirical std: 4.066752951339732
;;; Empirical skewness: 0.13290884527953406
;;; Empirical kurtosis: 3.020748560926667
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 11
;;; Elapsed time: 1296.374391 ms
;;; Empirical mean: 20.29273626486577
;;; Empirical variance: 16.901693846414798
;;; Empirical std: 4.111166968929235
;;; Empirical skewness: 0.15141019790621765
;;; Empirical kurtosis: 3.044697875421657
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 12
;;; Elapsed time: 1426.682609 ms
;;; Empirical mean: 20.232150347887767
;;; Empirical variance: 16.578026207463616
;;; Empirical std: 4.071612236874186
;;; Empirical skewness: 0.13322573521927467
;;; Empirical kurtosis: 3.1094719772782375
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 13
;;; Elapsed time: 1365.318685 ms
;;; Empirical mean: 20.280072234577762
;;; Empirical variance: 16.66668577647627
;;; Empirical std: 4.08248524510209
;;; Empirical skewness: 0.16844496798124325
;;; Empirical kurtosis: 3.0515736101143465
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 14
;;; Elapsed time: 1338.217646 ms
;;; Empirical mean: 20.302044880536872
;;; Empirical variance: 16.50907507628648
;;; Empirical std: 4.063136113433376
;;; Empirical skewness: 0.13093906030706665
;;; Empirical kurtosis: 3.021781493411969
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 15
;;; Elapsed time: 1430.481176 ms
;;; Empirical mean: 20.238392884941227
;;; Empirical variance: 16.393901561312223
;;; Empirical std: 4.048938325204796
;;; Empirical skewness: 0.1086682060330764
;;; Empirical kurtosis: 3.0607787873395305
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 16
;;; Elapsed time: 1344.831222 ms
;;; Empirical mean: 20.309180312762038
;;; Empirical variance: 17.184324475307584
;;; Empirical std: 4.145397987564955
;;; Empirical skewness: 0.1617429246677632
;;; Empirical kurtosis: 3.0562283399862222
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 17
;;; Elapsed time: 1409.077824 ms
;;; Empirical mean: 20.264898642832417
;;; Empirical variance: 16.612782489526957
;;; Empirical std: 4.0758781249599405
;;; Empirical skewness: 0.17048293057664027
;;; Empirical kurtosis: 3.0958914660417447
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 18
;;; Elapsed time: 1331.187057 ms
;;; Empirical mean: 20.278524153672254
;;; Empirical variance: 16.39477971977109
;;; Empirical std: 4.049046766804638
;;; Empirical skewness: 0.1408472574887886
;;; Empirical kurtosis: 3.0326111140863135
;;; 
;;; Method :smc with 10000 samples and options [:number-of-particles 100]
;;; Chain no. 19
;;; Elapsed time: 1458.724704 ms
;;; Empirical mean: 20.34896647686211
;;; Empirical variance: 17.61728192396339
;;; Empirical std: 4.197294595803753
;;; Empirical skewness: 0.22220070673520018
;;; Empirical kurtosis: 3.0727625151818163
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 1457.359176 ms
;;; Empirical mean: 20.31850000000001
;;; Empirical variance: 16.830057749999995
;;; Empirical std: 4.102445337844246
;;; Empirical skewness: 0.11255469479167422
;;; Empirical kurtosis: 2.978656394531978
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 1454.235701 ms
;;; Empirical mean: 20.2535
;;; Empirical variance: 15.955037749999988
;;; Empirical std: 3.994375764747226
;;; Empirical skewness: 0.15101428638878056
;;; Empirical kurtosis: 3.1825004330701403
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 1468.083539 ms
;;; Empirical mean: 20.3242
;;; Empirical variance: 16.206894359999986
;;; Empirical std: 4.025778727153293
;;; Empirical skewness: 0.15767707671233702
;;; Empirical kurtosis: 3.0226547489336157
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 1480.255938 ms
;;; Empirical mean: 20.285099999999996
;;; Empirical variance: 16.592017989999988
;;; Empirical std: 4.073330086059806
;;; Empirical skewness: 0.1336243955203499
;;; Empirical kurtosis: 2.940747184948381
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 1479.485822 ms
;;; Empirical mean: 20.2449
;;; Empirical variance: 16.821123989999986
;;; Empirical std: 4.101356359791232
;;; Empirical skewness: 0.17194131240450752
;;; Empirical kurtosis: 3.0403461016551634
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 1448.155208 ms
;;; Empirical mean: 20.269099999999998
;;; Empirical variance: 16.790285189999977
;;; Empirical std: 4.097595049538202
;;; Empirical skewness: 0.1477041720375121
;;; Empirical kurtosis: 3.031154359312019
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 1469.098688 ms
;;; Empirical mean: 20.302200000000003
;;; Empirical variance: 16.868675159999984
;;; Empirical std: 4.107149274131631
;;; Empirical skewness: 0.17303241603078595
;;; Empirical kurtosis: 3.139690054785741
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 1450.141701 ms
;;; Empirical mean: 20.140100000000004
;;; Empirical variance: 17.409871989999985
;;; Empirical std: 4.172513869359811
;;; Empirical skewness: 0.1608908195413759
;;; Empirical kurtosis: 2.9468635202295315
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 1438.919591 ms
;;; Empirical mean: 20.156699999999997
;;; Empirical variance: 16.737545109999978
;;; Empirical std: 4.091154495982764
;;; Empirical skewness: 0.14500480852354064
;;; Empirical kurtosis: 2.9825705203746122
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 1440.479241 ms
;;; Empirical mean: 20.328199999999992
;;; Empirical variance: 16.69288475999998
;;; Empirical std: 4.085692690352516
;;; Empirical skewness: 0.197686784378718
;;; Empirical kurtosis: 3.008291774134386
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 1424.127824 ms
;;; Empirical mean: 20.2207
;;; Empirical variance: 16.95019150999999
;;; Empirical std: 4.117061028209321
;;; Empirical skewness: 0.25637365033562454
;;; Empirical kurtosis: 3.2919353168384693
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 1467.321704 ms
;;; Empirical mean: 20.1783
;;; Empirical variance: 16.529109109999993
;;; Empirical std: 4.065600707152633
;;; Empirical skewness: 0.1453556709432332
;;; Empirical kurtosis: 3.016824145995487
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 1572.829117 ms
;;; Empirical mean: 20.22749999999999
;;; Empirical variance: 16.49774374999998
;;; Empirical std: 4.061741467646603
;;; Empirical skewness: 0.20751984174821528
;;; Empirical kurtosis: 3.212300902678972
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 1555.510946 ms
;;; Empirical mean: 20.236399999999996
;;; Empirical variance: 16.60871503999998
;;; Empirical std: 4.075379128375663
;;; Empirical skewness: 0.1645533005107788
;;; Empirical kurtosis: 3.062600787690433
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 1508.580243 ms
;;; Empirical mean: 20.251300000000004
;;; Empirical variance: 17.786148309999984
;;; Empirical std: 4.217362719757453
;;; Empirical skewness: 0.1986525245706075
;;; Empirical kurtosis: 3.190940305634858
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 1536.513538 ms
;;; Empirical mean: 20.135999999999992
;;; Empirical variance: 16.67470399999998
;;; Empirical std: 4.083467154269761
;;; Empirical skewness: 0.2488067926626555
;;; Empirical kurtosis: 3.2968784092506644
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 1439.880396 ms
;;; Empirical mean: 20.405799999999992
;;; Empirical variance: 17.18352635999998
;;; Empirical std: 4.145301721226089
;;; Empirical skewness: 0.19006849809182189
;;; Empirical kurtosis: 3.0090526398864603
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 1628.742873 ms
;;; Empirical mean: 20.323999999999998
;;; Empirical variance: 17.13022399999998
;;; Empirical std: 4.138867477946109
;;; Empirical skewness: 0.1502574198553603
;;; Empirical kurtosis: 3.0200471335793986
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 1532.480398 ms
;;; Empirical mean: 20.236100000000004
;;; Empirical variance: 17.003756789999983
;;; Empirical std: 4.123561178156568
;;; Empirical skewness: 0.0896339677229628
;;; Empirical kurtosis: 3.142595617839986
;;; 
;;; Method :pgibbs with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 1510.366041 ms
;;; Empirical mean: 20.1649
;;; Empirical variance: 15.765307989999988
;;; Empirical std: 3.970555123657143
;;; Empirical skewness: 0.17379609137083296
;;; Empirical kurtosis: 3.2408986667422677
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 0
;;; Elapsed time: 719.732729 ms
;;; Empirical mean: 20.54993383517293
;;; Empirical variance: 15.589432568761167
;;; Empirical std: 3.948345548297561
;;; Empirical skewness: 0.07446512941746448
;;; Empirical kurtosis: 2.9921838679611157
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 1
;;; Elapsed time: 717.517 ms
;;; Empirical mean: 20.46683142764373
;;; Empirical variance: 15.762165910755973
;;; Empirical std: 3.9701594314027204
;;; Empirical skewness: 0.14398431174532403
;;; Empirical kurtosis: 3.0635362623073403
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 2
;;; Elapsed time: 670.923544 ms
;;; Empirical mean: 20.621425940581506
;;; Empirical variance: 17.33093313630219
;;; Empirical std: 4.163043734613196
;;; Empirical skewness: 0.1299626472838958
;;; Empirical kurtosis: 3.043775249832292
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 3
;;; Elapsed time: 702.860094 ms
;;; Empirical mean: 20.564112894537413
;;; Empirical variance: 16.87342466183284
;;; Empirical std: 4.1077274327580255
;;; Empirical skewness: 0.21499537300731994
;;; Empirical kurtosis: 3.135333885772539
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 4
;;; Elapsed time: 711.773657 ms
;;; Empirical mean: 20.553998375290508
;;; Empirical variance: 15.48530927888987
;;; Empirical std: 3.9351377712717848
;;; Empirical skewness: 0.07684094282524273
;;; Empirical kurtosis: 3.0709919603798883
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 5
;;; Elapsed time: 647.408691 ms
;;; Empirical mean: 20.236238084873204
;;; Empirical variance: 15.1463944807807
;;; Empirical std: 3.891836903157775
;;; Empirical skewness: 0.126169314113673
;;; Empirical kurtosis: 3.0489289529810626
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 6
;;; Elapsed time: 672.612472 ms
;;; Empirical mean: 20.475008762607953
;;; Empirical variance: 16.452091075259165
;;; Empirical std: 4.056117734393217
;;; Empirical skewness: 0.11338019013884368
;;; Empirical kurtosis: 2.8842471967503367
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 7
;;; Elapsed time: 682.579738 ms
;;; Empirical mean: 20.562912088300653
;;; Empirical variance: 16.250499959916638
;;; Empirical std: 4.031190886067868
;;; Empirical skewness: 0.18634925776604772
;;; Empirical kurtosis: 2.926781771745335
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 8
;;; Elapsed time: 673.003381 ms
;;; Empirical mean: 20.565521314064455
;;; Empirical variance: 15.880845789721537
;;; Empirical std: 3.9850778900445016
;;; Empirical skewness: 0.14323178492756236
;;; Empirical kurtosis: 3.0244768553032424
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 9
;;; Elapsed time: 724.36845 ms
;;; Empirical mean: 20.450333615976607
;;; Empirical variance: 15.285938107022227
;;; Empirical std: 3.909723533323325
;;; Empirical skewness: 0.09043014161100456
;;; Empirical kurtosis: 3.0524253542335544
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 10
;;; Elapsed time: 701.648118 ms
;;; Empirical mean: 20.498187571094515
;;; Empirical variance: 15.694652294912826
;;; Empirical std: 3.961647674252826
;;; Empirical skewness: 0.15651353216220698
;;; Empirical kurtosis: 3.0065701878937485
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 11
;;; Elapsed time: 654.819796 ms
;;; Empirical mean: 20.306148855024475
;;; Empirical variance: 15.737710941141394
;;; Empirical std: 3.967078388580366
;;; Empirical skewness: 0.1648636245427464
;;; Empirical kurtosis: 3.0589033871058553
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 12
;;; Elapsed time: 652.190912 ms
;;; Empirical mean: 20.581362706472635
;;; Empirical variance: 15.343381765264438
;;; Empirical std: 3.9170629003456705
;;; Empirical skewness: 0.09154834598268215
;;; Empirical kurtosis: 3.040629193377683
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 13
;;; Elapsed time: 657.891551 ms
;;; Empirical mean: 20.50996669194541
;;; Empirical variance: 17.111470843688846
;;; Empirical std: 4.1366013638842265
;;; Empirical skewness: 0.22566504163855014
;;; Empirical kurtosis: 3.041454215829693
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 14
;;; Elapsed time: 686.653951 ms
;;; Empirical mean: 20.459246769531436
;;; Empirical variance: 16.23337155078867
;;; Empirical std: 4.029065840959746
;;; Empirical skewness: 0.25757216449640413
;;; Empirical kurtosis: 3.2119979691898055
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 15
;;; Elapsed time: 688.763009 ms
;;; Empirical mean: 20.353891257000747
;;; Empirical variance: 15.430637938453568
;;; Empirical std: 3.9281850692722675
;;; Empirical skewness: 0.16956723136066618
;;; Empirical kurtosis: 3.1254799356764384
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 16
;;; Elapsed time: 678.359567 ms
;;; Empirical mean: 20.514809377091595
;;; Empirical variance: 15.92099986886332
;;; Empirical std: 3.9901127639282725
;;; Empirical skewness: 0.15576791048711489
;;; Empirical kurtosis: 2.958958296532856
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 17
;;; Elapsed time: 684.071974 ms
;;; Empirical mean: 20.33738816485327
;;; Empirical variance: 15.354884005685786
;;; Empirical std: 3.918530847867066
;;; Empirical skewness: 0.0927732686002241
;;; Empirical kurtosis: 3.104276590059627
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 18
;;; Elapsed time: 728.172546 ms
;;; Empirical mean: 20.401542403437965
;;; Empirical variance: 16.355224333748378
;;; Empirical std: 4.044159286396664
;;; Empirical skewness: 0.23217522536882185
;;; Empirical kurtosis: 3.0075299178729233
;;; 
;;; Method :ipmcmc with 10000 samples and options []
;;; Chain no. 19
;;; Elapsed time: 694.897448 ms
;;; Empirical mean: 20.615967491819774
;;; Empirical variance: 17.640609387836005
;;; Empirical std: 4.2000725455444226
;;; Empirical skewness: 0.07157947164438502
;;; Empirical kurtosis: 2.834801530777913
;;; 
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[nil,#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0x4369e02b]</span>","value":"#multifn[print-method 0x4369e02b]"}],"value":"[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0x4369e02b]],#multifn[print-method 0x4369e02b]],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
