;; gorilla-repl.fileformat = 1

;; **
;;; # Gorilla REPL
;;;
;;; Welcome to gorilla :-)
;;;
;;; Shift + enter evaluates code. Hit alt+g twice in quick succession or click the menu icon (upper-right corner) for more commands ...
;;;
;;; It's a good habit to run each worksheet in its own namespace: feel free to use the declaration we've provided below if you'd like.
;; **

;; @@

(ns model
  (:require [gorilla-plot.core :as plot])
  (:use [anglican core emit runtime stat
          [state :only [get-predicts get-log-weight get-result]]]))

(defdist geometric
"Geometric distribution on support {0,1,2....}"
[p] []
(sample* [this]
        (loop [value 0]
            (if (sample* (flip p))
            value
            (recur (inc value)))))
(observe* [this value] (+ (log p) (* value (log (- 1 p))))))

(defdist dirac [x]
    (sample* [this] x)
    (observe* [this value]
              (if (= value x)
                0
                (- (/ 1.0 0.0)))))


(def data [4 5 4 0 1 4 3 4 0 6 3 3 4 0 2 6 3 3 5 4 5 3 1 4 4 1 5 5 3 4 2 5 2 2 3 4 2 1 3 -1 2 1 1 1 1 3 0 0 1 0 1 1 0 0 3 1 0 3 2 2 0 1 1 1 0 1 0 1 0 0 0 2 1 0 0 0 1 1 0 2 3 3 1 -1 2 1 1 1 1 2 4 2 0 0 1 4 0 0 0 1 0 0 0 0 0 1 0 0 1 0 1]
  )

(with-primitive-procedures [dirac geometric]
  (defquery model [method- options- ]
    (let [
      switchpoint (sample (uniform-discrete 0 (count data)))
      rate1 (sample (exponential 1))
      rate2 (sample (exponential 1))]
      (doall (map
        (fn [i]
          (let [x (nth data i)
                rate (if (< i switchpoint) rate1 rate2)
                dist (if (zero? rate) (dirac 0) (poisson rate))]
            (if (<= 0 (nth data i)) (observe dist (nth data i)))))
        (range 0 (count data))))
      switchpoint
    )
  )
)


(def model_name "cont_switchpoint")
(def outfile "output/cont_switchpoint_anglican.json")

; (def configurations [:rmh []])
(def configurations
  [
    [:importance []]
    [:lmh []]
    [:rmh []]
    [:rmh [:alpha 0.8 :sigma 2]]
    [:smc []]
    [:smc [:number-of-particles 1000]]
    [:pgibbs []]
    [:pgibbs [:number-of-particles 1000]]
    [:ipmcmc []]
    [:ipmcmc [:number-of-particles 1000 :number-of-nodes 32]]
  ])

; (def num_samples_options [1000])
(def num_samples_options [1000 10000])
(def thinning 1)

(spit outfile "[\n" :append false)

(def num-chains 20)

(doall
  (for [ num_samples num_samples_options
         [method options] configurations
         chain (range 0 num-chains)]
    (do
      (println (format "\nMethod %s with options %s" method options))
      (println (format "Chain no. %s" chain))
      (let [start (. System (nanoTime))
            warmup (/ num_samples 5)
            samples (take-nth thinning (take (* num_samples thinning) (drop warmup (apply doquery method model [method options] options))))
            results (collect-results samples)
            values (map (fn [s] (get-result s)) samples)
            max-value (apply max values)
            mean (empirical-mean results)
            variance (empirical-variance results)
            std (empirical-std results)
            skewness (if (zero? std) (/ 0.0 0.0) (empirical-skew results))
            kurtosis (if (zero? std) (/ 0.0 0.0) (empirical-kurtosis results))
            distribution (empirical-distribution (collect-results samples))
            masses (for [n (range 0 (inc max-value))] (get distribution n 0.0))
            end (. System (nanoTime))
            elapsed_ms (/ (- end start) 1e6)]
        (println (format "Elapsed time: %s ms" elapsed_ms))
        (println (format "Empirical mean: %s" mean))
        (println (format "Empirical variance: %s" variance))
        (println (format "Empirical std: %s" std))
        (println (format "Empirical skewness: %s" skewness))
        (println (format "Empirical kurtosis: %s" kurtosis))
        (spit outfile (format
                   "{\"model\": \"%s\", \"system\": \"anglican\", \"method\": \"%s\", \"options\": \"%s\", \"num_samples\": %s, \"time_ms\": %s, \"total\": 1.0, \"mean\": %s, \"variance\": %s, \"stddev\": %s, \"skewness\": %s, \"kurtosis\": %s, \"masses\": [%s] },\n"
                   model_name method options num_samples elapsed_ms mean variance std skewness kurtosis
                   (clojure.string/join ", " masses)) :append true)
        (if false (do
          (println "Empirical distribution:")
          (doall (for [n (range 0 (inc max-value))]
            (println (format "p(%s) = %s" n (get distribution n 0.0)))))))
        ; (println "List of samples (format: sample log-weight):")
        ; (doall (map (fn [s] (println (format "%s %s" (get-result s) (get-log-weight s)))) samples))
        ; values need to be adjusted if they are weighted!
        ; (plot/histogram values :normalize :probability)
      )
    )
  )
)

(spit outfile "]\n" :append true)
;; @@
;; ->
;;;
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 950.145448 ms
;;; Empirical mean: 42.2286356350629
;;; Empirical variance: 14.020441116616718
;;; Empirical std: 3.7443879495341714
;;; Empirical skewness: -0.14772826945149115
;;; Empirical kurtosis: 1.1649873417780836
;;;
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 965.317395 ms
;;; Empirical mean: 34.6988301079285
;;; Empirical variance: 13.415656270634607
;;; Empirical std: 3.6627389028750885
;;; Empirical skewness: 1.8923858920196108
;;; Empirical kurtosis: 5.6098794350964996
;;;
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 875.134135 ms
;;; Empirical mean: 38.381119898715575
;;; Empirical variance: 2.9572455032183513
;;; Empirical std: 1.7196643577216897
;;; Empirical skewness: 8.657360022179674
;;; Empirical kurtosis: 115.39592531385121
;;;
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 1129.50557 ms
;;; Empirical mean: 39.02355787774085
;;; Empirical variance: 0.2679107942850605
;;; Empirical std: 0.517600999115207
;;; Empirical skewness: 12.433385083152501
;;; Empirical kurtosis: 230.7319494739267
;;;
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 900.734464 ms
;;; Empirical mean: 41.24932961789924
;;; Empirical variance: 4.27518955805702
;;; Empirical std: 2.0676531522615247
;;; Empirical skewness: 0.45980878376218975
;;; Empirical kurtosis: 1.984943728881753
;;;
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 863.491641 ms
;;; Empirical mean: 40.01156952579471
;;; Empirical variance: 4.121371214524597
;;; Empirical std: 2.0301160593731082
;;; Empirical skewness: -1.6300892570399177
;;; Empirical kurtosis: 4.5044995189540185
;;;
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 932.927088 ms
;;; Empirical mean: 41.5643569985239
;;; Empirical variance: 0.3617293536415726
;;; Empirical std: 0.6014394014708153
;;; Empirical skewness: 2.029920052278035
;;; Empirical kurtosis: 75.2768033039401
;;;
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 911.186686 ms
;;; Empirical mean: 40.98560295224869
;;; Empirical variance: 0.21298972988847592
;;; Empirical std: 0.46150810381668916
;;; Empirical skewness: -9.202455054877635
;;; Empirical kurtosis: 513.8723535182405
;;;
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 882.408857 ms
;;; Empirical mean: 38.997986613919345
;;; Empirical variance: 0.16817241401511465
;;; Empirical std: 0.41008830026607035
;;; Empirical skewness: -0.6570871589168696
;;; Empirical kurtosis: 336.51926449784867
;;;
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 876.538817 ms
;;; Empirical mean: 39.6994406167813
;;; Empirical variance: 8.94805679263968
;;; Empirical std: 2.9913302714076355
;;; Empirical skewness: 0.21512174461506406
;;; Empirical kurtosis: 1.0959762342415222
;;;
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 926.885585 ms
;;; Empirical mean: 37.152100307009185
;;; Empirical variance: 1.4588032323012408
;;; Empirical std: 1.2078092698357803
;;; Empirical skewness: 8.296795588047289
;;; Empirical kurtosis: 77.11825714210032
;;;
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 886.933183 ms
;;; Empirical mean: 42.14852961112084
;;; Empirical variance: 0.7565087840375487
;;; Empirical std: 0.8697751341798343
;;; Empirical skewness: 2.4079685910025703
;;; Empirical kurtosis: 32.370147080436695
;;;
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 992.494562 ms
;;; Empirical mean: 43.74605749874436
;;; Empirical variance: 7.725061362409773
;;; Empirical std: 2.7793994607486296
;;; Empirical skewness: -0.3049777255991502
;;; Empirical kurtosis: 3.2230641031305476
;;;
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 1053.05827 ms
;;; Empirical mean: 37.9766992047493
;;; Empirical variance: 11.470443777971123
;;; Empirical std: 3.386804360746443
;;; Empirical skewness: 1.100024122278425
;;; Empirical kurtosis: 3.4830471271937906
;;;
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 986.140586 ms
;;; Empirical mean: 46.60760102284728
;;; Empirical variance: 2.388874897532292
;;; Empirical std: 1.5455985563956418
;;; Empirical skewness: -3.5831409872580884
;;; Empirical kurtosis: 17.283804215455568
;;;
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 1009.364686 ms
;;; Empirical mean: 36.45158428003864
;;; Empirical variance: 9.221727237383249
;;; Empirical std: 3.036729694487682
;;; Empirical skewness: 2.86791534441071
;;; Empirical kurtosis: 16.120706279396504
;;;
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 930.302399 ms
;;; Empirical mean: 38.97081451482904
;;; Empirical variance: 0.23109662230845116
;;; Empirical std: 0.4807251005600302
;;; Empirical skewness: -10.27252281657685
;;; Empirical kurtosis: 317.0054021678811
;;;
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 987.218904 ms
;;; Empirical mean: 36.970680700084394
;;; Empirical variance: 1.2590119179143084
;;; Empirical std: 1.1220570029701291
;;; Empirical skewness: 2.0962537896091202
;;; Empirical kurtosis: 32.58924324691055
;;;
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 1125.107192 ms
;;; Empirical mean: 42.19317283598885
;;; Empirical variance: 2.351645156949713
;;; Empirical std: 1.533507468827496
;;; Empirical skewness: 0.38998560342337396
;;; Empirical kurtosis: 3.0238739112903366
;;;
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 950.353481 ms
;;; Empirical mean: 39.11049219229492
;;; Empirical variance: 0.7038128216879868
;;; Empirical std: 0.8389355289222091
;;; Empirical skewness: -1.5947053136721634
;;; Empirical kurtosis: 38.8449061430484
;;;
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 1006.37153 ms
;;; Empirical mean: 38.159000000000006
;;; Empirical variance: 7.061719
;;; Empirical std: 2.6573895085214736
;;; Empirical skewness: 0.35823479812985604
;;; Empirical kurtosis: 2.233816074381699
;;;
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 961.360689 ms
;;; Empirical mean: 41.068999999999996
;;; Empirical variance: 9.866238999999998
;;; Empirical std: 3.141056987703343
;;; Empirical skewness: 0.3690288935826693
;;; Empirical kurtosis: 1.8924146953244334
;;;
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 930.047461 ms
;;; Empirical mean: 40.553000000000004
;;; Empirical variance: 3.523191000000001
;;; Empirical std: 1.8770165156439091
;;; Empirical skewness: -0.9056414374692745
;;; Empirical kurtosis: 3.128912446794035
;;;
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 909.112546 ms
;;; Empirical mean: 39.731
;;; Empirical variance: 3.792639000000002
;;; Empirical std: 1.947469897071583
;;; Empirical skewness: 0.7680864013461202
;;; Empirical kurtosis: 6.602383281908006
;;;
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 910.204659 ms
;;; Empirical mean: 41.17599999999999
;;; Empirical variance: 7.169024000000002
;;; Empirical std: 2.677503314657146
;;; Empirical skewness: -0.03890039703707648
;;; Empirical kurtosis: 2.841221485717671
;;;
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 945.706755 ms
;;; Empirical mean: 38.75499999999999
;;; Empirical variance: 3.6209750000000014
;;; Empirical std: 1.9028859661051687
;;; Empirical skewness: 0.22308494402385776
;;; Empirical kurtosis: 2.738648730449992
;;;
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 996.369808 ms
;;; Empirical mean: 40.442
;;; Empirical variance: 1.612636000000001
;;; Empirical std: 1.2698960587386674
;;; Empirical skewness: -0.706440881763637
;;; Empirical kurtosis: 3.604292044124834
;;;
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 896.094492 ms
;;; Empirical mean: 40.11
;;; Empirical variance: 6.205900000000002
;;; Empirical std: 2.491164386386415
;;; Empirical skewness: 0.40548225281228173
;;; Empirical kurtosis: 2.9839401310888043
;;;
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 989.570671 ms
;;; Empirical mean: 39.373999999999995
;;; Empirical variance: 11.006124000000002
;;; Empirical std: 3.3175478896317383
;;; Empirical skewness: -0.16835447459222705
;;; Empirical kurtosis: 2.4838919383674916
;;;
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 982.229152 ms
;;; Empirical mean: 39.571
;;; Empirical variance: 4.858959000000002
;;; Empirical std: 2.204304652265653
;;; Empirical skewness: -0.03142357694453965
;;; Empirical kurtosis: 2.7203516695345735
;;;
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 899.610988 ms
;;; Empirical mean: 39.54999999999999
;;; Empirical variance: 3.391500000000003
;;; Empirical std: 1.8416025629869228
;;; Empirical skewness: -0.23509416788162557
;;; Empirical kurtosis: 3.8856054072272475
;;;
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 897.868915 ms
;;; Empirical mean: 40.348
;;; Empirical variance: 0.8728959999999997
;;; Empirical std: 0.934289034506988
;;; Empirical skewness: -0.9778188586263912
;;; Empirical kurtosis: 2.6656874845546064
;;;
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 894.02463 ms
;;; Empirical mean: 41.145999999999994
;;; Empirical variance: 4.704684000000002
;;; Empirical std: 2.169028353894896
;;; Empirical skewness: 1.2014429464725025
;;; Empirical kurtosis: 4.089981322101016
;;;
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 895.038406 ms
;;; Empirical mean: 40.67700000000001
;;; Empirical variance: 6.196671000000007
;;; Empirical std: 2.4893113505546083
;;; Empirical skewness: 0.6801277704565034
;;; Empirical kurtosis: 3.155882495996566
;;;
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 991.584155 ms
;;; Empirical mean: 45.85900000000001
;;; Empirical variance: 309.0391190000005
;;; Empirical std: 17.579508497111075
;;; Empirical skewness: 2.5698816249520373
;;; Empirical kurtosis: 7.713458069403468
;;;
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 1140.878501 ms
;;; Empirical mean: 39.961
;;; Empirical variance: 2.3074790000000025
;;; Empirical std: 1.5190388408464093
;;; Empirical skewness: -0.39284069139380484
;;; Empirical kurtosis: 4.184675211127616
;;;
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 1021.568067 ms
;;; Empirical mean: 40.031000000000006
;;; Empirical variance: 7.122039000000003
;;; Empirical std: 2.6687148592534204
;;; Empirical skewness: 0.5517330929592994
;;; Empirical kurtosis: 2.964499132506496
;;;
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 929.741701 ms
;;; Empirical mean: 40.224000000000004
;;; Empirical variance: 2.339824000000002
;;; Empirical std: 1.529648325596443
;;; Empirical skewness: -1.9858583262905336
;;; Empirical kurtosis: 11.338449457575702
;;;
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 946.246289 ms
;;; Empirical mean: 40.437000000000005
;;; Empirical variance: 2.0400310000000004
;;; Empirical std: 1.4282965378379942
;;; Empirical skewness: -1.247502402063356
;;; Empirical kurtosis: 4.961138040083164
;;;
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 906.02154 ms
;;; Empirical mean: 60.35299999999998
;;; Empirical variance: 680.5063909999997
;;; Empirical std: 26.08651741800733
;;; Empirical skewness: 0.6188935816615992
;;; Empirical kurtosis: 1.4152856304744958
;;;
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 1091.118084 ms
;;; Empirical mean: 37.511
;;; Empirical variance: 6.099878999999997
;;; Empirical std: 2.4697933111902293
;;; Empirical skewness: 1.1475570747752968
;;; Empirical kurtosis: 2.451115233982759
;;;
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 1011.272878 ms
;;; Empirical mean: 38.540000000000006
;;; Empirical variance: 8.470400000000003
;;; Empirical std: 2.910395162173
;;; Empirical skewness: 0.9530377549851949
;;; Empirical kurtosis: 2.565003821302851
;;;
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 1122.981829 ms
;;; Empirical mean: 38.894999999999996
;;; Empirical variance: 1.7759750000000019
;;; Empirical std: 1.3326571201925879
;;; Empirical skewness: -1.3513843853507956
;;; Empirical kurtosis: 5.289031132309627
;;;
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 1066.029777 ms
;;; Empirical mean: 39.728
;;; Empirical variance: 2.6120160000000006
;;; Empirical std: 1.6161732580388777
;;; Empirical skewness: 0.513453487556649
;;; Empirical kurtosis: 3.4390435294132566
;;;
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 1216.336207 ms
;;; Empirical mean: 39.73300000000001
;;; Empirical variance: 2.225711000000003
;;; Empirical std: 1.4918816977227125
;;; Empirical skewness: 1.868041961802931
;;; Empirical kurtosis: 12.842799881721168
;;;
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 1005.428517 ms
;;; Empirical mean: 41.251999999999995
;;; Empirical variance: 1.3864960000000013
;;; Empirical std: 1.1774956475503429
;;; Empirical skewness: 0.8733383098734275
;;; Empirical kurtosis: 8.425254643680635
;;;
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 970.873469 ms
;;; Empirical mean: 41.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;;
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 973.817762 ms
;;; Empirical mean: 63.939999999999976
;;; Empirical variance: 781.9323999999999
;;; Empirical std: 27.963054196564435
;;; Empirical skewness: 0.26397267235480065
;;; Empirical kurtosis: 1.0836041556395337
;;;
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 968.317716 ms
;;; Empirical mean: 40.82
;;; Empirical variance: 0.6876000000000002
;;; Empirical std: 0.8292164976651153
;;; Empirical skewness: -4.389685938774028
;;; Empirical kurtosis: 20.26934264107041
;;;
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 963.1653 ms
;;; Empirical mean: 40.049
;;; Empirical variance: 1.1225990000000006
;;; Empirical std: 1.0595277249793893
;;; Empirical skewness: -0.9854677125871788
;;; Empirical kurtosis: 3.5759875313634084
;;;
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 1039.665407 ms
;;; Empirical mean: 39.440000000000005
;;; Empirical variance: 4.544400000000001
;;; Empirical std: 2.1317598363793238
;;; Empirical skewness: 0.008898826572121729
;;; Empirical kurtosis: 3.033280946423828
;;;
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 957.448504 ms
;;; Empirical mean: 40.392
;;; Empirical variance: 1.7423360000000008
;;; Empirical std: 1.3199757573531419
;;; Empirical skewness: -1.9154467136841367
;;; Empirical kurtosis: 5.019335716035607
;;;
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 955.665884 ms
;;; Empirical mean: 40.449999999999996
;;; Empirical variance: 2.239499999999999
;;; Empirical std: 1.4964959071110082
;;; Empirical skewness: 0.06789694591991749
;;; Empirical kurtosis: 1.0077862801580417
;;;
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 1067.900053 ms
;;; Empirical mean: 39.354
;;; Empirical variance: 4.428684000000003
;;; Empirical std: 2.1044438695294305
;;; Empirical skewness: 0.12150062504776375
;;; Empirical kurtosis: 2.005631814509732
;;;
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 974.36559 ms
;;; Empirical mean: 40.213
;;; Empirical variance: 3.3056310000000027
;;; Empirical std: 1.8181394335968852
;;; Empirical skewness: 1.1688724722092219
;;; Empirical kurtosis: 6.736565171849237
;;;
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 1009.925842 ms
;;; Empirical mean: 40.446999999999996
;;; Empirical variance: 1.4431910000000008
;;; Empirical std: 1.201328847568392
;;; Empirical skewness: -0.7611455238711875
;;; Empirical kurtosis: 2.8290788697798748
;;;
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 943.534161 ms
;;; Empirical mean: 40.416000000000004
;;; Empirical variance: 22.742943999999998
;;; Empirical std: 4.768956279942184
;;; Empirical skewness: 0.17408073212964378
;;; Empirical kurtosis: 1.1117699501136467
;;;
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 954.654903 ms
;;; Empirical mean: 38.068000000000005
;;; Empirical variance: 5.277376
;;; Empirical std: 2.2972540129467616
;;; Empirical skewness: 0.5289821457488202
;;; Empirical kurtosis: 1.6293739053707657
;;;
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 1001.773302 ms
;;; Empirical mean: 38.977999999999994
;;; Empirical variance: 2.0375159999999997
;;; Empirical std: 1.4274158469065696
;;; Empirical skewness: 0.17070886734012278
;;; Empirical kurtosis: 1.8643446647705457
;;;
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 1009.337057 ms
;;; Empirical mean: 37.385999999999996
;;; Empirical variance: 5.795004000000001
;;; Empirical std: 2.4072814542549863
;;; Empirical skewness: 1.9252671661812693
;;; Empirical kurtosis: 6.534220226248964
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 997.543148 ms
;;; Empirical mean: 39.79199999999999
;;; Empirical variance: 5.764736000000002
;;; Empirical std: 2.400986463935189
;;; Empirical skewness: 0.6048812986263301
;;; Empirical kurtosis: 2.7106063290021645
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 1000.191074 ms
;;; Empirical mean: 41.004
;;; Empirical variance: 0.003984000000000004
;;; Empirical std: 0.06311893535223803
;;; Empirical skewness: 15.716361413007157
;;; Empirical kurtosis: 248.00401606425908
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 1057.059981 ms
;;; Empirical mean: 38.925999999999995
;;; Empirical variance: 5.080523999999999
;;; Empirical std: 2.2540017746221936
;;; Empirical skewness: 0.029191045436834834
;;; Empirical kurtosis: 1.2683380432454228
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 1026.680651 ms
;;; Empirical mean: 39.172
;;; Empirical variance: 7.024416000000002
;;; Empirical std: 2.650361484779011
;;; Empirical skewness: 0.6615541199789033
;;; Empirical kurtosis: 3.377228743552784
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 959.105507 ms
;;; Empirical mean: 40.39
;;; Empirical variance: 0.6359000000000008
;;; Empirical std: 0.7974333827975857
;;; Empirical skewness: -2.0101485834676542
;;; Empirical kurtosis: 10.294279546436826
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 970.24743 ms
;;; Empirical mean: 38.841
;;; Empirical variance: 3.435719000000002
;;; Empirical std: 1.8535692595638291
;;; Empirical skewness: 1.212851991945353
;;; Empirical kurtosis: 5.636280380224408
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 965.835817 ms
;;; Empirical mean: 40.59599999999999
;;; Empirical variance: 2.518784000000001
;;; Empirical std: 1.5870677364246306
;;; Empirical skewness: 0.45546321629434344
;;; Empirical kurtosis: 4.123244089479249
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 982.033599 ms
;;; Empirical mean: 38.52400000000001
;;; Empirical variance: 6.617424000000003
;;; Empirical std: 2.572435421929966
;;; Empirical skewness: 1.147133352823923
;;; Empirical kurtosis: 3.830916556782818
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 985.648691 ms
;;; Empirical mean: 45.807
;;; Empirical variance: 314.55575100000016
;;; Empirical std: 17.735719635808415
;;; Empirical skewness: 2.534334256824509
;;; Empirical kurtosis: 7.441500382799026
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 1112.442407 ms
;;; Empirical mean: 38.02299999999999
;;; Empirical variance: 4.956471000000002
;;; Empirical std: 2.2263133202673884
;;; Empirical skewness: 0.817695647423443
;;; Empirical kurtosis: 2.141246344895418
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 1077.163206 ms
;;; Empirical mean: 40.897000000000006
;;; Empirical variance: 0.18839099999999998
;;; Empirical std: 0.43404032070765036
;;; Empirical skewness: -4.056467333432437
;;; Empirical kurtosis: 17.644882180599588
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 1001.122946 ms
;;; Empirical mean: 40.564
;;; Empirical variance: 0.247904
;;; Empirical std: 0.49789958827056685
;;; Empirical skewness: -0.2824219799399459
;;; Empirical kurtosis: 1.144141669766812
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 1030.458594 ms
;;; Empirical mean: 40.918
;;; Empirical variance: 5.875276000000002
;;; Empirical std: 2.423896862492297
;;; Empirical skewness: 0.8511913198228593
;;; Empirical kurtosis: 2.9537728626114097
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 1025.203804 ms
;;; Empirical mean: 38.801
;;; Empirical variance: 8.147399000000007
;;; Empirical std: 2.854364903091405
;;; Empirical skewness: 1.2246863923174947
;;; Empirical kurtosis: 3.9514296943193923
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 1015.278399 ms
;;; Empirical mean: 39.409000000000006
;;; Empirical variance: 3.0617190000000005
;;; Empirical std: 1.7497768429145473
;;; Empirical skewness: 0.0455176525112766
;;; Empirical kurtosis: 2.9384155301465507
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 969.464592 ms
;;; Empirical mean: 39.66899999999999
;;; Empirical variance: 3.3114390000000014
;;; Empirical std: 1.8197359698593643
;;; Empirical skewness: -0.38640852401818465
;;; Empirical kurtosis: 2.651308760711026
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 1040.490356 ms
;;; Empirical mean: 39.757000000000005
;;; Empirical variance: 5.339951000000003
;;; Empirical std: 2.3108333994470485
;;; Empirical skewness: -0.5053451910880681
;;; Empirical kurtosis: 2.4538028682904294
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 1566.581483 ms
;;; Empirical mean: 40.28
;;; Empirical variance: 2.629600000000003
;;; Empirical std: 1.6216041440499598
;;; Empirical skewness: -2.748205463942261
;;; Empirical kurtosis: 12.275989278041127
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 1088.431851 ms
;;; Empirical mean: 39.956999999999994
;;; Empirical variance: 2.5251510000000015
;;; Empirical std: 1.5890723709132952
;;; Empirical skewness: -1.6341270219378312
;;; Empirical kurtosis: 4.505866049321073
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 1191.545859 ms
;;; Empirical mean: 40.49900000000001
;;; Empirical variance: 2.213999000000004
;;; Empirical std: 1.487951276084
;;; Empirical skewness: 0.5519775074280505
;;; Empirical kurtosis: 16.198926273722154
;;;
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 1977.183946 ms
;;; Empirical mean: 39.91773147717533
;;; Empirical variance: 0.6887161002875002
;;; Empirical std: 0.8298892096463841
;;; Empirical skewness: -6.5626307734556395
;;; Empirical kurtosis: 90.79066064273104
;;;
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 1873.410424 ms
;;; Empirical mean: 36.03931028122706
;;; Empirical variance: 0.36583563175421796
;;; Empirical std: 0.6048434770700747
;;; Empirical skewness: 16.1030781287423
;;; Empirical kurtosis: 265.3727087769062
;;;
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 1875.135283 ms
;;; Empirical mean: 45.798163582001784
;;; Empirical variance: 2.242893201704312
;;; Empirical std: 1.4976291936605375
;;; Empirical skewness: -8.160035256501004
;;; Empirical kurtosis: 70.19091860059851
;;;
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 1872.674659 ms
;;; Empirical mean: 41.04450145516061
;;; Empirical variance: 0.3674546230507112
;;; Empirical std: 0.606180355216755
;;; Empirical skewness: 5.296922072935727
;;; Empirical kurtosis: 147.49144058320852
;;;
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 1891.226644 ms
;;; Empirical mean: 46.87930656079159
;;; Empirical variance: 0.7269834554261716
;;; Empirical std: 0.8526332479009786
;;; Empirical skewness: -0.6797042909662036
;;; Empirical kurtosis: 53.78704890253616
;;;
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 1889.467522 ms
;;; Empirical mean: 41.12099572697886
;;; Empirical variance: 9.824641837451702
;;; Empirical std: 3.134428470622946
;;; Empirical skewness: -0.31464305886884003
;;; Empirical kurtosis: 3.2812860501187564
;;;
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 2187.44415 ms
;;; Empirical mean: 36.13980006763101
;;; Empirical variance: 0.9655581175881193
;;; Empirical std: 0.9826281685297441
;;; Empirical skewness: 7.404407413885154
;;; Empirical kurtosis: 60.07253641013635
;;;
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 1905.426888 ms
;;; Empirical mean: 38.451709143941244
;;; Empirical variance: 6.4923395010286775
;;; Empirical std: 2.548006966440374
;;; Empirical skewness: 0.7698836087091182
;;; Empirical kurtosis: 2.263819772684243
;;;
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 1849.625185 ms
;;; Empirical mean: 47.017800352286464
;;; Empirical variance: 14.528325390913325
;;; Empirical std: 3.8116040443510557
;;; Empirical skewness: -3.5777616289984198
;;; Empirical kurtosis: 15.833204197205655
;;;
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 1842.462886 ms
;;; Empirical mean: 41.987497774936124
;;; Empirical variance: 0.04061598899424008
;;; Empirical std: 0.20153408891361302
;;; Empirical skewness: 13.973226256311003
;;; Empirical kurtosis: 1856.057361790421
;;;
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 1938.237013 ms
;;; Empirical mean: 39.55123671731076
;;; Empirical variance: 3.237038589045713
;;; Empirical std: 1.79917719778951
;;; Empirical skewness: 3.4759926640775736
;;; Empirical kurtosis: 15.192571250013412
;;;
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 2010.976839 ms
;;; Empirical mean: 35.006072394553364
;;; Empirical variance: 0.12879690364794083
;;; Empirical std: 0.35888285504874823
;;; Empirical skewness: 27.4618574632962
;;; Empirical kurtosis: 1775.6123414731594
;;;
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 2353.029659 ms
;;; Empirical mean: 41.60436341587366
;;; Empirical variance: 0.754849235777306
;;; Empirical std: 0.8688206004563347
;;; Empirical skewness: -2.526589325986115
;;; Empirical kurtosis: 14.119951550702032
;;;
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 1870.609195 ms
;;; Empirical mean: 41.20745557710422
;;; Empirical variance: 1.0187826567921319
;;; Empirical std: 1.009347639216604
;;; Empirical skewness: -0.6813635190640185
;;; Empirical kurtosis: 3.9488014888848526
;;;
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 1897.997435 ms
;;; Empirical mean: 37.423281548407914
;;; Empirical variance: 2.9010084226284336
;;; Empirical std: 1.7032346939363447
;;; Empirical skewness: 3.834403340753626
;;; Empirical kurtosis: 22.44986523153695
;;;
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 1971.516115 ms
;;; Empirical mean: 36.57858371766672
;;; Empirical variance: 1.8434346531657906
;;; Empirical std: 1.3577314363178716
;;; Empirical skewness: 4.417818758449228
;;; Empirical kurtosis: 25.241193581994334
;;;
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 1962.143593 ms
;;; Empirical mean: 43.489392100064286
;;; Empirical variance: 3.634833795177487
;;; Empirical std: 1.9065240085499806
;;; Empirical skewness: -0.9336478236888416
;;; Empirical kurtosis: 3.2538305029744254
;;;
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 2090.427461 ms
;;; Empirical mean: 39.568150217333866
;;; Empirical variance: 11.08514616589092
;;; Empirical std: 3.3294363135358096
;;; Empirical skewness: -0.23362461822744868
;;; Empirical kurtosis: 1.7419584155438184
;;;
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 1924.135583 ms
;;; Empirical mean: 40.00916797899754
;;; Empirical variance: 0.08122397747354312
;;; Empirical std: 0.28499820608828946
;;; Empirical skewness: 18.840018626779187
;;; Empirical kurtosis: 605.6523539004158
;;;
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 1943.108023 ms
;;; Empirical mean: 40.23146167189926
;;; Empirical variance: 2.9411210316773744
;;; Empirical std: 1.7149696882678056
;;; Empirical skewness: 4.809374825833208
;;; Empirical kurtosis: 41.91239860248884
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 2736.72974 ms
;;; Empirical mean: 39.213420206838244
;;; Empirical variance: 2.530727543255347
;;; Empirical std: 1.5908260568821933
;;; Empirical skewness: 7.323318590085432
;;; Empirical kurtosis: 54.659538570477814
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 2451.159745 ms
;;; Empirical mean: 40.036470310435
;;; Empirical variance: 0.03514022689177886
;;; Empirical std: 0.18745726684174946
;;; Empirical skewness: 4.945443805668116
;;; Empirical kurtosis: 25.45741443502089
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 2410.373467 ms
;;; Empirical mean: 41.35069708355269
;;; Empirical variance: 8.29304506010345
;;; Empirical std: 2.879764757771622
;;; Empirical skewness: -1.1560208017798796
;;; Empirical kurtosis: 2.8962547062035453
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 2430.08654 ms
;;; Empirical mean: 37.24066370628408
;;; Empirical variance: 1.4133496301161403
;;; Empirical std: 1.1888438207418754
;;; Empirical skewness: 1.6343684205120705
;;; Empirical kurtosis: 11.2405441614269
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 2536.575609 ms
;;; Empirical mean: 38.15238695371436
;;; Empirical variance: 11.476273206174724
;;; Empirical std: 3.3876648603683814
;;; Empirical skewness: 0.8199259911991446
;;; Empirical kurtosis: 3.1576483718528134
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 2382.211437 ms
;;; Empirical mean: 44.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 2752.911621 ms
;;; Empirical mean: 42.34727466883416
;;; Empirical variance: 10.691116622401307
;;; Empirical std: 3.2697273009230154
;;; Empirical skewness: -0.6449297515852281
;;; Empirical kurtosis: 2.214942864926272
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 2393.986911 ms
;;; Empirical mean: 40.14686927669634
;;; Empirical variance: 1.1399687552709474
;;; Empirical std: 1.0676931934179161
;;; Empirical skewness: 7.2517873113309195
;;; Empirical kurtosis: 54.48093281260729
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 2442.229185 ms
;;; Empirical mean: 46.14781087870123
;;; Empirical variance: 16.94347579322905
;;; Empirical std: 4.116245351437284
;;; Empirical skewness: -1.7723972053451658
;;; Empirical kurtosis: 4.141391853515343
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 2593.945486 ms
;;; Empirical mean: 38.82992142108821
;;; Empirical variance: 0.48130901373071694
;;; Empirical std: 0.6937643791163661
;;; Empirical skewness: -3.833928235929617
;;; Empirical kurtosis: 15.699005718258398
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 2553.792071 ms
;;; Empirical mean: 36.08213738870356
;;; Empirical variance: 0.5801235995949258
;;; Empirical std: 0.7616584533732464
;;; Empirical skewness: 4.718834551143762
;;; Empirical kurtosis: 39.38326282616042
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 2410.585339 ms
;;; Empirical mean: 37.04001882506805
;;; Empirical variance: 7.23851144404852
;;; Empirical std: 2.690448186464203
;;; Empirical skewness: 2.2003628910779875
;;; Empirical kurtosis: 5.841596852433069
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 2517.294592 ms
;;; Empirical mean: 38.99761181749475
;;; Empirical variance: 7.718386782361028
;;; Empirical std: 2.778198477855934
;;; Empirical skewness: 1.6939836759903664
;;; Empirical kurtosis: 4.513732734535608
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 2423.723318 ms
;;; Empirical mean: 40.21483901127365
;;; Empirical variance: 8.263841260213644
;;; Empirical std: 2.874689767646875
;;; Empirical skewness: -0.5641890517614971
;;; Empirical kurtosis: 3.7262316101444206
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 2516.774725 ms
;;; Empirical mean: 40.8175996152031
;;; Empirical variance: 0.5139312540166294
;;; Empirical std: 0.7168899873876252
;;; Empirical skewness: -3.675876740877583
;;; Empirical kurtosis: 14.512069814124851
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 2674.496338 ms
;;; Empirical mean: 38.30953675788446
;;; Empirical variance: 4.7948677700924085
;;; Empirical std: 2.1897186508984228
;;; Empirical skewness: 1.0884389436303719
;;; Empirical kurtosis: 2.191234595213239
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 2572.332715 ms
;;; Empirical mean: 42.65170726605564
;;; Empirical variance: 3.107369876763625
;;; Empirical std: 1.7627733480977141
;;; Empirical skewness: -4.922550281235098
;;; Empirical kurtosis: 25.519967732557202
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 2427.168367 ms
;;; Empirical mean: 39.00065604350481
;;; Empirical variance: 0.009184178674274809
;;; Empirical std: 0.09583412061617098
;;; Empirical skewness: 146.07206517871725
;;; Empirical kurtosis: 21338.04822557542
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 2781.778174 ms
;;; Empirical mean: 41.65084108404922
;;; Empirical variance: 19.018769627823744
;;; Empirical std: 4.3610514360442645
;;; Empirical skewness: -0.5688101532046606
;;; Empirical kurtosis: 1.7734367079417097
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 2648.824098 ms
;;; Empirical mean: 43.039135637630416
;;; Empirical variance: 0.6378899902425116
;;; Empirical std: 0.7986801551575646
;;; Empirical skewness: 1.6735459812579845
;;; Empirical kurtosis: 24.951744231105543
;;;
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 1862.509525 ms
;;; Empirical mean: 42.62400000000001
;;; Empirical variance: 820.2866240000003
;;; Empirical std: 28.64064636142139
;;; Empirical skewness: 0.35685353445165235
;;; Empirical kurtosis: 2.4130646091265944
;;;
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 1797.508889 ms
;;; Empirical mean: 48.574000000000005
;;; Empirical variance: 911.9205239999999
;;; Empirical std: 30.198021855744127
;;; Empirical skewness: 0.4462320544389152
;;; Empirical kurtosis: 2.1134723098867387
;;;
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 1691.048241 ms
;;; Empirical mean: 53.77000000000002
;;; Empirical variance: 828.5930999999999
;;; Empirical std: 28.785293119924972
;;; Empirical skewness: 0.09624782976828822
;;; Empirical kurtosis: 1.7784401819931408
;;;
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 1780.397615 ms
;;; Empirical mean: 62.72
;;; Empirical variance: 534.6656000000002
;;; Empirical std: 23.1228371961574
;;; Empirical skewness: 4.958802175223551E-5
;;; Empirical kurtosis: 2.0610152119459517
;;;
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 1764.922985 ms
;;; Empirical mean: 56.70799999999997
;;; Empirical variance: 1075.0027360000001
;;; Empirical std: 32.7872343450923
;;; Empirical skewness: -0.48432568630132267
;;; Empirical kurtosis: 1.6769644514070723
;;;
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 1815.557863 ms
;;; Empirical mean: 70.22400000000002
;;; Empirical variance: 853.2018240000002
;;; Empirical std: 29.209618689739862
;;; Empirical skewness: -0.8792037025567655
;;; Empirical kurtosis: 3.0481095157804767
;;;
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 2013.450566 ms
;;; Empirical mean: 58.94199999999998
;;; Empirical variance: 1401.4586360000003
;;; Empirical std: 37.43606063677107
;;; Empirical skewness: -0.15077995540021624
;;; Empirical kurtosis: 1.4652856159746377
;;;
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 1693.415312 ms
;;; Empirical mean: 61.413999999999966
;;; Empirical variance: 535.9066039999999
;;; Empirical std: 23.1496566713202
;;; Empirical skewness: 0.27225184330984614
;;; Empirical kurtosis: 2.825080938819008
;;;
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 1723.885671 ms
;;; Empirical mean: 57.50399999999999
;;; Empirical variance: 841.9779840000001
;;; Empirical std: 29.01685689388153
;;; Empirical skewness: -0.09221331865065492
;;; Empirical kurtosis: 1.8670048782480497
;;;
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 1943.564774 ms
;;; Empirical mean: 51.05600000000001
;;; Empirical variance: 770.0168640000006
;;; Empirical std: 27.749177717546885
;;; Empirical skewness: 0.40918219661778643
;;; Empirical kurtosis: 2.396565702020384
;;;
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 1965.369206 ms
;;; Empirical mean: 57.81400000000005
;;; Empirical variance: 747.5714040000006
;;; Empirical std: 27.34175202871975
;;; Empirical skewness: -0.13205921473929133
;;; Empirical kurtosis: 2.104780346825597
;;;
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 1879.87036 ms
;;; Empirical mean: 45.30600000000003
;;; Empirical variance: 1516.156364
;;; Empirical std: 38.93785258588357
;;; Empirical skewness: 0.19620785154663933
;;; Empirical kurtosis: 1.4473182118384103
;;;
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 1759.245542 ms
;;; Empirical mean: 52.57799999999999
;;; Empirical variance: 1064.695916
;;; Empirical std: 32.62967845382482
;;; Empirical skewness: 0.3314837934763097
;;; Empirical kurtosis: 1.7874472042165084
;;;
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 1794.00086 ms
;;; Empirical mean: 61.06600000000002
;;; Empirical variance: 954.5336439999998
;;; Empirical std: 30.895527896444815
;;; Empirical skewness: -0.16704655248575467
;;; Empirical kurtosis: 1.79768972466093
;;;
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 2185.417798 ms
;;; Empirical mean: 61.55799999999998
;;; Empirical variance: 661.278636
;;; Empirical std: 25.71533853559
;;; Empirical skewness: -0.4135008837080506
;;; Empirical kurtosis: 2.296949232444073
;;;
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 1988.316117 ms
;;; Empirical mean: 55.97799999999999
;;; Empirical variance: 1068.049516
;;; Empirical std: 32.68102685045254
;;; Empirical skewness: -0.15906803878162007
;;; Empirical kurtosis: 1.6480593467161888
;;;
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 1897.737096 ms
;;; Empirical mean: 53.394
;;; Empirical variance: 1050.614764
;;; Empirical std: 32.41318811841871
;;; Empirical skewness: 0.44282217273128227
;;; Empirical kurtosis: 1.840022587144714
;;;
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 2024.227388 ms
;;; Empirical mean: 53.03199999999998
;;; Empirical variance: 873.6709759999999
;;; Empirical std: 29.557925772963163
;;; Empirical skewness: 0.19099296382639633
;;; Empirical kurtosis: 2.2999101897761074
;;;
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 1900.826052 ms
;;; Empirical mean: 64.71000000000001
;;; Empirical variance: 1162.5939
;;; Empirical std: 34.09683123106897
;;; Empirical skewness: -0.27333468711783887
;;; Empirical kurtosis: 1.705192449149035
;;;
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 1807.971108 ms
;;; Empirical mean: 52.66799999999998
;;; Empirical variance: 841.4817759999999
;;; Empirical std: 29.008305293484483
;;; Empirical skewness: 0.2215871056851981
;;; Empirical kurtosis: 2.165557724384576
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 2849.276366 ms
;;; Empirical mean: 40.952
;;; Empirical variance: 0.18969600000000034
;;; Empirical std: 0.43554104284211875
;;; Empirical skewness: -8.963563972121854
;;; Empirical kurtosis: 81.34547908232094
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 3228.481457 ms
;;; Empirical mean: 43.809
;;; Empirical variance: 16.066519000000003
;;; Empirical std: 4.008306250774758
;;; Empirical skewness: -1.3333625058162288
;;; Empirical kurtosis: 2.88004149127681
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 3104.399124 ms
;;; Empirical mean: 43.705000000000005
;;; Empirical variance: 8.227974999999997
;;; Empirical std: 2.8684447005302363
;;; Empirical skewness: -1.4557372386066738
;;; Empirical kurtosis: 3.201338632913688
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 2679.924231 ms
;;; Empirical mean: 45.083999999999996
;;; Empirical variance: 7.660943999999998
;;; Empirical std: 2.7678410358978347
;;; Empirical skewness: 0.5761502954508126
;;; Empirical kurtosis: 1.3936048275636914
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 2434.628999 ms
;;; Empirical mean: 41.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 2436.4974 ms
;;; Empirical mean: 34.800000000000004
;;; Empirical variance: 2.56
;;; Empirical std: 1.6
;;; Empirical skewness: 1.4999999999999918
;;; Empirical kurtosis: 3.249999999999983
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 2460.565398 ms
;;; Empirical mean: 34.64
;;; Empirical variance: 4.7104
;;; Empirical std: 2.170345594600086
;;; Empirical skewness: 0.6078224527424954
;;; Empirical kurtosis: 1.4507205148275037
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 2486.181148 ms
;;; Empirical mean: 41.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 2522.072299 ms
;;; Empirical mean: 45.592999999999996
;;; Empirical variance: 0.4613510000000002
;;; Empirical std: 0.679228238517805
;;; Empirical skewness: -7.710783218167126
;;; Empirical kurtosis: 118.61855805063286
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 2659.104676 ms
;;; Empirical mean: 45.529999999999994
;;; Empirical variance: 2.0491000000000006
;;; Empirical std: 1.4314677781913223
;;; Empirical skewness: -2.8217945596909373
;;; Empirical kurtosis: 9.037003400970049
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 2631.626815 ms
;;; Empirical mean: 41.995
;;; Empirical variance: 3.984975
;;; Empirical std: 1.9962402160060797
;;; Empirical skewness: 1.507834566133619
;;; Empirical kurtosis: 3.2735650788273554
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 2624.882985 ms
;;; Empirical mean: 42.0
;;; Empirical variance: 0.0
;;; Empirical std: 0.0
;;; Empirical skewness: NaN
;;; Empirical kurtosis: NaN
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 3133.086707 ms
;;; Empirical mean: 37.513
;;; Empirical variance: 10.617831000000013
;;; Empirical std: 3.2585013426420453
;;; Empirical skewness: 1.7526669986669345
;;; Empirical kurtosis: 7.304156980058477
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 2471.14943 ms
;;; Empirical mean: 41.14300000000001
;;; Empirical variance: 15.762551000000013
;;; Empirical std: 3.970207929063667
;;; Empirical skewness: -0.49929300946736904
;;; Empirical kurtosis: 2.771789670832245
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 2737.723377 ms
;;; Empirical mean: 39.565000000000005
;;; Empirical variance: 8.321774999999997
;;; Empirical std: 2.884748689227538
;;; Empirical skewness: 0.3149694641827109
;;; Empirical kurtosis: 1.147514517611998
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 2839.530557 ms
;;; Empirical mean: 41.366
;;; Empirical variance: 22.964043999999998
;;; Empirical std: 4.792081384951637
;;; Empirical skewness: 0.4387132782161906
;;; Empirical kurtosis: 1.4643392482479733
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 2805.531527 ms
;;; Empirical mean: 46.07300000000001
;;; Empirical variance: 0.5536710000000005
;;; Empirical std: 0.7440907202754248
;;; Empirical skewness: -10.254449401382727
;;; Empirical kurtosis: 284.34648857505647
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 2562.694117 ms
;;; Empirical mean: 40.69
;;; Empirical variance: 3.1539000000000006
;;; Empirical std: 1.7759222955974174
;;; Empirical skewness: 0.8592837200680004
;;; Empirical kurtosis: 5.6891283401401616
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 3019.52526 ms
;;; Empirical mean: 42.03599999999999
;;; Empirical variance: 15.550703999999998
;;; Empirical std: 3.9434380938465354
;;; Empirical skewness: -1.5318095541731376
;;; Empirical kurtosis: 3.388300802600298
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 2783.99069 ms
;;; Empirical mean: 39.23199999999999
;;; Empirical variance: 3.4101759999999985
;;; Empirical std: 1.8466661853188298
;;; Empirical skewness: 0.8317691698756209
;;; Empirical kurtosis: 1.6918399519555736
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 623.96168 ms
;;; Empirical mean: 65.05158751544255
;;; Empirical variance: 1155.5568475152452
;;; Empirical std: 33.993482427007166
;;; Empirical skewness: -0.15909137797862044
;;; Empirical kurtosis: 1.6387188832629171
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 440.251339 ms
;;; Empirical mean: 61.78099121202483
;;; Empirical variance: 1176.0165695695555
;;; Empirical std: 34.293097987343685
;;; Empirical skewness: -0.3850664499534397
;;; Empirical kurtosis: 1.6989041725122904
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 461.10947 ms
;;; Empirical mean: 61.7579362704628
;;; Empirical variance: 932.3421515158338
;;; Empirical std: 30.534278303503978
;;; Empirical skewness: -0.2877385565856876
;;; Empirical kurtosis: 2.016430907616677
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 457.864661 ms
;;; Empirical mean: 55.706346781178745
;;; Empirical variance: 830.3026824924297
;;; Empirical std: 28.81497323428272
;;; Empirical skewness: -0.12562078986144165
;;; Empirical kurtosis: 2.139236407719701
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 467.458181 ms
;;; Empirical mean: 47.746031620857714
;;; Empirical variance: 1183.459308624463
;;; Empirical std: 34.40144340902665
;;; Empirical skewness: 0.3093153380996414
;;; Empirical kurtosis: 1.9241676878092384
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 589.159329 ms
;;; Empirical mean: 63.73762461197175
;;; Empirical variance: 940.9582767675628
;;; Empirical std: 30.675043223564703
;;; Empirical skewness: -0.5940388016288676
;;; Empirical kurtosis: 2.2365076041417353
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 613.361283 ms
;;; Empirical mean: 52.376985861578774
;;; Empirical variance: 833.5823358881476
;;; Empirical std: 28.87182598811768
;;; Empirical skewness: -0.04643869761909838
;;; Empirical kurtosis: 1.6458795638873465
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 552.337117 ms
;;; Empirical mean: 65.86507936507937
;;; Empirical variance: 1067.1087805492568
;;; Empirical std: 32.666630994782075
;;; Empirical skewness: -0.3383964950550724
;;; Empirical kurtosis: 1.9565550451554643
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 481.548176 ms
;;; Empirical mean: 62.69047618952501
;;; Empirical variance: 1158.293083933155
;;; Empirical std: 34.033705116151474
;;; Empirical skewness: -0.32396151714926874
;;; Empirical kurtosis: 1.7926392729622185
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 477.810255 ms
;;; Empirical mean: 65.82539688612523
;;; Empirical variance: 705.7393553958996
;;; Empirical std: 26.565755313860354
;;; Empirical skewness: -0.1474947795661414
;;; Empirical kurtosis: 1.7492075981909267
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 457.309363 ms
;;; Empirical mean: 58.74206349216835
;;; Empirical variance: 829.7945798633519
;;; Empirical std: 28.80615524264479
;;; Empirical skewness: -0.35202291563756766
;;; Empirical kurtosis: 2.193347561184358
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 442.126275 ms
;;; Empirical mean: 54.02983263884077
;;; Empirical variance: 586.5151355381698
;;; Empirical std: 24.21807456298229
;;; Empirical skewness: 0.578058509105352
;;; Empirical kurtosis: 2.257989984099275
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 476.703771 ms
;;; Empirical mean: 53.16923975910689
;;; Empirical variance: 1236.9989361590985
;;; Empirical std: 35.170995666302915
;;; Empirical skewness: -0.054946331093800344
;;; Empirical kurtosis: 1.6975449920164773
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 476.4628 ms
;;; Empirical mean: 46.61976924617088
;;; Empirical variance: 1104.4660755035745
;;; Empirical std: 33.23350832373216
;;; Empirical skewness: 0.7038019372947416
;;; Empirical kurtosis: 2.081609027828967
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 442.577939 ms
;;; Empirical mean: 61.84895457415785
;;; Empirical variance: 861.4703296100379
;;; Empirical std: 29.35081480317093
;;; Empirical skewness: 0.23893376333885838
;;; Empirical kurtosis: 1.7490458645448563
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 475.605927 ms
;;; Empirical mean: 54.43439023825707
;;; Empirical variance: 811.3548151860276
;;; Empirical std: 28.484290673738528
;;; Empirical skewness: 0.6218396942206355
;;; Empirical kurtosis: 2.1209280521545737
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 448.844621 ms
;;; Empirical mean: 72.06283375592464
;;; Empirical variance: 579.0593848965261
;;; Empirical std: 24.063652775431375
;;; Empirical skewness: -0.8090632185540892
;;; Empirical kurtosis: 2.7740711292385836
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 465.042223 ms
;;; Empirical mean: 76.05158730143606
;;; Empirical variance: 476.14416415176714
;;; Empirical std: 21.820727855682705
;;; Empirical skewness: -0.689037559621636
;;; Empirical kurtosis: 2.1869217611924583
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 629.61097 ms
;;; Empirical mean: 62.23505976219166
;;; Empirical variance: 1108.0363803436594
;;; Empirical std: 33.287180420451044
;;; Empirical skewness: -0.26712804638949733
;;; Empirical kurtosis: 2.0884297536932075
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 533.372715 ms
;;; Empirical mean: 55.5856573705338
;;; Empirical variance: 1074.824336120759
;;; Empirical std: 32.78451366302021
;;; Empirical skewness: 0.18630377446886057
;;; Empirical kurtosis: 1.371464795618388
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 9313.92545 ms
;;; Empirical mean: 39.067441225645126
;;; Empirical variance: 0.4000990349542359
;;; Empirical std: 0.6325338211939626
;;; Empirical skewness: 9.272417303534576
;;; Empirical kurtosis: 86.97772265088747
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 9750.767462 ms
;;; Empirical mean: 40.86140871772144
;;; Empirical variance: 1.118106623920153
;;; Empirical std: 1.0574056099341222
;;; Empirical skewness: -7.676503335285745
;;; Empirical kurtosis: 61.43380907283403
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 9447.058212 ms
;;; Empirical mean: 37.46530936414784
;;; Empirical variance: 4.43101937061093
;;; Empirical std: 2.1049986628525277
;;; Empirical skewness: 4.504148553605964
;;; Empirical kurtosis: 21.917467465763433
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 11064.889773 ms
;;; Empirical mean: 37.00078590368838
;;; Empirical variance: 0.007072515550814051
;;; Empirical std: 0.08409824939208932
;;; Empirical skewness: 106.99899531404125
;;; Empirical kurtosis: 11449.784998214222
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 10482.319707 ms
;;; Empirical mean: 41.67257552262328
;;; Empirical variance: 4.9694227723217015
;;; Empirical std: 2.2292202162015538
;;; Empirical skewness: -0.6944958354284662
;;; Empirical kurtosis: 5.769033205752458
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 9547.176779 ms
;;; Empirical mean: 39.77186295196312
;;; Empirical variance: 14.8101215964928
;;; Empirical std: 3.8483920793615614
;;; Empirical skewness: 0.11806062593573964
;;; Empirical kurtosis: 1.1845677395485013
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 9845.288356 ms
;;; Empirical mean: 39.36001056160143
;;; Empirical variance: 0.3178788307245249
;;; Empirical std: 0.5638074411751984
;;; Empirical skewness: 1.2970410045273055
;;; Empirical kurtosis: 3.6972503427288888
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 11555.633831 ms
;;; Empirical mean: 41.468642575458816
;;; Empirical variance: 7.845438901506095
;;; Empirical std: 2.8009710640251346
;;; Empirical skewness: -0.05399327960813846
;;; Empirical kurtosis: 3.175961557753749
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 12297.09857 ms
;;; Empirical mean: 41.10133693185473
;;; Empirical variance: 0.30922733988732154
;;; Empirical std: 0.5560821341198812
;;; Empirical skewness: -2.951106743583702
;;; Empirical kurtosis: 92.28413911576028
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 12602.079877 ms
;;; Empirical mean: 44.83734503306618
;;; Empirical variance: 1.3345375034766556
;;; Empirical std: 1.1552218416722633
;;; Empirical skewness: -10.717422953706908
;;; Empirical kurtosis: 133.07436767659047
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 10954.200896 ms
;;; Empirical mean: 48.99580195896592
;;; Empirical variance: 0.019215262119275035
;;; Empirical std: 0.13861912609476024
;;; Empirical skewness: -2.1449470935696224
;;; Empirical kurtosis: 296.1435394948644
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 11636.060007 ms
;;; Empirical mean: 44.01693470181379
;;; Empirical variance: 0.13519083038474233
;;; Empirical std: 0.36768305697263554
;;; Empirical skewness: 21.6657538205936
;;; Empirical kurtosis: 470.4048886145671
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 13198.662012 ms
;;; Empirical mean: 39.53028632942432
;;; Empirical variance: 4.259344793707482
;;; Empirical std: 2.0638180137084476
;;; Empirical skewness: -0.28709758086575443
;;; Empirical kurtosis: 2.666925350349594
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 12215.459186 ms
;;; Empirical mean: 36.99973924551789
;;; Empirical variance: 7.821954534500135E-4
;;; Empirical std: 0.0279677573904311
;;; Empirical skewness: -107.2477299185247
;;; Empirical kurtosis: 11503.07557267679
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 11508.51754 ms
;;; Empirical mean: 38.02455253207425
;;; Empirical variance: 7.01775781713264
;;; Empirical std: 2.649105097411698
;;; Empirical skewness: 0.5091437469947536
;;; Empirical kurtosis: 1.6532272362242555
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 11842.823881 ms
;;; Empirical mean: 39.81100319575015
;;; Empirical variance: 0.5254929973331997
;;; Empirical std: 0.7249089579617565
;;; Empirical skewness: -3.5895473161307927
;;; Empirical kurtosis: 13.92581976435478
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 11086.740111 ms
;;; Empirical mean: 42.533059195434866
;;; Empirical variance: 6.364873899373605
;;; Empirical std: 2.522870170931038
;;; Empirical skewness: -0.5626151620951351
;;; Empirical kurtosis: 1.659951248054185
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 20788.009024 ms
;;; Empirical mean: 39.7699267950536
;;; Empirical variance: 37.841681161302056
;;; Empirical std: 6.151559246345763
;;; Empirical skewness: -0.13404562259965067
;;; Empirical kurtosis: 1.102546437816345
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 21954.645859 ms
;;; Empirical mean: 39.42410185757321
;;; Empirical variance: 7.780486374128698
;;; Empirical std: 2.789352321620325
;;; Empirical skewness: -1.7134472700043928
;;; Empirical kurtosis: 6.211794459771139
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 18574.292894 ms
;;; Empirical mean: 40.35319701768321
;;; Empirical variance: 10.461312990248947
;;; Empirical std: 3.2343953051921384
;;; Empirical skewness: -1.639198797340749
;;; Empirical kurtosis: 3.9865399961000727
;;;
;;; Method :importance with options []
;;; Chain no. 0
;;; Elapsed time: 9002.129708 ms
;;; Empirical mean: 39.47787948182638
;;; Empirical variance: 2.089281577331411
;;; Empirical std: 1.4454347364483153
;;; Empirical skewness: 0.2528082001129349
;;; Empirical kurtosis: 12.199582936148186
;;;
;;; Method :importance with options []
;;; Chain no. 1
;;; Elapsed time: 8800.887145 ms
;;; Empirical mean: 40.61582212846028
;;; Empirical variance: 6.3475343435429235
;;; Empirical std: 2.519431353211062
;;; Empirical skewness: 0.68145512096059
;;; Empirical kurtosis: 4.281883192353494
;;;
;;; Method :importance with options []
;;; Chain no. 2
;;; Elapsed time: 8721.965275 ms
;;; Empirical mean: 39.68030277161031
;;; Empirical variance: 6.106088634368682
;;; Empirical std: 2.4710501076199733
;;; Empirical skewness: -0.1745449124162888
;;; Empirical kurtosis: 3.289536702258819
;;;
;;; Method :importance with options []
;;; Chain no. 3
;;; Elapsed time: 9031.043963 ms
;;; Empirical mean: 39.68781737675997
;;; Empirical variance: 7.6843689306124805
;;; Empirical std: 2.7720694310591285
;;; Empirical skewness: 0.8249002852548185
;;; Empirical kurtosis: 3.386373991749025
;;;
;;; Method :importance with options []
;;; Chain no. 4
;;; Elapsed time: 8789.373734 ms
;;; Empirical mean: 38.86385464649344
;;; Empirical variance: 3.013976753315319
;;; Empirical std: 1.7360808602468143
;;; Empirical skewness: -0.411695645528245
;;; Empirical kurtosis: 2.3103461067639217
;;;
;;; Method :importance with options []
;;; Chain no. 5
;;; Elapsed time: 8754.197281 ms
;;; Empirical mean: 39.39656578880089
;;; Empirical variance: 6.7567635141963756
;;; Empirical std: 2.59937752436932
;;; Empirical skewness: 0.7216190001524849
;;; Empirical kurtosis: 3.343305875872252
;;;
;;; Method :importance with options []
;;; Chain no. 6
;;; Elapsed time: 9141.875556 ms
;;; Empirical mean: 39.80233193467357
;;; Empirical variance: 9.526764096963692
;;; Empirical std: 3.0865456576833092
;;; Empirical skewness: 0.08646844050483746
;;; Empirical kurtosis: 2.607736356325663
;;;
;;; Method :importance with options []
;;; Chain no. 7
;;; Elapsed time: 8901.966647 ms
;;; Empirical mean: 39.600398992865706
;;; Empirical variance: 5.620695382818866
;;; Empirical std: 2.3708005784584385
;;; Empirical skewness: 0.31718868452078064
;;; Empirical kurtosis: 4.354404742778393
;;;
;;; Method :importance with options []
;;; Chain no. 8
;;; Elapsed time: 8961.708286 ms
;;; Empirical mean: 39.34580683755565
;;; Empirical variance: 11.47886171688005
;;; Empirical std: 3.388046888235175
;;; Empirical skewness: 0.8010403459748395
;;; Empirical kurtosis: 2.3505990129129555
;;;
;;; Method :importance with options []
;;; Chain no. 9
;;; Elapsed time: 9057.525579 ms
;;; Empirical mean: 39.929668708653615
;;; Empirical variance: 3.0778128387499035
;;; Empirical std: 1.7543696414239227
;;; Empirical skewness: 1.5169979581567492
;;; Empirical kurtosis: 8.44354753074102
;;;
;;; Method :importance with options []
;;; Chain no. 10
;;; Elapsed time: 9299.032788 ms
;;; Empirical mean: 40.420020535224054
;;; Empirical variance: 8.470380168441967
;;; Empirical std: 2.9103917551494622
;;; Empirical skewness: 0.4037943126158755
;;; Empirical kurtosis: 2.996834736300035
;;;
;;; Method :importance with options []
;;; Chain no. 11
;;; Elapsed time: 9358.80668 ms
;;; Empirical mean: 39.83655990295461
;;; Empirical variance: 7.476523898565766
;;; Empirical std: 2.7343232981060903
;;; Empirical skewness: 0.9883844089092212
;;; Empirical kurtosis: 3.7558351982661486
;;;
;;; Method :importance with options []
;;; Chain no. 12
;;; Elapsed time: 9056.25563 ms
;;; Empirical mean: 38.10935647255367
;;; Empirical variance: 6.305889319159136
;;; Empirical std: 2.5111529860124286
;;; Empirical skewness: 1.974964302845194
;;; Empirical kurtosis: 6.227461624519526
;;;
;;; Method :importance with options []
;;; Chain no. 13
;;; Elapsed time: 9371.037406 ms
;;; Empirical mean: 38.39158633381137
;;; Empirical variance: 7.981174537763787
;;; Empirical std: 2.8250972616467185
;;; Empirical skewness: 1.9480906681044758
;;; Empirical kurtosis: 5.421345937480642
;;;
;;; Method :importance with options []
;;; Chain no. 14
;;; Elapsed time: 8657.008198 ms
;;; Empirical mean: 40.285582617427465
;;; Empirical variance: 3.863143793366348
;;; Empirical std: 1.9654881819452257
;;; Empirical skewness: -1.5764879326823775
;;; Empirical kurtosis: 10.106108407677306
;;;
;;; Method :importance with options []
;;; Chain no. 15
;;; Elapsed time: 9232.591967 ms
;;; Empirical mean: 40.2071818509439
;;; Empirical variance: 10.167120224398706
;;; Empirical std: 3.188592201018924
;;; Empirical skewness: 0.08483880881148519
;;; Empirical kurtosis: 2.241094685031136
;;;
;;; Method :importance with options []
;;; Chain no. 16
;;; Elapsed time: 8699.64852 ms
;;; Empirical mean: 39.8120489220851
;;; Empirical variance: 2.2196179286902047
;;; Empirical std: 1.4898382223215394
;;; Empirical skewness: -1.2735135674387488
;;; Empirical kurtosis: 7.497083693316523
;;;
;;; Method :importance with options []
;;; Chain no. 17
;;; Elapsed time: 8880.807513 ms
;;; Empirical mean: 38.88049110518536
;;; Empirical variance: 3.931178639483062
;;; Empirical std: 1.9827200103602782
;;; Empirical skewness: 0.9575679735365838
;;; Empirical kurtosis: 4.939243911715569
;;;
;;; Method :importance with options []
;;; Chain no. 18
;;; Elapsed time: 9387.278835 ms
;;; Empirical mean: 39.75167784464212
;;; Empirical variance: 5.398494419730817
;;; Empirical std: 2.3234660358461916
;;; Empirical skewness: 0.6486594762063386
;;; Empirical kurtosis: 4.945091969508445
;;;
;;; Method :importance with options []
;;; Chain no. 19
;;; Elapsed time: 9265.749907 ms
;;; Empirical mean: 39.97713602732286
;;; Empirical variance: 8.05972987301438
;;; Empirical std: 2.838966338830804
;;; Empirical skewness: 0.41905151061441187
;;; Empirical kurtosis: 3.031979750176833
;;;
;;; Method :lmh with options []
;;; Chain no. 0
;;; Elapsed time: 9401.821905 ms
;;; Empirical mean: 39.45259999999999
;;; Empirical variance: 4.268753240000001
;;; Empirical std: 2.066096135227013
;;; Empirical skewness: -0.08860061202189112
;;; Empirical kurtosis: 3.100236148254932
;;;
;;; Method :lmh with options []
;;; Chain no. 1
;;; Elapsed time: 9933.368194 ms
;;; Empirical mean: 40.02579999999999
;;; Empirical variance: 6.023734359999997
;;; Empirical std: 2.4543297170510723
;;; Empirical skewness: 0.6805863828327731
;;; Empirical kurtosis: 4.253778449270797
;;;
;;; Method :lmh with options []
;;; Chain no. 2
;;; Elapsed time: 9139.621975 ms
;;; Empirical mean: 39.67659999999999
;;; Empirical variance: 6.355412439999993
;;; Empirical std: 2.5209943355747537
;;; Empirical skewness: -0.13226567770868186
;;; Empirical kurtosis: 3.712731732421464
;;;
;;; Method :lmh with options []
;;; Chain no. 3
;;; Elapsed time: 9389.552697 ms
;;; Empirical mean: 39.7389
;;; Empirical variance: 5.534526790000001
;;; Empirical std: 2.3525574998286443
;;; Empirical skewness: 0.3447768056702894
;;; Empirical kurtosis: 4.3451497419313965
;;;
;;; Method :lmh with options []
;;; Chain no. 4
;;; Elapsed time: 9144.06554 ms
;;; Empirical mean: 40.33420000000002
;;; Empirical variance: 7.397710360000005
;;; Empirical std: 2.719873224986783
;;; Empirical skewness: 0.5241496818727791
;;; Empirical kurtosis: 3.4447373980297695
;;;
;;; Method :lmh with options []
;;; Chain no. 5
;;; Elapsed time: 9131.396643 ms
;;; Empirical mean: 39.8996
;;; Empirical variance: 5.906519840000003
;;; Empirical std: 2.430333277556805
;;; Empirical skewness: 0.22689645026468713
;;; Empirical kurtosis: 3.2050154771495314
;;;
;;; Method :lmh with options []
;;; Chain no. 6
;;; Elapsed time: 10313.776309 ms
;;; Empirical mean: 39.8589
;;; Empirical variance: 6.524790790000002
;;; Empirical std: 2.5543670037799977
;;; Empirical skewness: 0.4629859757438951
;;; Empirical kurtosis: 3.1113141271393507
;;;
;;; Method :lmh with options []
;;; Chain no. 7
;;; Elapsed time: 10414.489445 ms
;;; Empirical mean: 40.12100000000001
;;; Empirical variance: 10.396759
;;; Empirical std: 3.224400564446049
;;; Empirical skewness: 1.029415030008039
;;; Empirical kurtosis: 8.546928648129587
;;;
;;; Method :lmh with options []
;;; Chain no. 8
;;; Elapsed time: 9442.847738 ms
;;; Empirical mean: 39.4063
;;; Empirical variance: 5.31402031
;;; Empirical std: 2.305215892275602
;;; Empirical skewness: 0.08453647375087978
;;; Empirical kurtosis: 2.352443924539363
;;;
;;; Method :lmh with options []
;;; Chain no. 9
;;; Elapsed time: 9577.111832 ms
;;; Empirical mean: 39.8349
;;; Empirical variance: 5.280441990000002
;;; Empirical std: 2.2979212323315177
;;; Empirical skewness: 0.42437665566144883
;;; Empirical kurtosis: 3.4876943312618702
;;;
;;; Method :lmh with options []
;;; Chain no. 10
;;; Elapsed time: 9396.807228 ms
;;; Empirical mean: 39.59910000000001
;;; Empirical variance: 5.953979190000001
;;; Empirical std: 2.440077701631651
;;; Empirical skewness: 0.3580205269844387
;;; Empirical kurtosis: 3.2492938199849135
;;;
;;; Method :lmh with options []
;;; Chain no. 11
;;; Elapsed time: 9189.560529 ms
;;; Empirical mean: 40.012599999999985
;;; Empirical variance: 6.2124412399999995
;;; Empirical std: 2.4924769286795816
;;; Empirical skewness: 0.15850617633329053
;;; Empirical kurtosis: 4.225360974164278
;;;
;;; Method :lmh with options []
;;; Chain no. 12
;;; Elapsed time: 9158.827956 ms
;;; Empirical mean: 39.28969999999999
;;; Empirical variance: 6.479173909999999
;;; Empirical std: 2.5454221476996697
;;; Empirical skewness: 0.3666536030099034
;;; Empirical kurtosis: 3.009517750380934
;;;
;;; Method :lmh with options []
;;; Chain no. 13
;;; Elapsed time: 9374.09801 ms
;;; Empirical mean: 40.05910000000001
;;; Empirical variance: 5.329407189999999
;;; Empirical std: 2.3085508852957948
;;; Empirical skewness: 0.37313762370692716
;;; Empirical kurtosis: 3.523311402253195
;;;
;;; Method :lmh with options []
;;; Chain no. 14
;;; Elapsed time: 10271.876195 ms
;;; Empirical mean: 39.94319999999999
;;; Empirical variance: 6.159973759999998
;;; Empirical std: 2.4819294429938976
;;; Empirical skewness: 0.06597343403811033
;;; Empirical kurtosis: 3.615045764628097
;;;
;;; Method :lmh with options []
;;; Chain no. 15
;;; Elapsed time: 10077.022942 ms
;;; Empirical mean: 40.02569999999999
;;; Empirical variance: 5.500039509999997
;;; Empirical std: 2.3452163034568896
;;; Empirical skewness: 0.5092131808960692
;;; Empirical kurtosis: 3.6019346695162837
;;;
;;; Method :lmh with options []
;;; Chain no. 16
;;; Elapsed time: 10433.020975 ms
;;; Empirical mean: 39.760600000000004
;;; Empirical variance: 6.07948764
;;; Empirical std: 2.4656617042895403
;;; Empirical skewness: 0.1609082762902039
;;; Empirical kurtosis: 3.3508578597672565
;;;
;;; Method :lmh with options []
;;; Chain no. 17
;;; Elapsed time: 10247.686055 ms
;;; Empirical mean: 40.07489999999999
;;; Empirical variance: 5.381289990000001
;;; Empirical std: 2.3197607613717413
;;; Empirical skewness: 0.4001747777282227
;;; Empirical kurtosis: 3.924924868519931
;;;
;;; Method :lmh with options []
;;; Chain no. 18
;;; Elapsed time: 9757.312358 ms
;;; Empirical mean: 39.8942
;;; Empirical variance: 5.98580636
;;; Empirical std: 2.4465907626736434
;;; Empirical skewness: 0.2522584498938238
;;; Empirical kurtosis: 4.028645500562508
;;;
;;; Method :lmh with options []
;;; Chain no. 19
;;; Elapsed time: 16282.178185 ms
;;; Empirical mean: 39.4041
;;; Empirical variance: 5.06880319
;;; Empirical std: 2.2514002731633482
;;; Empirical skewness: 0.1768680752164399
;;; Empirical kurtosis: 3.033934689779063
;;;
;;; Method :rmh with options []
;;; Chain no. 0
;;; Elapsed time: 20695.466747 ms
;;; Empirical mean: 40.0409
;;; Empirical variance: 1.9794271900000044
;;; Empirical std: 1.4069211740534735
;;; Empirical skewness: -0.5286527981778327
;;; Empirical kurtosis: 4.986684753325416
;;;
;;; Method :rmh with options []
;;; Chain no. 1
;;; Elapsed time: 11471.815758 ms
;;; Empirical mean: 39.0893
;;; Empirical variance: 6.390125509999995
;;; Empirical std: 2.5278697573253246
;;; Empirical skewness: 1.1478739763153871
;;; Empirical kurtosis: 5.578349191691564
;;;
;;; Method :rmh with options []
;;; Chain no. 2
;;; Elapsed time: 10473.018171 ms
;;; Empirical mean: 40.093699999999984
;;; Empirical variance: 4.054520310000002
;;; Empirical std: 2.0135839465987013
;;; Empirical skewness: 0.07204106498901404
;;; Empirical kurtosis: 4.182653120786107
;;;
;;; Method :rmh with options []
;;; Chain no. 3
;;; Elapsed time: 9948.309265 ms
;;; Empirical mean: 39.95289999999999
;;; Empirical variance: 2.6806815900000056
;;; Empirical std: 1.6372787148191983
;;; Empirical skewness: -0.9242638484187744
;;; Empirical kurtosis: 2.723796432558012
;;;
;;; Method :rmh with options []
;;; Chain no. 4
;;; Elapsed time: 10087.724865 ms
;;; Empirical mean: 39.9654
;;; Empirical variance: 2.564202840000001
;;; Empirical std: 1.6013128488836905
;;; Empirical skewness: 0.721271328156984
;;; Empirical kurtosis: 5.717914436409545
;;;
;;; Method :rmh with options []
;;; Chain no. 5
;;; Elapsed time: 9692.837507 ms
;;; Empirical mean: 39.849399999999996
;;; Empirical variance: 3.262919640000002
;;; Empirical std: 1.806355347100897
;;; Empirical skewness: -0.24832018100509343
;;; Empirical kurtosis: 4.663083200457539
;;;
;;; Method :rmh with options []
;;; Chain no. 6
;;; Elapsed time: 10151.892856 ms
;;; Empirical mean: 39.9508
;;; Empirical variance: 3.5819793600000027
;;; Empirical std: 1.8926117826960718
;;; Empirical skewness: -0.4647946618594393
;;; Empirical kurtosis: 3.7924919754068886
;;;
;;; Method :rmh with options []
;;; Chain no. 7
;;; Elapsed time: 10472.297372 ms
;;; Empirical mean: 39.856599999999986
;;; Empirical variance: 3.7542364400000032
;;; Empirical std: 1.9375852084489094
;;; Empirical skewness: -0.27651036077260055
;;; Empirical kurtosis: 4.052693855618206
;;;
;;; Method :rmh with options []
;;; Chain no. 8
;;; Elapsed time: 9957.464211 ms
;;; Empirical mean: 39.8204
;;; Empirical variance: 4.492343840000001
;;; Empirical std: 2.119515001126437
;;; Empirical skewness: 0.4352705796066971
;;; Empirical kurtosis: 4.560052144864518
;;;
;;; Method :rmh with options []
;;; Chain no. 9
;;; Elapsed time: 9618.803658 ms
;;; Empirical mean: 39.767
;;; Empirical variance: 4.647510999999999
;;; Empirical std: 2.1558086649793387
;;; Empirical skewness: 0.3159610614392392
;;; Empirical kurtosis: 5.067363230225595
;;;
;;; Method :rmh with options []
;;; Chain no. 10
;;; Elapsed time: 9731.689143 ms
;;; Empirical mean: 39.830499999999994
;;; Empirical variance: 5.069969749999998
;;; Empirical std: 2.251659332581196
;;; Empirical skewness: 0.43825485762956945
;;; Empirical kurtosis: 3.380437580682397
;;;
;;; Method :rmh with options []
;;; Chain no. 11
;;; Elapsed time: 10095.639978 ms
;;; Empirical mean: 39.96619999999999
;;; Empirical variance: 5.222657560000001
;;; Empirical std: 2.285313448960558
;;; Empirical skewness: 0.35005217622727236
;;; Empirical kurtosis: 4.167379147860043
;;;
;;; Method :rmh with options []
;;; Chain no. 12
;;; Elapsed time: 10778.077627 ms
;;; Empirical mean: 40.114599999999996
;;; Empirical variance: 3.3576668400000096
;;; Empirical std: 1.832393745896337
;;; Empirical skewness: -0.2230728668973578
;;; Empirical kurtosis: 4.453531522648453
;;;
;;; Method :rmh with options []
;;; Chain no. 13
;;; Elapsed time: 10855.425752 ms
;;; Empirical mean: 39.1103
;;; Empirical variance: 4.22133391
;;; Empirical std: 2.054588501379291
;;; Empirical skewness: -0.09370132077909338
;;; Empirical kurtosis: 2.284609485096441
;;;
;;; Method :rmh with options []
;;; Chain no. 14
;;; Elapsed time: 11080.602559 ms
;;; Empirical mean: 39.4192
;;; Empirical variance: 3.30967136
;;; Empirical std: 1.8192502191837163
;;; Empirical skewness: -0.1448082911418673
;;; Empirical kurtosis: 2.6391738154500963
;;;
;;; Method :rmh with options []
;;; Chain no. 15
;;; Elapsed time: 9843.481294 ms
;;; Empirical mean: 39.95010000000001
;;; Empirical variance: 3.328409990000005
;;; Empirical std: 1.8243930470159122
;;; Empirical skewness: 0.30744176099624276
;;; Empirical kurtosis: 4.042366161406561
;;;
;;; Method :rmh with options []
;;; Chain no. 16
;;; Elapsed time: 10406.1382 ms
;;; Empirical mean: 39.91040000000001
;;; Empirical variance: 4.516971840000005
;;; Empirical std: 2.125316879902855
;;; Empirical skewness: 0.7346545757066267
;;; Empirical kurtosis: 4.764753900577343
;;;
;;; Method :rmh with options []
;;; Chain no. 17
;;; Elapsed time: 10774.655014 ms
;;; Empirical mean: 39.62409999999999
;;; Empirical variance: 2.458399190000007
;;; Empirical std: 1.5679283114989686
;;; Empirical skewness: 0.8118499618715712
;;; Empirical kurtosis: 5.991731113925503
;;;
;;; Method :rmh with options []
;;; Chain no. 18
;;; Elapsed time: 9718.256383 ms
;;; Empirical mean: 39.6857
;;; Empirical variance: 3.850115510000002
;;; Empirical std: 1.9621711214876245
;;; Empirical skewness: 0.5482791738962633
;;; Empirical kurtosis: 4.698797237250671
;;;
;;; Method :rmh with options []
;;; Chain no. 19
;;; Elapsed time: 9598.371921 ms
;;; Empirical mean: 39.507299999999994
;;; Empirical variance: 3.947346710000002
;;; Empirical std: 1.9867930717616271
;;; Empirical skewness: -0.41883285856037505
;;; Empirical kurtosis: 3.584538503926444
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 0
;;; Elapsed time: 9735.316094 ms
;;; Empirical mean: 39.7921
;;; Empirical variance: 4.224477590000003
;;; Empirical std: 2.055353397836976
;;; Empirical skewness: 0.5821734797899092
;;; Empirical kurtosis: 4.883990688672083
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 1
;;; Elapsed time: 9633.542901 ms
;;; Empirical mean: 40.07559999999999
;;; Empirical variance: 3.6294846400000003
;;; Empirical std: 1.905120636600213
;;; Empirical skewness: 0.7609400644888542
;;; Empirical kurtosis: 5.201455298155477
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 2
;;; Elapsed time: 9679.867296 ms
;;; Empirical mean: 39.339600000000004
;;; Empirical variance: 3.7926718400000023
;;; Empirical std: 1.947478328505866
;;; Empirical skewness: -0.27101271154340556
;;; Empirical kurtosis: 2.3011278850731354
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 3
;;; Elapsed time: 9592.527175 ms
;;; Empirical mean: 39.595099999999995
;;; Empirical variance: 6.538155990000008
;;; Empirical std: 2.556981812606419
;;; Empirical skewness: 0.8226715798821618
;;; Empirical kurtosis: 3.913597509536526
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 4
;;; Elapsed time: 9733.53275 ms
;;; Empirical mean: 39.465900000000005
;;; Empirical variance: 4.093037190000002
;;; Empirical std: 2.023125599165806
;;; Empirical skewness: -0.07512233463002678
;;; Empirical kurtosis: 3.035094215465464
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 5
;;; Elapsed time: 9750.52553 ms
;;; Empirical mean: 39.70410000000001
;;; Empirical variance: 2.4029431900000024
;;; Empirical std: 1.5501429579235595
;;; Empirical skewness: -0.5763097320858944
;;; Empirical kurtosis: 2.950820067718393
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 6
;;; Elapsed time: 9613.382636 ms
;;; Empirical mean: 39.950199999999995
;;; Empirical variance: 4.0533199600000005
;;; Empirical std: 2.0132858614712417
;;; Empirical skewness: 0.33582406380637814
;;; Empirical kurtosis: 4.035556630012213
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 7
;;; Elapsed time: 9747.670204 ms
;;; Empirical mean: 39.646
;;; Empirical variance: 4.733483999999999
;;; Empirical std: 2.1756571421067243
;;; Empirical skewness: 0.38005003950698013
;;; Empirical kurtosis: 3.8308228425796944
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 8
;;; Elapsed time: 10086.996329 ms
;;; Empirical mean: 40.02319999999999
;;; Empirical variance: 3.5192617600000085
;;; Empirical std: 1.8759695519917183
;;; Empirical skewness: -0.584419387477528
;;; Empirical kurtosis: 3.7474144248526593
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 9
;;; Elapsed time: 9873.51062 ms
;;; Empirical mean: 39.8984
;;; Empirical variance: 5.561077439999999
;;; Empirical std: 2.3581936816131113
;;; Empirical skewness: 0.3605940180169424
;;; Empirical kurtosis: 3.5500407357379387
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 10
;;; Elapsed time: 9897.303059 ms
;;; Empirical mean: 39.43
;;; Empirical variance: 5.591499999999997
;;; Empirical std: 2.3646352784309035
;;; Empirical skewness: 0.26022499701131896
;;; Empirical kurtosis: 2.979198807697032
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 11
;;; Elapsed time: 9859.691799 ms
;;; Empirical mean: 40.00370000000001
;;; Empirical variance: 2.018886310000003
;;; Empirical std: 1.4208751915632856
;;; Empirical skewness: -1.2794841652527726
;;; Empirical kurtosis: 4.728144854822912
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 12
;;; Elapsed time: 9648.379822 ms
;;; Empirical mean: 40.02479999999999
;;; Empirical variance: 2.75158496
;;; Empirical std: 1.6587902097613187
;;; Empirical skewness: 0.039188208497538866
;;; Empirical kurtosis: 5.3961584983102355
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 13
;;; Elapsed time: 9612.710719 ms
;;; Empirical mean: 40.0012
;;; Empirical variance: 3.5947985599999988
;;; Empirical std: 1.8959954008383033
;;; Empirical skewness: 0.6569309939302165
;;; Empirical kurtosis: 4.96728157966742
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 14
;;; Elapsed time: 9535.742658 ms
;;; Empirical mean: 39.8905
;;; Empirical variance: 3.86790975000001
;;; Empirical std: 1.9667002186403524
;;; Empirical skewness: -0.2866798056894022
;;; Empirical kurtosis: 3.846973239507513
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 15
;;; Elapsed time: 9588.323032 ms
;;; Empirical mean: 39.264399999999995
;;; Empirical variance: 3.0022926400000025
;;; Empirical std: 1.7327125093332716
;;; Empirical skewness: -0.32884772206387775
;;; Empirical kurtosis: 2.712170541297867
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 16
;;; Elapsed time: 10145.387894 ms
;;; Empirical mean: 40.3382
;;; Empirical variance: 3.297420760000008
;;; Empirical std: 1.8158801612441302
;;; Empirical skewness: -0.13785648498788033
;;; Empirical kurtosis: 5.104619280857172
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 17
;;; Elapsed time: 9934.562304 ms
;;; Empirical mean: 39.708499999999994
;;; Empirical variance: 4.520527750000005
;;; Empirical std: 2.126153275283794
;;; Empirical skewness: 0.046827098815194784
;;; Empirical kurtosis: 3.410626975521736
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 18
;;; Elapsed time: 9559.988089 ms
;;; Empirical mean: 40.06609999999999
;;; Empirical variance: 3.0861307900000012
;;; Empirical std: 1.7567386800546065
;;; Empirical skewness: 0.11406964311001076
;;; Empirical kurtosis: 4.790412667509761
;;;
;;; Method :rmh with options [:alpha 0.8 :sigma 2]
;;; Chain no. 19
;;; Elapsed time: 9661.734431 ms
;;; Empirical mean: 39.64779999999999
;;; Empirical variance: 2.7537551600000003
;;; Empirical std: 1.6594442322657308
;;; Empirical skewness: -0.2665148422172437
;;; Empirical kurtosis: 4.156562491116381
;;;
;;; Method :smc with options []
;;; Chain no. 0
;;; Elapsed time: 18067.72375 ms
;;; Empirical mean: 39.53519711498565
;;; Empirical variance: 5.334890178005713
;;; Empirical std: 2.309738118922947
;;; Empirical skewness: 1.1662889958833815
;;; Empirical kurtosis: 5.695341114710127
;;;
;;; Method :smc with options []
;;; Chain no. 1
;;; Elapsed time: 18882.173392 ms
;;; Empirical mean: 38.60856335197843
;;; Empirical variance: 11.759419063719337
;;; Empirical std: 3.4292009366205614
;;; Empirical skewness: 1.369982082919225
;;; Empirical kurtosis: 3.4097419623744845
;;;
;;; Method :smc with options []
;;; Chain no. 2
;;; Elapsed time: 18308.189993 ms
;;; Empirical mean: 41.169247760147584
;;; Empirical variance: 5.164696897468839
;;; Empirical std: 2.272596950070302
;;; Empirical skewness: 0.35450653855922215
;;; Empirical kurtosis: 3.98170122104393
;;;
;;; Method :smc with options []
;;; Chain no. 3
;;; Elapsed time: 17959.345072 ms
;;; Empirical mean: 37.589542113402366
;;; Empirical variance: 3.6981046495133323
;;; Empirical std: 1.9230456701579743
;;; Empirical skewness: 2.5758524622925667
;;; Empirical kurtosis: 9.927184606863783
;;;
;;; Method :smc with options []
;;; Chain no. 4
;;; Elapsed time: 18735.345412 ms
;;; Empirical mean: 37.81051722111869
;;; Empirical variance: 6.467985922660038
;;; Empirical std: 2.5432235298258856
;;; Empirical skewness: 1.0908123358054889
;;; Empirical kurtosis: 3.3275795708489566
;;;
;;; Method :smc with options []
;;; Chain no. 5
;;; Elapsed time: 18918.490536 ms
;;; Empirical mean: 39.47009523102814
;;; Empirical variance: 7.457581415144621
;;; Empirical std: 2.730857267442702
;;; Empirical skewness: 0.37322971075083566
;;; Empirical kurtosis: 2.6912858585831714
;;;
;;; Method :smc with options []
;;; Chain no. 6
;;; Elapsed time: 18232.099763 ms
;;; Empirical mean: 39.652917321948784
;;; Empirical variance: 2.5640255361835105
;;; Empirical std: 1.6012574859102175
;;; Empirical skewness: 0.8069755827263383
;;; Empirical kurtosis: 7.802818383403628
;;;
;;; Method :smc with options []
;;; Chain no. 7
;;; Elapsed time: 21661.52344 ms
;;; Empirical mean: 38.623687626357004
;;; Empirical variance: 4.27131151877673
;;; Empirical std: 2.066715151823475
;;; Empirical skewness: 1.2774593423885798
;;; Empirical kurtosis: 7.6419656650229815
;;;
;;; Method :smc with options []
;;; Chain no. 8
;;; Elapsed time: 20801.868879 ms
;;; Empirical mean: 39.0647726326017
;;; Empirical variance: 10.157757804699692
;;; Empirical std: 3.187123751080226
;;; Empirical skewness: 0.5716446752634904
;;; Empirical kurtosis: 2.422658788358564
;;;
;;; Method :smc with options []
;;; Chain no. 9
;;; Elapsed time: 21893.475913 ms
;;; Empirical mean: 42.63208007818617
;;; Empirical variance: 7.922096927267241
;;; Empirical std: 2.8146219865671553
;;; Empirical skewness: -1.1354991939804882
;;; Empirical kurtosis: 4.601390580384124
;;;
;;; Method :smc with options []
;;; Chain no. 10
;;; Elapsed time: 32246.23528 ms
;;; Empirical mean: 39.056202138411464
;;; Empirical variance: 5.693884281674632
;;; Empirical std: 2.3861861372647843
;;; Empirical skewness: 0.4100865629076165
;;; Empirical kurtosis: 3.921771291809904
;;;
;;; Method :smc with options []
;;; Chain no. 11
;;; Elapsed time: 38293.052543 ms
;;; Empirical mean: 39.03781622463787
;;; Empirical variance: 9.051868292606052
;;; Empirical std: 3.008632296011936
;;; Empirical skewness: 1.272761915802368
;;; Empirical kurtosis: 3.3874818261135187
;;;
;;; Method :smc with options []
;;; Chain no. 12
;;; Elapsed time: 19827.009627 ms
;;; Empirical mean: 40.078827014926496
;;; Empirical variance: 2.5518558342182702
;;; Empirical std: 1.5974529208143413
;;; Empirical skewness: -0.06684261865340166
;;; Empirical kurtosis: 3.954020246584598
;;;
;;; Method :smc with options []
;;; Chain no. 13
;;; Elapsed time: 18629.625833 ms
;;; Empirical mean: 39.663391451505234
;;; Empirical variance: 7.9239087622455155
;;; Empirical std: 2.814943829323334
;;; Empirical skewness: 0.9032461272239821
;;; Empirical kurtosis: 3.8390379050737464
;;;
;;; Method :smc with options []
;;; Chain no. 14
;;; Elapsed time: 18373.531984 ms
;;; Empirical mean: 38.955237719014704
;;; Empirical variance: 1.5893526167139054
;;; Empirical std: 1.2606952909858533
;;; Empirical skewness: 2.749882936545169
;;; Empirical kurtosis: 24.65440603911476
;;;
;;; Method :smc with options []
;;; Chain no. 15
;;; Elapsed time: 19598.386034 ms
;;; Empirical mean: 39.907282213670655
;;; Empirical variance: 5.783968636976427
;;; Empirical std: 2.4049882820871344
;;; Empirical skewness: 0.08432098731181982
;;; Empirical kurtosis: 3.8270424094265687
;;;
;;; Method :smc with options []
;;; Chain no. 16
;;; Elapsed time: 20096.985495 ms
;;; Empirical mean: 39.441333172693405
;;; Empirical variance: 9.819864231318139
;;; Empirical std: 3.133666260359922
;;; Empirical skewness: 0.6522248955131807
;;; Empirical kurtosis: 2.4674225820381426
;;;
;;; Method :smc with options []
;;; Chain no. 17
;;; Elapsed time: 18448.880872 ms
;;; Empirical mean: 40.512079274187826
;;; Empirical variance: 2.1401411886059845
;;; Empirical std: 1.4629221403089039
;;; Empirical skewness: 0.6594159870620597
;;; Empirical kurtosis: 8.34273598036189
;;;
;;; Method :smc with options []
;;; Chain no. 18
;;; Elapsed time: 18514.735914 ms
;;; Empirical mean: 40.319198771521236
;;; Empirical variance: 10.509508447355689
;;; Empirical std: 3.2418372024757334
;;; Empirical skewness: -1.2866562952111964
;;; Empirical kurtosis: 4.370648058509051
;;;
;;; Method :smc with options []
;;; Chain no. 19
;;; Elapsed time: 18312.311961 ms
;;; Empirical mean: 40.36480027496413
;;; Empirical variance: 3.900819930858788
;;; Empirical std: 1.9750493489679664
;;; Empirical skewness: 0.7906217284079435
;;; Empirical kurtosis: 4.0993151779994665
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 18082.647437 ms
;;; Empirical mean: 39.98716836199377
;;; Empirical variance: 2.790873767786566
;;; Empirical std: 1.6705908439191703
;;; Empirical skewness: -0.5771626863197612
;;; Empirical kurtosis: 7.700579153331939
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 17492.842652 ms
;;; Empirical mean: 38.99969945698812
;;; Empirical variance: 7.670111201911084
;;; Empirical std: 2.769496561093926
;;; Empirical skewness: 0.4351938651705555
;;; Empirical kurtosis: 1.658285291413335
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 26209.955261 ms
;;; Empirical mean: 39.463559677814324
;;; Empirical variance: 3.57312373440852
;;; Empirical std: 1.8902708098070287
;;; Empirical skewness: 1.1649673683670687
;;; Empirical kurtosis: 6.494293261718378
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 20046.068093 ms
;;; Empirical mean: 40.94852571687922
;;; Empirical variance: 5.121278476015938
;;; Empirical std: 2.2630241881199455
;;; Empirical skewness: -0.7192804042769313
;;; Empirical kurtosis: 6.001886611360029
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 15410.285096 ms
;;; Empirical mean: 39.72598309626807
;;; Empirical variance: 4.306176174149425
;;; Empirical std: 2.0751328087978913
;;; Empirical skewness: -0.09180406717822528
;;; Empirical kurtosis: 3.455748628608656
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 15641.652723 ms
;;; Empirical mean: 41.162321406755986
;;; Empirical variance: 2.9698339606594253
;;; Empirical std: 1.7233206203894345
;;; Empirical skewness: 0.3954479746408931
;;; Empirical kurtosis: 7.539157056069677
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 23751.545081 ms
;;; Empirical mean: 39.807840901663745
;;; Empirical variance: 3.0453127726834026
;;; Empirical std: 1.7450824544082157
;;; Empirical skewness: -0.11115133785139633
;;; Empirical kurtosis: 3.9019931294298287
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 24059.721874 ms
;;; Empirical mean: 39.332400401463
;;; Empirical variance: 5.248997817762994
;;; Empirical std: 2.291069142946802
;;; Empirical skewness: 0.6542799308198624
;;; Empirical kurtosis: 5.217677948962966
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 14777.161163 ms
;;; Empirical mean: 39.93123130000364
;;; Empirical variance: 3.8907838125927157
;;; Empirical std: 1.9725069867031437
;;; Empirical skewness: -0.08272185555468367
;;; Empirical kurtosis: 4.4774660146903
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 15917.08936 ms
;;; Empirical mean: 39.96641120819721
;;; Empirical variance: 3.019898956596553
;;; Empirical std: 1.7377856474826097
;;; Empirical skewness: -0.4038448746588986
;;; Empirical kurtosis: 3.5495775570738184
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 16002.035599 ms
;;; Empirical mean: 38.66221925658791
;;; Empirical variance: 4.029077662918335
;;; Empirical std: 2.0072562524297526
;;; Empirical skewness: 0.478684592825881
;;; Empirical kurtosis: 3.768618213121951
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 15042.173741 ms
;;; Empirical mean: 40.32094492171364
;;; Empirical variance: 4.484976471724117
;;; Empirical std: 2.117776303513692
;;; Empirical skewness: 0.41820485708543487
;;; Empirical kurtosis: 4.542401418722018
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 15300.445993 ms
;;; Empirical mean: 40.32284056718487
;;; Empirical variance: 3.0402910163731782
;;; Empirical std: 1.7436430300876318
;;; Empirical skewness: 1.5153725074706774
;;; Empirical kurtosis: 8.309749852031976
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 14790.259669 ms
;;; Empirical mean: 39.403182087647394
;;; Empirical variance: 7.398256816751176
;;; Empirical std: 2.7199736794225005
;;; Empirical skewness: 0.2871018868664153
;;; Empirical kurtosis: 2.6769449782284958
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 14380.213943 ms
;;; Empirical mean: 38.05405500103303
;;; Empirical variance: 6.142263181058828
;;; Empirical std: 2.4783589693704235
;;; Empirical skewness: 0.720877896850809
;;; Empirical kurtosis: 2.2790201857145824
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 14371.682213 ms
;;; Empirical mean: 38.81147439494706
;;; Empirical variance: 3.296726423255649
;;; Empirical std: 1.8156889665511682
;;; Empirical skewness: 0.3113558632534145
;;; Empirical kurtosis: 3.9910933716168704
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 16628.895832 ms
;;; Empirical mean: 39.29795056631778
;;; Empirical variance: 7.414550999450569
;;; Empirical std: 2.7229673151638396
;;; Empirical skewness: 0.7897449575265574
;;; Empirical kurtosis: 2.9031815246215302
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 14527.395225 ms
;;; Empirical mean: 39.32397725404681
;;; Empirical variance: 3.4937617490298636
;;; Empirical std: 1.8691607071169305
;;; Empirical skewness: 0.3965349125200681
;;; Empirical kurtosis: 3.5634721836395706
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 15993.945209 ms
;;; Empirical mean: 38.44741406615047
;;; Empirical variance: 9.418600338488586
;;; Empirical std: 3.0689738249924168
;;; Empirical skewness: 0.8590223440120534
;;; Empirical kurtosis: 2.24935189352913
;;;
;;; Method :smc with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 15397.693458 ms
;;; Empirical mean: 40.25157176728038
;;; Empirical variance: 4.999642273216195
;;; Empirical std: 2.2359879859284115
;;; Empirical skewness: 0.21128262952454036
;;; Empirical kurtosis: 2.493622523355892
;;;
;;; Method :pgibbs with options []
;;; Chain no. 0
;;; Elapsed time: 17722.185047 ms
;;; Empirical mean: 58.34039999999998
;;; Empirical variance: 962.9097278399995
;;; Empirical std: 31.03078677442774
;;; Empirical skewness: -0.04161625325394701
;;; Empirical kurtosis: 1.9212583178737597
;;;
;;; Method :pgibbs with options []
;;; Chain no. 1
;;; Elapsed time: 17017.396305 ms
;;; Empirical mean: 53.556599999999996
;;; Empirical variance: 990.2339964399999
;;; Empirical std: 31.46798367293335
;;; Empirical skewness: 0.08900599641855579
;;; Empirical kurtosis: 1.8598706600973363
;;;
;;; Method :pgibbs with options []
;;; Chain no. 2
;;; Elapsed time: 19915.942901 ms
;;; Empirical mean: 53.506199999999986
;;; Empirical variance: 973.7639615599999
;;; Empirical std: 31.20519125978881
;;; Empirical skewness: 0.07194905671631853
;;; Empirical kurtosis: 1.902744350766951
;;;
;;; Method :pgibbs with options []
;;; Chain no. 3
;;; Elapsed time: 18332.442621 ms
;;; Empirical mean: 57.45400000000003
;;; Empirical variance: 951.9634840000009
;;; Empirical std: 30.853905490229288
;;; Empirical skewness: -0.1698197627718876
;;; Empirical kurtosis: 1.9671062569826547
;;;
;;; Method :pgibbs with options []
;;; Chain no. 4
;;; Elapsed time: 17024.864535 ms
;;; Empirical mean: 59.14839999999999
;;; Empirical variance: 1097.3955774400006
;;; Empirical std: 33.12696148818966
;;; Empirical skewness: -0.09884267370189796
;;; Empirical kurtosis: 1.7271821496873465
;;;
;;; Method :pgibbs with options []
;;; Chain no. 5
;;; Elapsed time: 16740.807248 ms
;;; Empirical mean: 54.2868
;;; Empirical variance: 1013.6261457599998
;;; Empirical std: 31.837495909069226
;;; Empirical skewness: -0.016059099972572703
;;; Empirical kurtosis: 1.834585714981016
;;;
;;; Method :pgibbs with options []
;;; Chain no. 6
;;; Elapsed time: 17863.30004 ms
;;; Empirical mean: 53.77600000000002
;;; Empirical variance: 978.299424
;;; Empirical std: 31.277778437734355
;;; Empirical skewness: -0.020260704158589226
;;; Empirical kurtosis: 1.8779751948471775
;;;
;;; Method :pgibbs with options []
;;; Chain no. 7
;;; Elapsed time: 17732.937813 ms
;;; Empirical mean: 59.67219999999999
;;; Empirical variance: 831.2735471600004
;;; Empirical std: 28.831814843328896
;;; Empirical skewness: -0.13007883804246623
;;; Empirical kurtosis: 1.9300010331649804
;;;
;;; Method :pgibbs with options []
;;; Chain no. 8
;;; Elapsed time: 17002.012251 ms
;;; Empirical mean: 60.666799999999995
;;; Empirical variance: 988.3309777600007
;;; Empirical std: 31.437731752783957
;;; Empirical skewness: -0.25484127586428124
;;; Empirical kurtosis: 1.9691664150656067
;;;
;;; Method :pgibbs with options []
;;; Chain no. 9
;;; Elapsed time: 16915.734508 ms
;;; Empirical mean: 56.54419999999998
;;; Empirical variance: 926.0632463600001
;;; Empirical std: 30.43128729383626
;;; Empirical skewness: -0.006169660135061634
;;; Empirical kurtosis: 1.9261613867157636
;;;
;;; Method :pgibbs with options []
;;; Chain no. 10
;;; Elapsed time: 16777.841936 ms
;;; Empirical mean: 59.34579999999999
;;; Empirical variance: 1130.64902236
;;; Empirical std: 33.62512486757484
;;; Empirical skewness: -0.22998617234762497
;;; Empirical kurtosis: 1.7491431438706715
;;;
;;; Method :pgibbs with options []
;;; Chain no. 11
;;; Elapsed time: 16855.321622 ms
;;; Empirical mean: 52.14559999999997
;;; Empirical variance: 1110.9432006399995
;;; Empirical std: 33.33081458110497
;;; Empirical skewness: 0.10558032728317479
;;; Empirical kurtosis: 1.7687313387299797
;;;
;;; Method :pgibbs with options []
;;; Chain no. 12
;;; Elapsed time: 16988.418032 ms
;;; Empirical mean: 56.47859999999997
;;; Empirical variance: 1169.0727420399987
;;; Empirical std: 34.19170574920179
;;; Empirical skewness: -0.06356534800933539
;;; Empirical kurtosis: 1.6700366459300142
;;;
;;; Method :pgibbs with options []
;;; Chain no. 13
;;; Elapsed time: 16760.635561 ms
;;; Empirical mean: 56.79919999999993
;;; Empirical variance: 1014.6660793599995
;;; Empirical std: 31.853823622290612
;;; Empirical skewness: -0.13981178337605626
;;; Empirical kurtosis: 1.9171830682306132
;;;
;;; Method :pgibbs with options []
;;; Chain no. 14
;;; Elapsed time: 16725.913531 ms
;;; Empirical mean: 56.03569999999997
;;; Empirical variance: 978.9256255100003
;;; Empirical std: 31.287787162245916
;;; Empirical skewness: -0.11047990897229751
;;; Empirical kurtosis: 1.998164164286568
;;;
;;; Method :pgibbs with options []
;;; Chain no. 15
;;; Elapsed time: 16748.196049 ms
;;; Empirical mean: 58.08719999999993
;;; Empirical variance: 993.3291961599995
;;; Empirical std: 31.517125442527266
;;; Empirical skewness: -0.10033063315056752
;;; Empirical kurtosis: 1.8201255457685896
;;;
;;; Method :pgibbs with options []
;;; Chain no. 16
;;; Elapsed time: 16999.525326 ms
;;; Empirical mean: 57.149199999999986
;;; Empirical variance: 928.3165393600001
;;; Empirical std: 30.46828743726828
;;; Empirical skewness: -0.14792214602831855
;;; Empirical kurtosis: 1.9771993755029413
;;;
;;; Method :pgibbs with options []
;;; Chain no. 17
;;; Elapsed time: 19805.801944 ms
;;; Empirical mean: 55.55440000000001
;;; Empirical variance: 987.0830406400005
;;; Empirical std: 31.41787772336
;;; Empirical skewness: -0.03578208987481512
;;; Empirical kurtosis: 1.8780554643302854
;;;
;;; Method :pgibbs with options []
;;; Chain no. 18
;;; Elapsed time: 21148.392171 ms
;;; Empirical mean: 55.75279999999999
;;; Empirical variance: 1133.9496921600007
;;; Empirical std: 33.67416950958109
;;; Empirical skewness: -0.08520630713898482
;;; Empirical kurtosis: 1.774103324228866
;;;
;;; Method :pgibbs with options []
;;; Chain no. 19
;;; Elapsed time: 18197.538805 ms
;;; Empirical mean: 55.9922
;;; Empirical variance: 953.8193391600003
;;; Empirical std: 30.883965729161147
;;; Empirical skewness: -0.09953046096311097
;;; Empirical kurtosis: 1.9808494831183183
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 0
;;; Elapsed time: 20485.680175 ms
;;; Empirical mean: 41.241600000000005
;;; Empirical variance: 5.2256294400000165
;;; Empirical std: 2.285963569263521
;;; Empirical skewness: 0.24946104190831359
;;; Empirical kurtosis: 3.256701921595067
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 1
;;; Elapsed time: 26550.634525 ms
;;; Empirical mean: 38.416399999999996
;;; Empirical variance: 8.052811040000002
;;; Empirical std: 2.837747529291498
;;; Empirical skewness: 0.8152965148073965
;;; Empirical kurtosis: 3.6583940917476196
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 2
;;; Elapsed time: 14686.650247 ms
;;; Empirical mean: 39.80749999999999
;;; Empirical variance: 7.254843750000016
;;; Empirical std: 2.6934817151783332
;;; Empirical skewness: 0.23409011769021904
;;; Empirical kurtosis: 4.203141302318431
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 3
;;; Elapsed time: 14505.131766 ms
;;; Empirical mean: 38.1354
;;; Empirical variance: 7.717466840000013
;;; Empirical std: 2.7780329083723996
;;; Empirical skewness: 0.7612386672108271
;;; Empirical kurtosis: 1.9904362249757555
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 4
;;; Elapsed time: 14586.391245 ms
;;; Empirical mean: 38.54030000000001
;;; Empirical variance: 11.087575910000037
;;; Empirical std: 3.3298011817524538
;;; Empirical skewness: 1.5427576171515924
;;; Empirical kurtosis: 4.009168398951378
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 5
;;; Elapsed time: 14596.757412 ms
;;; Empirical mean: 39.61189999999999
;;; Empirical variance: 6.957478390000026
;;; Empirical std: 2.6377032414583765
;;; Empirical skewness: 1.2339712519202795
;;; Empirical kurtosis: 4.493980309638495
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 6
;;; Elapsed time: 14549.325772 ms
;;; Empirical mean: 40.391
;;; Empirical variance: 1.0677190000000052
;;; Empirical std: 1.0333048920817152
;;; Empirical skewness: -1.1340835421103606
;;; Empirical kurtosis: 21.44404015159923
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 7
;;; Elapsed time: 14493.109154 ms
;;; Empirical mean: 40.2312
;;; Empirical variance: 5.335346560000004
;;; Empirical std: 2.309836911991841
;;; Empirical skewness: -1.3492414047648564
;;; Empirical kurtosis: 3.8264185193703906
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 8
;;; Elapsed time: 14740.499082 ms
;;; Empirical mean: 40.19599999999999
;;; Empirical variance: 3.779984
;;; Empirical std: 1.9442180947620047
;;; Empirical skewness: 1.4660326281292027
;;; Empirical kurtosis: 7.063020359198156
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 9
;;; Elapsed time: 14606.479686 ms
;;; Empirical mean: 40.20929999999999
;;; Empirical variance: 2.2436935100000097
;;; Empirical std: 1.497896361568453
;;; Empirical skewness: -0.662958332935576
;;; Empirical kurtosis: 5.161854180981403
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 10
;;; Elapsed time: 14749.133267 ms
;;; Empirical mean: 39.7477
;;; Empirical variance: 9.320444710000025
;;; Empirical std: 3.0529403384278613
;;; Empirical skewness: 0.6208382646843488
;;; Empirical kurtosis: 3.1788513003630885
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 11
;;; Elapsed time: 14655.85486 ms
;;; Empirical mean: 41.7269
;;; Empirical variance: 2.6985163900000146
;;; Empirical std: 1.6427161623360302
;;; Empirical skewness: 0.34604084853044315
;;; Empirical kurtosis: 7.8350070033852095
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 12
;;; Elapsed time: 14581.34961 ms
;;; Empirical mean: 37.116000000000014
;;; Empirical variance: 4.716544000000019
;;; Empirical std: 2.17176057612252
;;; Empirical skewness: 1.9764598235000113
;;; Empirical kurtosis: 6.760597466540587
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 13
;;; Elapsed time: 14648.515887 ms
;;; Empirical mean: 38.58750000000001
;;; Empirical variance: 7.626743750000004
;;; Empirical std: 2.7616559796614792
;;; Empirical skewness: 0.5579307743389683
;;; Empirical kurtosis: 2.0213601867908695
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 14
;;; Elapsed time: 15649.075382 ms
;;; Empirical mean: 37.967300000000016
;;; Empirical variance: 5.958830710000008
;;; Empirical std: 2.4410716314766363
;;; Empirical skewness: 0.5849811415121249
;;; Empirical kurtosis: 2.0918824603650803
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 15
;;; Elapsed time: 15566.724404 ms
;;; Empirical mean: 38.80499999999999
;;; Empirical variance: 3.334375000000011
;;; Empirical std: 1.8260271082325177
;;; Empirical skewness: 0.9609380516165192
;;; Empirical kurtosis: 10.395976841473212
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 16
;;; Elapsed time: 14809.123238 ms
;;; Empirical mean: 39.28750000000001
;;; Empirical variance: 11.58644375
;;; Empirical std: 3.4038865653837527
;;; Empirical skewness: 0.39399711317286545
;;; Empirical kurtosis: 1.6329886005993477
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 17
;;; Elapsed time: 15802.714545 ms
;;; Empirical mean: 40.0047
;;; Empirical variance: 3.2682779100000072
;;; Empirical std: 1.8078379103227167
;;; Empirical skewness: 0.04610584875142828
;;; Empirical kurtosis: 6.4758540973631
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 18
;;; Elapsed time: 15348.430028 ms
;;; Empirical mean: 39.879699999999985
;;; Empirical variance: 5.15362791000001
;;; Empirical std: 2.270160326937287
;;; Empirical skewness: 0.21843427881450675
;;; Empirical kurtosis: 4.9506297642718655
;;;
;;; Method :pgibbs with options [:number-of-particles 1000]
;;; Chain no. 19
;;; Elapsed time: 15426.608923 ms
;;; Empirical mean: 40.534
;;; Empirical variance: 3.088244
;;; Empirical std: 1.7573400353944026
;;; Empirical skewness: -0.44735540328752665
;;; Empirical kurtosis: 3.1610120000212496
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 0
;;; Elapsed time: 4948.350475 ms
;;; Empirical mean: 62.81469655524067
;;; Empirical variance: 819.5455281361592
;;; Empirical std: 28.62770560377061
;;; Empirical skewness: -0.9361657312101125
;;; Empirical kurtosis: 2.7410401852702253
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 1
;;; Elapsed time: 4729.267515 ms
;;; Empirical mean: 62.90774762652779
;;; Empirical variance: 677.5166477883248
;;; Empirical std: 26.029149962845977
;;; Empirical skewness: 0.21099373266619925
;;; Empirical kurtosis: 1.8637406496793227
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 2
;;; Elapsed time: 4562.366333 ms
;;; Empirical mean: 62.10303514378131
;;; Empirical variance: 1102.0261249722657
;;; Empirical std: 33.19677883428249
;;; Empirical skewness: -0.07411659146355347
;;; Empirical kurtosis: 1.8009661000212898
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 3
;;; Elapsed time: 4491.652005 ms
;;; Empirical mean: 62.277337449996075
;;; Empirical variance: 937.4033492733527
;;; Empirical std: 30.617043444352245
;;; Empirical skewness: -0.4076475497332284
;;; Empirical kurtosis: 1.850517542645437
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 4
;;; Elapsed time: 4635.002682 ms
;;; Empirical mean: 61.809987311574204
;;; Empirical variance: 940.2949515647873
;;; Empirical std: 30.664229185889987
;;; Empirical skewness: -0.10168859964493125
;;; Empirical kurtosis: 1.8381603225786294
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 5
;;; Elapsed time: 4678.042298 ms
;;; Empirical mean: 47.78556679773096
;;; Empirical variance: 1021.890764576717
;;; Empirical std: 31.967026207902368
;;; Empirical skewness: 0.5720439801444412
;;; Empirical kurtosis: 2.1529018990058106
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 6
;;; Elapsed time: 4558.236387 ms
;;; Empirical mean: 64.56316969909933
;;; Empirical variance: 1031.9078035802365
;;; Empirical std: 32.12332180177256
;;; Empirical skewness: -0.34993821082522947
;;; Empirical kurtosis: 1.6266946952522692
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 7
;;; Elapsed time: 4778.016876 ms
;;; Empirical mean: 60.41902831437678
;;; Empirical variance: 610.1537048389246
;;; Empirical std: 24.70128953797604
;;; Empirical skewness: -0.5162526886356815
;;; Empirical kurtosis: 3.1175309049183384
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 8
;;; Elapsed time: 4698.888347 ms
;;; Empirical mean: 48.770305765919986
;;; Empirical variance: 999.8541536816582
;;; Empirical std: 31.620470484824512
;;; Empirical skewness: 0.06553890077527705
;;; Empirical kurtosis: 1.9414656351261097
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 9
;;; Elapsed time: 4579.112963 ms
;;; Empirical mean: 55.090254366922004
;;; Empirical variance: 870.9610735385301
;;; Empirical std: 29.51204963296399
;;; Empirical skewness: 0.5036042471228404
;;; Empirical kurtosis: 2.0689652395935094
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 10
;;; Elapsed time: 4589.617765 ms
;;; Empirical mean: 53.76277955274574
;;; Empirical variance: 1420.9716817284013
;;; Empirical std: 37.695778035854374
;;; Empirical skewness: -0.01464307711047492
;;; Empirical kurtosis: 1.582975808206542
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 11
;;; Elapsed time: 4602.559288 ms
;;; Empirical mean: 70.70303377125195
;;; Empirical variance: 997.8258920163929
;;; Empirical std: 31.588382231706532
;;; Empirical skewness: -0.5337369408655319
;;; Empirical kurtosis: 1.8491548593618048
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 12
;;; Elapsed time: 4594.235793 ms
;;; Empirical mean: 56.901757188498415
;;; Empirical variance: 1041.6764505864096
;;; Empirical std: 32.27501278987213
;;; Empirical skewness: -0.07938161998976126
;;; Empirical kurtosis: 1.9170939357043446
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 13
;;; Elapsed time: 4658.685081 ms
;;; Empirical mean: 70.32920495397518
;;; Empirical variance: 816.1561067196965
;;; Empirical std: 28.56844599763341
;;; Empirical skewness: -0.4197656883435361
;;; Empirical kurtosis: 2.228732128289645
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 14
;;; Elapsed time: 4746.590376 ms
;;; Empirical mean: 74.30515021252205
;;; Empirical variance: 975.3223369433068
;;; Empirical std: 31.23015108742362
;;; Empirical skewness: -0.6959956980535527
;;; Empirical kurtosis: 2.068137462988175
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 15
;;; Elapsed time: 4869.700994 ms
;;; Empirical mean: 53.68330633202536
;;; Empirical variance: 963.7124030434262
;;; Empirical std: 31.043717609903396
;;; Empirical skewness: -0.05648826897299967
;;; Empirical kurtosis: 1.9820109214893666
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 16
;;; Elapsed time: 4698.342669 ms
;;; Empirical mean: 64.35391119611057
;;; Empirical variance: 720.0161013768546
;;; Empirical std: 26.833115759763245
;;; Empirical skewness: -0.6912216611585816
;;; Empirical kurtosis: 2.8235401086631486
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 17
;;; Elapsed time: 5048.0572 ms
;;; Empirical mean: 51.40168749925459
;;; Empirical variance: 1082.6187581992901
;;; Empirical std: 32.90317246405413
;;; Empirical skewness: 0.3098172851494974
;;; Empirical kurtosis: 1.7495767534576812
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 18
;;; Elapsed time: 4912.358551 ms
;;; Empirical mean: 67.40287769783396
;;; Empirical variance: 702.7033969944068
;;; Empirical std: 26.508553279920932
;;; Empirical skewness: -0.6092447392197647
;;; Empirical kurtosis: 2.2557737764006616
;;;
;;; Method :ipmcmc with options []
;;; Chain no. 19
;;; Elapsed time: 4899.019366 ms
;;; Empirical mean: 68.46509605674912
;;; Empirical variance: 1226.2734737741005
;;; Empirical std: 35.01818775685145
;;; Empirical skewness: -0.37663707263167523
;;; Empirical kurtosis: 1.5828851395590717
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 0
;;; Elapsed time: 12355.412936 ms
;;; Empirical mean: 40.138465862744006
;;; Empirical variance: 5.569386976874971
;;; Empirical std: 2.3599548675504307
;;; Empirical skewness: 0.1777707111068216
;;; Empirical kurtosis: 2.7311474859041374
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 1
;;; Elapsed time: 14427.662158 ms
;;; Empirical mean: 40.06399234707678
;;; Empirical variance: 5.404310927556098
;;; Empirical std: 2.324717386599089
;;; Empirical skewness: -0.704267758739215
;;; Empirical kurtosis: 5.075216929262515
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 2
;;; Elapsed time: 10911.566481 ms
;;; Empirical mean: 39.408744635642435
;;; Empirical variance: 8.736432992811544
;;; Empirical std: 2.955745759163251
;;; Empirical skewness: 0.6330656121126763
;;; Empirical kurtosis: 2.7721285648142704
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 3
;;; Elapsed time: 11020.443814 ms
;;; Empirical mean: 39.32858305874532
;;; Empirical variance: 11.854302967925268
;;; Empirical std: 3.443007837331374
;;; Empirical skewness: 1.0252306870412924
;;; Empirical kurtosis: 2.672443030511833
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 4
;;; Elapsed time: 11029.728097 ms
;;; Empirical mean: 39.29811218215221
;;; Empirical variance: 8.691642707498524
;;; Empirical std: 2.9481592066064755
;;; Empirical skewness: 0.5380733371607942
;;; Empirical kurtosis: 2.3367381652258477
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 5
;;; Elapsed time: 11557.051782 ms
;;; Empirical mean: 40.42964337017816
;;; Empirical variance: 7.345227875533502
;;; Empirical std: 2.710208087127906
;;; Empirical skewness: -0.09355659971071673
;;; Empirical kurtosis: 3.0348872814756516
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 6
;;; Elapsed time: 11088.950381 ms
;;; Empirical mean: 39.92132691832981
;;; Empirical variance: 6.228290068403623
;;; Empirical std: 2.4956542365487295
;;; Empirical skewness: -0.4889711362853994
;;; Empirical kurtosis: 2.741894495398141
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 7
;;; Elapsed time: 11034.708266 ms
;;; Empirical mean: 39.01273152027971
;;; Empirical variance: 11.143462118226841
;;; Empirical std: 3.338182457300206
;;; Empirical skewness: 0.4520957001256084
;;; Empirical kurtosis: 3.273478775738403
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 8
;;; Elapsed time: 11624.173982 ms
;;; Empirical mean: 38.99521811132949
;;; Empirical variance: 5.918760951159501
;;; Empirical std: 2.4328503758265736
;;; Empirical skewness: 0.2931783148288853
;;; Empirical kurtosis: 1.8321167423477038
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 9
;;; Elapsed time: 11272.982008 ms
;;; Empirical mean: 39.2678098023895
;;; Empirical variance: 10.31460007779991
;;; Empirical std: 3.211635109690998
;;; Empirical skewness: 0.20210816074259433
;;; Empirical kurtosis: 3.9675556686386786
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 10
;;; Elapsed time: 11552.091733 ms
;;; Empirical mean: 41.53605669852851
;;; Empirical variance: 5.168566828284297
;;; Empirical std: 2.2734482242365446
;;; Empirical skewness: 0.1272478249090145
;;; Empirical kurtosis: 4.79552081900075
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 11
;;; Elapsed time: 12738.78644 ms
;;; Empirical mean: 39.991489256987805
;;; Empirical variance: 6.368114890432693
;;; Empirical std: 2.5235124113886767
;;; Empirical skewness: -0.38493903362763
;;; Empirical kurtosis: 3.942704686820115
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 12
;;; Elapsed time: 11715.67951 ms
;;; Empirical mean: 41.28649830357435
;;; Empirical variance: 8.910645485921771
;;; Empirical std: 2.9850704323217854
;;; Empirical skewness: -1.1922917907720811
;;; Empirical kurtosis: 3.165061460959061
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 13
;;; Elapsed time: 10855.70353 ms
;;; Empirical mean: 41.24613040537414
;;; Empirical variance: 2.773264912993789
;;; Empirical std: 1.6653122569037284
;;; Empirical skewness: -1.1773643416726693
;;; Empirical kurtosis: 4.623444798915256
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 14
;;; Elapsed time: 10242.626484 ms
;;; Empirical mean: 41.10538182557373
;;; Empirical variance: 10.31789649436212
;;; Empirical std: 3.2121482678049156
;;; Empirical skewness: -0.3061391412989578
;;; Empirical kurtosis: 2.451345582868551
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 15
;;; Elapsed time: 10579.035578 ms
;;; Empirical mean: 40.275882650772836
;;; Empirical variance: 10.709045520036687
;;; Empirical std: 3.2724678027501946
;;; Empirical skewness: 0.36214512640096647
;;; Empirical kurtosis: 1.8896222365272488
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 16
;;; Elapsed time: 12469.829357 ms
;;; Empirical mean: 40.898543339044146
;;; Empirical variance: 9.97096543759481
;;; Empirical std: 3.157683555645627
;;; Empirical skewness: -0.6813476080803276
;;; Empirical kurtosis: 4.222109525524921
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 17
;;; Elapsed time: 13046.903689 ms
;;; Empirical mean: 40.47467322688145
;;; Empirical variance: 9.543457910586008
;;; Empirical std: 3.0892487615253663
;;; Empirical skewness: -0.10475986245729159
;;; Empirical kurtosis: 2.5345998657403537
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 18
;;; Elapsed time: 10569.43839 ms
;;; Empirical mean: 39.669904149213245
;;; Empirical variance: 5.065666039465791
;;; Empirical std: 2.2507034543595013
;;; Empirical skewness: 0.8372227895036238
;;; Empirical kurtosis: 3.4193659070195372
;;;
;;; Method :ipmcmc with options [:number-of-particles 1000 :number-of-nodes 32]
;;; Chain no. 19
;;; Elapsed time: 11564.386676 ms
;;; Empirical mean: 40.85037864518371
;;; Empirical variance: 8.770407907373814
;;; Empirical std: 2.9614874484579223
;;; Empirical skewness: -0.7725871026512924
;;; Empirical kurtosis: 7.029195449221268
;;;
;; <-
;; =>
;;; {"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"list-like","open":"","close":"","separator":"</pre><pre>","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0xbeff340]</span>","value":"#multifn[print-method 0xbeff340]"}],"value":"[nil,#multifn[print-method 0xbeff340]]"},{"type":"html","content":"<span class='clj-unkown'>#multifn[print-method 0xbeff340]</span>","value":"#multifn[print-method 0xbeff340]"}],"value":"[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/data</span>","value":"#'model/data"}],"value":"[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model</span>","value":"#'model/model"}],"value":"[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/model_name</span>","value":"#'model/model_name"}],"value":"[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/outfile</span>","value":"#'model/outfile"}],"value":"[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/configurations</span>","value":"#'model/configurations"}],"value":"[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num_samples_options</span>","value":"#'model/num_samples_options"}],"value":"[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/thinning</span>","value":"#'model/thinning"}],"value":"[[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil]"},{"type":"html","content":"<span class='clj-var'>#&#x27;model/num-chains</span>","value":"#'model/num-chains"}],"value":"[[[[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains]"},{"type":"list-like","open":"<span class='clj-lazy-seq'>(</span>","close":"<span class='clj-lazy-seq'>)</span>","separator":" ","items":[{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)"}],"value":"[[[[[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)]"},{"type":"html","content":"<span class='clj-nil'>nil</span>","value":"nil"}],"value":"[[[[[[[[[[[[[nil,#multifn[print-method 0xbeff340]],#multifn[print-method 0xbeff340]],#'model/data],#'model/model],#'model/model_name],#'model/outfile],#'model/configurations],#'model/num_samples_options],#'model/thinning],nil],#'model/num-chains],(nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil nil)],nil]"}
;; <=

;; @@

;; @@
