import numpy as np
from PIL import Image
import cv2

from tqdm import tqdm
from collections.abc import Iterable

from interpretdl.data_processor.readers import images_transform_pipeline, preprocess_save_path
from interpretdl.data_processor.visualizer import explanation_to_vis, show_vis_explanation, save_image


class SmoothGradInterpreter():
    """
    Smooth Gradients Interpreter.

    For input gradient based interpreters, the target issue is generally the vanilla input gradient's noises.
    The basic idea of reducing the noises is to use different similar inputs to get the input gradients and 
    do the average. 

    Smooth Gradients method solves the problem of meaningless local variations in partial derivatives
    by adding random noise to the inputs multiple times and take the average of the gradients.

    More details regarding the Smooth Gradients method can be found in the original paper:
    http://arxiv.org/abs/1706.03825.
    """

    def __init__(self, model: callable, device: str = 'cuda', **kwargs):
        """

        Args:
            model (callable): A model with :py:func:`forward` and possibly :py:func:`backward` functions.
            device (str): The device used for running ``model``, options: ``"cpu"``, ``"gpu:0"``, ``"gpu:1"`` 
                etc.
        """
        self.device = device
        self.model = model
        self.predict_fn = None

    def interpret(self,
                  path: str or list(str) or np.ndarray,
                  processor: callable,
                  labels: list or np.ndarray = None,
                  noise_amount: int = 0.1,
                  n_samples: int = 50,
                  gradient_of: str = 'probability',
                  resize_to: int = 224,
                  crop_to: int = None,
                  visual: bool = True,
                  save_path: str = None) -> np.ndarray:
        """The technical details of the SmoothGrad method are described as follows:
        SmoothGrad generates ``n_samples`` noised inputs, with the noise scale of ``noise_amount``, and then computes 
        the gradients *w.r.t.* these noised inputs. The final explanation is averaged gradients.

        Args:
            inputs (str or list): The input image filepath or a list of filepaths or numpy array of read images.
            labels (list or np.ndarray, optional): The target labels to analyze. The number of labels should be equal 
                to the number of images. If None, the most likely label for each image will be used. Default: None.
            noise_amount (int, optional): Noise level of added noise to the image. The std of Gaussian random noise 
                is ``noise_amount`` * (x :sub:`max` - x :sub:`min`). Default: ``0.1``.
            n_samples (int, optional): The number of new images generated by adding noise. Default: ``50``.
            gradient_of (str, optional): compute the gradient of ['probability', 'logit' or 'loss']. Default: 
                ``'probability'``. SmoothGrad uses probability for all tasks by default.
            resize_to (int, optional): Images will be rescaled with the shorter edge being ``resize_to``. Defaults to 
                ``224``.
            crop_to (int, optional): After resize, images will be center cropped to a square image with the size 
                ``crop_to``. If None, no crop will be performed. Defaults to ``None``.
            visual (bool, optional): Whether or not to visualize the processed image. Default: ``True``.
            save_path (str, optional): The filepath(s) to save the processed image(s). If None, the image will not be 
                saved. Default: ``None``.

        Returns:
            np.ndarray: the explanation result.
        """

        # previous for paddle
        # imgs, data = images_transform_pipeline(path, resize_to, crop_to)
        
        # now for hf.        
        image = Image.open(path).convert('RGB')
        inputs = processor(images=image, return_tensors="pt")        
        data = inputs['pixel_values'].numpy()
        
        # for visualization.
        imgs = cv2.resize(np.array(image), (224, 224))
        imgs = np.expand_dims(imgs, axis=0)
        
        # print(imgs.shape, data.shape, imgs.dtype, data.dtype)  # (1, 224, 224, 3) (1, 3, 224, 224) uint8 float32

        bsz = len(data)

        self._build_predict_fn_hf(gradient_of=gradient_of)

        # obtain the labels (and initialization).
        _, predicted_label, predicted_proba = self.predict_fn(
            data, labels
        )
        self.predicted_label = predicted_label
        self.predicted_proba = predicted_proba
        if labels is None:
            labels = predicted_label
            
        labels = np.array(labels).reshape((bsz, ))

        # SmoothGrad
        max_axis = tuple(np.arange(1, data.ndim))
        stds = noise_amount * (np.max(data, axis=max_axis) - np.min(data, axis=max_axis))

        total_gradients = np.zeros_like(data)
        for i in tqdm(range(n_samples), leave=True, position=0):
            noise = np.concatenate(
                [np.float32(np.random.normal(0.0, stds[j], (1, ) + tuple(d.shape))) for j, d in enumerate(data)])
            _noised_data = data + noise
            gradients, _, _ = self.predict_fn(_noised_data, labels)
            total_gradients += gradients

        avg_gradients = total_gradients / n_samples

        # visualize and save image.
        if save_path is None and not visual:
            # no need to visualize or save explanation results.
            pass
        else:
            save_path = preprocess_save_path(save_path, bsz)
            for i in range(bsz):
                # print(imgs[i].shape, avg_gradients[i].shape)
                vis_explanation = explanation_to_vis(imgs[i],
                                                     np.abs(avg_gradients[i]).sum(0),
                                                     style='overlay_grayscale')
                if visual:
                    show_vis_explanation(vis_explanation)
                if save_path[i] is not None:
                    save_image(save_path[i], vis_explanation)

        # intermediate results, for possible further usages.
        self.labels = labels

        return avg_gradients

    def _build_predict_fn_hf(self, rebuild: bool = False, gradient_of: str = 'probability'):
        if self.predict_fn is not None:
            assert callable(self.predict_fn), "predict_fn is predefined before, but is not callable." \
                "Check it again."

        if self.predict_fn is None or rebuild:
            assert gradient_of in ['loss', 'logit', 'probability']

            import torch
            if 'cuda' in self.device:
                if torch.cuda.is_available():
                    self.model.cuda().eval()
                else:
                    print("PyTorch is not compiled with CUDA support. Change to CPU version now.")
                    self.device = 'cpu'
                    self.model.eval()
            else:
                # default to cpu.
                pass

            def predict_fn_hf(inputs, labels=None):
                assert labels is None or \
                    (isinstance(labels, (list, np.ndarray)) and len(labels) == inputs.shape[0])

                import torch
                import torch.nn.functional as F
                
                inputs = torch.from_numpy(inputs).cuda()

                inputs.requires_grad_(True)
                # get logits and probas, [bs, num_c]
                logits = self.model(**{'pixel_values': inputs}).logits
                num_samples, num_classes = logits.shape[0], logits.shape[1]
                probas = F.softmax(logits, dim=-1)

                # get predictions.
                pred = torch.argmax(logits, dim=1)
                if labels is None:
                    labels = pred
                
                try:
                    labels = torch.from_numpy(labels).cuda()
                except:
                    labels = labels

                # get gradients
                if gradient_of == 'loss':
                    # cross-entropy loss
                    loss = F.cross_entropy(logits, labels, reduction='sum')
                else:
                    # logits or probas
                    # labels = np.array(labels).reshape((num_samples, ))
                    labels_onehot = F.one_hot(labels, num_classes=num_classes)
                    if gradient_of == 'logit':
                        loss = torch.sum(logits * labels_onehot, dim=-1)
                    else:
                        loss = torch.sum(probas * labels_onehot, dim=1)

                # loss.backward()
                # gradients = tensor_inputs[0].grad
                gradients = torch.autograd.grad(loss, inputs)

                if isinstance(gradients, tuple):
                    gradients = gradients[0]
                if isinstance(gradients, torch.Tensor):
                    gradients = gradients.cpu().detach().numpy()

                return gradients, labels.cpu().detach().numpy(), probas.cpu().detach().numpy()

            self.predict_fn = predict_fn_hf

            
            

class IntGradInterpreter(SmoothGradInterpreter):

    def __init__(self, model: callable, device: str = 'cuda', **kwargs):
        """

        Args:
            model (callable): A model with :py:func:`forward` and possibly :py:func:`backward` functions.
            device (str): The device used for running ``model``, options: ``"cpu"``, ``"gpu:0"``, ``"gpu:1"`` 
                etc.
        """
        self.device = device
        self.model = model
        self.predict_fn = None

    def interpret(self,
                  path: str or list(str) or np.ndarray,
                  processor: callable,
                  labels: list or tuple or np.ndarray or None = None,
                  baselines: np.ndarray or None = None,
                  steps: int = 50,
                  num_random_trials: int = 10,
                  gradient_of: str = 'probability',
                  resize_to: int = 224,
                  crop_to: int = None,
                  visual: bool = True,
                  save_path: str = None) -> np.ndarray:
        
        image = Image.open(path).convert('RGB')
        inputs = processor(images=image, return_tensors="pt")        
        data = inputs['pixel_values'].numpy()
        
        # for visualization.
        imgs = cv2.resize(np.array(image), (224, 224))
        imgs = np.expand_dims(imgs, axis=0)

        bsz = len(data)
        self._build_predict_fn_hf(gradient_of=gradient_of)
        
        self.data_type = np.array(data).dtype

        if baselines is None:
            num_random_trials = 1
            self.baselines = np.zeros((num_random_trials, ) + data.shape, dtype=self.data_type)
        elif baselines == 'random':
            self.baselines = np.random.normal(size=(num_random_trials, ) + data.shape).astype(self.data_type)
        else:
            self.baselines = baselines

        # obtain the labels (and initialization).
        _, predicted_label, predicted_proba = self.predict_fn(data, labels)
        self.predicted_label = predicted_label
        self.predicted_proba = predicted_proba
        if labels is None:
            labels = predicted_label

        labels = np.array(labels).reshape((bsz, ))

        # IntGrad.
        gradients_list = []
        with tqdm(total=num_random_trials * steps, leave=True, position=0) as pbar:
            for i in range(num_random_trials):
                total_gradients = np.zeros_like(data)
                for alpha in np.linspace(0, 1, steps):
                    data_scaled = data * alpha + self.baselines[i] * (1 - alpha)
                    gradients, _, _ = self.predict_fn(data_scaled, labels)
                    total_gradients += gradients
                    pbar.update(1)

                ig_gradients = total_gradients * (data - self.baselines[i]) / steps
                gradients_list.append(ig_gradients)
        avg_gradients = np.average(np.array(gradients_list), axis=0)

        # visualization and save image.
        if save_path is None and not visual:
            # no need to visualize or save explanation results.
            pass
        else:
            save_path = preprocess_save_path(save_path, bsz)
            for i in range(bsz):
                vis_explanation = explanation_to_vis(imgs[i],
                                                     np.abs(avg_gradients[i]).sum(0),
                                                     style='overlay_grayscale')
                if visual:
                    show_vis_explanation(vis_explanation)
                if save_path[i] is not None:
                    save_image(save_path[i], vis_explanation)

        return avg_gradients
            
            
            
            
# from PIL import Image
# import torch
# image = Image.open(path).convert('RGB')
# inputs = processor(images=image, return_tensors="pt")
# inputs['pixel_values'].requires_grad_(True)
# outputs = model(**inputs)
# logits = outputs.logits