import torch
import torch.nn.functional as F
from sklearn.metrics import roc_auc_score
from copy import deepcopy
from pathlib import Path
import glob
import re
import numpy as np
import networkx as nx
import dgl
import pandas as pd
import collections

def compute_loss(pos_score, neg_score):
    scores = torch.cat([pos_score, neg_score])
    labels = torch.cat([torch.ones(pos_score.shape[0]), torch.zeros(neg_score.shape[0])]).to(pos_score.device)
    #return F.binary_cross_entropy_with_logits(scores, labels)    #only one type
    return F.binary_cross_entropy(scores, labels)

def compute_auc(pos_score, neg_score):
    scores = torch.cat([pos_score, neg_score]).cpu().numpy()
    scores = np.nan_to_num(scores, nan=0, posinf=0, neginf=0)
    
    #deal with overflow
    labels = torch.cat(
        [torch.ones(pos_score.shape[0]), torch.zeros(neg_score.shape[0])]).numpy()
    return roc_auc_score(labels, scores)

def compute_mr(predictions, test_g):
    _, indices = torch.sort(predictions, descending = True)
    hits = (test_g.edata['etype'].view(-1, 1) == indices).nonzero()[:,-1]
    return hits.float().mean()

def compute_mrr(predictions, test_g, weights = None):
    """Computing the MRR and Weighted MRR metrics for recommendations.
       MRR (Mean Reciprocal Rank) = 1/N * Sum_{i=1 to N}(1/r_i),
       where r is the rank of the ground truth rule in the graph, averaged for N test cases.
       Weighted MRR = 1/(W_1 + ... + W_N) * Sum_{i=1 to N}(W_i/r_i),
       W_i is the total available rules between the two node types for each omitted edge.

    Args:
        predictions (torch.Tensor): prediction outputs of the model for test graph test_g and the targeted nodes
        test_g (_type_): is the test graph
        weights (_type_, optional): Weights for each prediction. If not provided, it will be considered 1. Defaults to None.

    Raises:
        ValueError: if number of predictions and weights are different
        ValueError: If there are negative weights or all weights are zero

    Returns:
        _type_: _description_
    """
    if weights is None:
        weights = torch.ones(len(predictions))
    if len(weights) != len(predictions):
        raise ValueError("The number of weights ({}) and number of predictions ({}) mismatched!".format(len(weights),len(predictions)))
    if weights.sum() == 0 or (weights < 0).any():
        raise ValueError("Weights cannot be less than zero or summed to zero.")

    _, indices = torch.sort(predictions, descending = True)
    hits = (test_g.edata['etype'].view(-1, 1) == indices).nonzero()[:,-1] + 1.0
    return (weights/hits).float().sum() / sum(weights)

def zero_copy(model,rnn=False):
    if rnn:
        model.hidden = None
    tmp_model = deepcopy(model)
    for tp in tmp_model.parameters():
        tp.data = torch.zeros_like(tp.data)
    if rnn:
        model.init_hidden()
    return tmp_model


def increment_dir(dir, comment=''):
    # Increments a directory runs/exp1 --> runs/exp2_comment
    n = 0  # number
    dir = str(Path(dir))  # os-agnostic
    dirs = sorted(glob.glob(dir + '*'))  # directories
    if dirs:
        matches = [re.search(r"exp(\d+)", d) for d in dirs]
        idxs = [int(m.groups()[0]) for m in matches if m]
        if idxs:
            n = max(idxs) + 1  # increment
    return dir + str(n) + ('_' + comment if comment else '')


def weighted_state_alpha(personal_model, model, alpha):
    weighted_state = dict()
    
    personal_state = personal_model.state_dict()
    local_state = model.state_dict()
    
    for key in personal_state:
        weighted_state[key] = alpha * personal_state[key] + (1 - alpha) * local_state[key]
    return weighted_state


def inference_personal(personal_model, model, alpha, _input1, _input2):
    """Inference on the given model and get loss and accuracy."""
    # TODO: merge with inference
    output1 = personal_model(_input1, _input2)
    output2 = model(_input1, _input2)
    output = alpha * output1 + (1-alpha) * output2
    
    return output


def alpha_update(model_local, model_personal,alpha, eta):
    grad_alpha = 0
    for l_params, p_params in zip(model_local.parameters(), model_personal.parameters()):
        dif = p_params.data - l_params.data
        grad = alpha * p_params.grad.data + (1-alpha)*l_params.grad.data
        grad_alpha += dif.view(-1).T.dot(grad.view(-1))
    
    grad_alpha += 0.02 * alpha
    alpha_n = alpha - eta*grad_alpha
    alpha_n = np.clip(alpha_n.item(),0.0,1.0)
    return alpha_n



def inv_d(current_dict):
    inv_dict = dict()
    for i in current_dict:
        if current_dict[i] not in inv_dict:
            inv_dict[current_dict[i]] = i
    return inv_dict


def get_rules(edges, edata, inv_all_devices, inv_all_trigger_actions):
    L = []
    for i in range(len(edges[0])):
        L.append([inv_all_devices[int(edges[0][i])], inv_all_devices[int(edges[1][i])], inv_all_trigger_actions[int(edata[i])]])
    return L

def get_recommendations(G, model, pred, all_devices, topk = 10, args_dataset = 'ifttt'): #, all_devices, all_trigger_actions,
    
    Complete_G = nx.complete_graph(list(G.nodes()), nx.MultiDiGraph())
    if args_dataset == 'wyze':
        for node in Complete_G.nodes():
            Complete_G.add_edge(node, node)
    
    class_num = len(set(all_devices.values()))
    one_hot_matrix = F.one_hot(torch.tensor(list(range(class_num))), num_classes=class_num)

    devices = list(Complete_G.nodes())

    
    Complete_G = dgl.from_networkx(Complete_G)
    
    
    model.eval()
    pred.eval()
    with torch.no_grad():
        h = model(G, G.ndata['feat'])
        #h = model(G, G.ndata['feat'], G.edata['efeat'])
        
        ###########
        #predictor, use node embeddings of source node and target node as input, predict the link probability of current edge
        #need a complete graph as input
        scores = pred(Complete_G, h)
    L = []
    edges = Complete_G.edges()
    for i in range(scores.shape[0]):
        for j in range(scores.shape[1]):
            L.append([int(edges[0][i]), int(edges[1][i]), j, float(scores[i][j])])
    L = torch.tensor(sorted(L, key= lambda e:e[3], reverse = True))[:,:-1]
    
    
    
    Z = combine_triple_inv(G.edges()[0], G.edges()[1], G.edata['etype']).tolist()
    L = torch.stack([j for j in L if j.tolist() not in Z])
    L = L[:topk]
    
    #get_rules([L[:,0],L[:,1]], L[:,2], inv_d(all_devices), inv_d(all_trigger_actions))
    return L


#the node id in user_graph is not the device id
#get the device id by node feature
def transform_edges(edges, edata, node_transfer):
    New_edges = []
    for i in range(len(edges[0])):
        New_edges.append([node_transfer[int(edges[0][i])], node_transfer[int(edges[1][i])], int(edata[i])])
    return np.array(New_edges)

def get_recommendations_with_rule_filter(G, model, pred, all_devices, all_trigger_actions, rules_dict, topk = 10, args_dataset = 'ifttt'): #, all_devices, all_trigger_actions,
    
    Complete_G = nx.complete_graph(list(G.nodes()), nx.MultiDiGraph())
    if args_dataset == 'wyze':
        for node in Complete_G.nodes():
            Complete_G.add_edge(node, node)
    
    class_num = len(set(all_devices.values()))
    one_hot_matrix = F.one_hot(torch.tensor(list(range(class_num))), num_classes=class_num)

    devices = list(Complete_G.nodes())
    
    Complete_G = dgl.from_networkx(Complete_G)
    
    
    model.eval()
    pred.eval()
    with torch.no_grad():
        h = model(G, G.ndata['feat'])
        #h = model(G, G.ndata['feat'], G.edata['efeat'])
        
        ###########
        #predictor, use node embeddings of source node and target node as input, predict the link probability of current edge
        #need a complete graph as input
        scores = pred(Complete_G, h)
    L = []
    edges = Complete_G.edges()
    for i in range(scores.shape[0]):
        for j in range(scores.shape[1]):
            L.append([int(edges[0][i]), int(edges[1][i]), j, float(scores[i][j])])
    L = torch.tensor(sorted(L, key= lambda e:e[3], reverse = True))[:,:-1]
    
    
    
    Z = combine_triple_inv(G.edges()[0], G.edges()[1], G.edata['etype']).tolist()
    L = torch.stack([j for j in L if j.tolist() not in Z])
    
    L = L.numpy()
    
    node_transfer = np.argmax(G.ndata['feat'].numpy(), axis = 1)

    transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)
    
    new_L = []
    for i in range(len(L)):
        j = transformed_L[i]
        if ' '.join([str(j[0]), str(j[1]), str(j[2])]) in rules_dict:
            new_L.append(L[i])

    L = new_L
    
    L = torch.tensor(L[:topk])

    #get_rules([L[:,0],L[:,1]], L[:,2], inv_d(all_devices), inv_d(all_trigger_actions))
    return L


def get_recommendations_with_rule_filter_with_probablity(G, model, pred, all_devices, all_trigger_actions, rules_dict, topk = 10, args_dataset = 'ifttt'): #, all_devices, all_trigger_actions,
    
    Complete_G = nx.complete_graph(list(G.nodes()), nx.MultiDiGraph())
    if args_dataset == 'wyze':
        for node in Complete_G.nodes():
            Complete_G.add_edge(node, node)
    class_num = len(set(all_devices.values()))
    one_hot_matrix = F.one_hot(torch.tensor(list(range(class_num))), num_classes=class_num)

    devices = list(Complete_G.nodes())

    
    Complete_G = dgl.from_networkx(Complete_G)
    
    
    model.eval()
    pred.eval()
    with torch.no_grad():
        h = model(G, G.ndata['feat'])
        #h = model(G, G.ndata['feat'], G.edata['efeat'])
        
        ###########
        #predictor, use node embeddings of source node and target node as input, predict the link probability of current edge
        #need a complete graph as input
        scores = pred(Complete_G, h)
    L = []
    edges = Complete_G.edges()
    for i in range(scores.shape[0]):
        for j in range(scores.shape[1]):
            L.append([int(edges[0][i]), int(edges[1][i]), j, float(scores[i][j])])
    L = torch.tensor(sorted(L, key= lambda e:e[3], reverse = True))
    
    
    
    Z = combine_triple_inv(G.edges()[0], G.edges()[1], G.edata['etype']).tolist()
    L = torch.stack([j for j in L if j[:-1].tolist() not in Z])
    
    L = L.numpy()
    
    node_transfer = np.argmax(G.ndata['feat'].numpy(), axis = 1)

    transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)
    
    
    
    new_L = []
    for i in range(len(L)):
        j = transformed_L[i]
        if ' '.join([str(j[0]), str(j[1]), str(j[2])]) in rules_dict:
            new_L.append(L[i])

    L = new_L
    
    L = torch.tensor(L[:topk])

    #get_rules([L[:,0],L[:,1]], L[:,2], inv_d(all_devices), inv_d(all_trigger_actions))
    return L



def combine_triple_inv(a, b, c):
    d = torch.stack((a,b,c))
    return torch.transpose(d, 0, 1)

def get_hit_rate(train_gs, test_pos_gs, model, pred, all_devices, topk = 10, dataset = 'ifttt'):
    total_hit_rate = 0
    for user_index in train_gs:
        
        L = get_recommendations(train_gs[user_index], model, pred, all_devices, topk, dataset).tolist()
        Test_Z = combine_triple_inv(test_pos_gs[user_index].edges()[0], test_pos_gs[user_index].edges()[1], test_pos_gs[user_index].edata['etype'])
        total_hit_rate += len([j for j in Test_Z.tolist() if j in L]) / len(Test_Z)
    return total_hit_rate / len(test_pos_gs)









def get_hit_rate_with_rule_filter(train_gs, test_pos_gs, model, pred, all_devices, all_trigger_actions, rules_dict, topk = 10, dataset = 'ifttt'):
    total_hit_rate = 0
    for user_index in train_gs:
        
        L = get_recommendations_with_rule_filter(train_gs[user_index], model, pred, all_devices, all_trigger_actions, rules_dict, topk, dataset).tolist()
        Test_Z = combine_triple_inv(test_pos_gs[user_index].edges()[0], test_pos_gs[user_index].edges()[1], test_pos_gs[user_index].edata['etype'])
        total_hit_rate += len([j for j in Test_Z.tolist() if j in L]) / len(Test_Z)
    return total_hit_rate / len(test_pos_gs)

def get_rule_dict(all_devices, all_trigger_actions, dataset):
    if dataset == 'wyze':
        df = pd.read_csv('data/wyze/wyze_same_type_devices.csv')


        rules_dict = collections.defaultdict(int)
        for index, row in df.iterrows():
            #### do not consider self loop 
            #if all_devices[row['trigger_device']] != all_devices[row['action_device']]:
                if row['trigger_state'] + " " + row['action'] in all_trigger_actions and row['trigger_device'] in all_devices and row['action_device'] in all_devices:
                    current_rule = ' '.join([str(all_devices[row['trigger_device']]), str(all_devices[row['action_device']]), str(all_trigger_actions[row['trigger_state'] + " " + row['action']])])
                    rules_dict[current_rule] += 1

                #[row['trigger_device'], row['action_device'], row['trigger_state'], row['action']]
    elif dataset == 'ifttt':
        rules_dict = collections.defaultdict(int)
        with open("data/ifttt/rules", "r") as a:
            for line in a:
                line = line.strip()
                line = line.split()
                
                
                if line[1].split('_')[1] in all_devices:
                    trigger_device = all_devices[line[1].split('_')[1]]
                    trigger_state = line[1].split(';')[0].split("_")[2:]
                elif ' '.join(line[1].split('_')[1:3]) in all_devices:
                    trigger_device = all_devices[' '.join(line[1].split('_')[1:3])]
                    trigger_state = line[1].split(';')[0].split("_")[3:]
                else:
                    continue

                if line[1].split(';')[1].split("_")[1] in all_devices:
                    action_device = all_devices[line[1].split(';')[1].split("_")[1]]
                    action_state = line[1].split(';')[1].split("_")[2:]
                elif ' '.join(line[1].split(';')[1].split("_")[1:3]) in all_devices:
                    action_device = all_devices[' '.join(line[1].split(';')[1].split("_")[1:3])]
                    action_state = line[1].split(';')[1].split("_")[3:]
                else:
                    continue
                
                trigger_action = '_'.join(trigger_state) + ' ' + '_'.join(action_state)
                if trigger_action in all_trigger_actions:
                    rules_dict[str(trigger_device) + ' ' + str(action_device) + ' ' + str(all_trigger_actions[trigger_action])] += 1 

    return rules_dict



def get_results(selected_nodes, graph_gen, all_devices, all_trigger_actions, rules_dict, top_k = 10, dataset = 'ifttt'):
    sys_device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

    model = GraphSAGE(len(set(all_devices.values())), 16)  
    pred = HeteroMLPPredictor(16, len(set(all_trigger_actions.values())))
    model.load_state_dict(torch.load("models/wyzefedgatefed_model"))
    pred.load_state_dict(torch.load("models/wyzefedgatefed_pred"))

    G = nx.MultiDiGraph()
    class_num = len(set(all_devices.values()))
    one_hot_matrix = F.one_hot(torch.tensor(list(range(class_num))), num_classes=class_num)

    for key in selected_nodes:
        device = all_devices[selected_nodes[key][0]]
        G.add_node(selected_nodes[key][2], feat = one_hot_matrix[device]) 

    for i in graph_gen.G.edges(data=True):
        rule = i[2]['rule']
        rule = ' '.join(rule.split(":"))
        G.add_edge(i[0], i[1], etype = all_trigger_actions[rule])

    G = dgl.from_networkx(G, ["feat"], ["etype"])
    G.ndata['feat'] = G.ndata['feat'].float()
    G = G.to(sys_device)
    node_transfer = np.argmax(G.ndata['feat'].numpy(), axis = 1)
    transformed_L = transform_edges(G.edges(), G.edata['etype'], node_transfer)
    get_rules((transformed_L[:,0], transformed_L[:,1]), transformed_L[:,2], inv_d(all_devices), inv_d(all_trigger_actions))

    L = get_recommendations_with_rule_filter_with_probablity(G, model, pred, all_devices, all_trigger_actions, rules_dict, top_k, dataset)
    transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)
    return get_rules((transformed_L[:,0], transformed_L[:,1]), transformed_L[:,2], inv_d(all_devices), inv_d(all_trigger_actions)), L[:,-1]

 

def mean_rank_no_filter(model, pred, train_gs, test_pos_gs):
    model.eval()
    pred.eval()
    loss = None
    total_pos_MR = 0

    with torch.no_grad():
        for user_index in test_pos_gs:
            user_rank = 0
            
            train_g = train_gs[user_index]
            test_pos_g = test_pos_gs[user_index]
            
            h = model(train_g, train_g.ndata['feat'])
            
            sortedd, indices = torch.sort(pred(test_pos_g, h), descending = True)

            Edges = [test_pos_g.edges()[0].tolist(), test_pos_g.edges()[1].tolist()]
            Z = combine_triple_inv(train_g.edges()[0], train_g.edges()[1], train_g.edata['etype']).tolist()
            user_rank = 0
            for edge_index in range(len(indices)):
                each_test_edge = indices[edge_index].tolist()
                
                
                L = [[Edges[0][edge_index], Edges[1][edge_index], j] for j in each_test_edge]
                
                L = np.array([j for j in L if j not in Z])
                

                L = np.array(L)
                
                rank = np.where(L[:,2] == int(test_pos_g.edata['etype'][edge_index]))[0][0]
                user_rank += rank
                
            user_rank = user_rank / len(indices)
            total_pos_MR += user_rank
            #print(user_rank)

    total_pos_MR /= len(test_pos_gs)

    return total_pos_MR

    


def mean_rank_after_filter(model, pred, train_gs, test_pos_gs, rules_dict):
    model.eval()
    pred.eval()
    loss = None
    total_pos_MR = 0

    with torch.no_grad():
        for user_index in test_pos_gs:
            user_rank = 0
            
            train_g = train_gs[user_index]
            test_pos_g = test_pos_gs[user_index]
            
            h = model(train_g, train_g.ndata['feat'])
            
            sortedd, indices = torch.sort(pred(test_pos_g, h), descending = True)

            Edges = [test_pos_g.edges()[0].tolist(), test_pos_g.edges()[1].tolist()]
            Z = combine_triple_inv(train_g.edges()[0], train_g.edges()[1], train_g.edata['etype']).tolist()
            user_rank = 0
            for edge_index in range(len(indices)):
                each_test_edge = indices[edge_index].tolist()
                
                
                L = [[Edges[0][edge_index], Edges[1][edge_index], j] for j in each_test_edge]
                
                L = np.array([j for j in L if j not in Z])
                
                node_transfer = np.argmax(train_g.ndata['feat'].numpy(), axis = 1)

                transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)
                
                new_L = []
                for i in range(len(L)):
                    j = transformed_L[i]
                    if ' '.join([str(j[0]), str(j[1]), str(j[2])]) in rules_dict:
                        new_L.append(L[i])
                
                L = np.array(new_L)


                rank = np.where(L[:,2] == int(test_pos_g.edata['etype'][edge_index]))[0][0]
                user_rank += rank
            user_rank = user_rank / len(indices)
            total_pos_MR += user_rank

    total_pos_MR /= len(test_pos_gs)

    return total_pos_MR

    


def hit_rate_mean_rank_after_filter(model, pred, train_gs, test_pos_gs, rules_dict, top_k = 10):
    model.eval()
    pred.eval()
    loss = None
    total_hit_rate = 0

    with torch.no_grad():
        for user_index in test_pos_gs:
            user_hit_rate = 0
            
            train_g = train_gs[user_index]
            test_pos_g = test_pos_gs[user_index]
            
            h = model(train_g, train_g.ndata['feat'])
            
            sortedd, indices = torch.sort(pred(test_pos_g, h), descending = True)

            Edges = [test_pos_g.edges()[0].tolist(), test_pos_g.edges()[1].tolist()]
            Z = combine_triple_inv(train_g.edges()[0], train_g.edges()[1], train_g.edata['etype']).tolist()
            user_rank = 0
            for edge_index in range(len(indices)):
                each_test_edge = indices[edge_index].tolist()
                
                
                L = [[Edges[0][edge_index], Edges[1][edge_index], j] for j in each_test_edge]
                
                L = np.array([j for j in L if j not in Z])
                

                node_transfer = np.argmax(train_g.ndata['feat'].numpy(), axis = 1)

                transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)

                new_L = []
                for i in range(len(L)):
                    j = transformed_L[i]
                    if ' '.join([str(j[0]), str(j[1]), str(j[2])]) in rules_dict:
                        new_L.append(L[i])

                L = np.array(new_L)
                
                rank = np.where(L[:,2] == int(test_pos_g.edata['etype'][edge_index]))[0][0]
                if rank < top_k:
                    user_hit_rate += 1
            user_hit_rate = user_hit_rate / len(indices)
            total_hit_rate += user_hit_rate

    total_hit_rate /= len(test_pos_gs)

    return total_hit_rate



    

def hit_rate_mean_rank_no_filter(model, pred, train_gs, test_pos_gs, top_k = 10):
    model.eval()
    pred.eval()
    loss = None
    total_hit_rate = 0

    with torch.no_grad():
        for user_index in test_pos_gs:
            user_hit_rate = 0
            
            train_g = train_gs[user_index]
            test_pos_g = test_pos_gs[user_index]
            
            h = model(train_g, train_g.ndata['feat'])
            
            sortedd, indices = torch.sort(pred(test_pos_g, h), descending = True)

            Edges = [test_pos_g.edges()[0].tolist(), test_pos_g.edges()[1].tolist()]
            Z = combine_triple_inv(train_g.edges()[0], train_g.edges()[1], train_g.edata['etype']).tolist()
            user_rank = 0
            for edge_index in range(len(indices)):
                each_test_edge = indices[edge_index].tolist()
                
                
                L = [[Edges[0][edge_index], Edges[1][edge_index], j] for j in each_test_edge]
                
                L = np.array([j for j in L if j not in Z])
                
                L = np.array(L)
                
                rank = np.where(L[:,2] == int(test_pos_g.edata['etype'][edge_index]))[0][0]
                if rank < top_k:
                    user_hit_rate += 1
            user_hit_rate = user_hit_rate / len(indices)
            total_hit_rate += user_hit_rate

    total_hit_rate /= len(test_pos_gs)

    return total_hit_rate


def get_results(selected_nodes, graph_gen, all_devices, all_trigger_actions, rules_dict, top_k = 10):
    sys_device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')

    model = GraphSAGE(len(set(all_devices.values())), 16)
    pred = HeteroMLPPredictor(16, len(set(all_trigger_actions.values())))
    model.load_state_dict(torch.load("models/wyzefedgatefed_model"))
    pred.load_state_dict(torch.load("models/wyzefedgatefed_pred"))

    G = nx.MultiDiGraph()
    class_num = len(set(all_devices.values()))
    one_hot_matrix = F.one_hot(torch.tensor(list(range(class_num))), num_classes=class_num)

    for key in selected_nodes:
        device = all_devices[selected_nodes[key][0]]
        G.add_node(selected_nodes[key][2], feat = one_hot_matrix[device]) 

    for i in graph_gen.G.edges(data=True):
        rule = i[2]['rule']
        rule = ' '.join(rule.split(":"))
        G.add_edge(i[0], i[1], etype = all_trigger_actions[rule])

    G = dgl.from_networkx(G, ["feat"], ["etype"])
    G.ndata['feat'] = G.ndata['feat'].float()
    G = G.to(sys_device)
    node_transfer = np.argmax(G.ndata['feat'].numpy(), axis = 1)
    transformed_L = transform_edges(G.edges(), G.edata['etype'], node_transfer)
    get_rules((transformed_L[:,0], transformed_L[:,1]), transformed_L[:,2], inv_d(all_devices), inv_d(all_trigger_actions))

    L = get_recommendations_with_rule_filter_with_probablity(G, model, pred, all_devices, all_trigger_actions, rules_dict, topk = top_k)
    transformed_L = transform_edges((L[:,0], L[:,1]), L[:,2], node_transfer)
    return get_rules((transformed_L[:,0], transformed_L[:,1]), transformed_L[:,2], inv_d(all_devices), inv_d(all_trigger_actions)), L[:,-1]

 







