import argparse
import os
import random
import shutil
import time
import warnings
import sys

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
# import torchvision.models as models
import torch.nn.functional as F
from torch.optim.lr_scheduler import CosineAnnealingLR

from util import mixup_data, mixup_criterion

import models

model_names = sorted(name for name in models.__dict__
    if name.islower() and not name.startswith("__")
    and callable(models.__dict__[name]))

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')

parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet101',
                    choices=model_names,
                    help='model architecture: ' +
                        ' | '.join(model_names) +
                        ' (default: resnet18)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=90, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=32, type=int,
                    metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--base_lr', default=0.1, type=float,
                    metavar='base_lr', help='base learning rate (default=0.1)', dest='base_lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)',
                    dest='weight_decay')
parser.add_argument('-p', '--print-freq', default=10, type=int,
                    metavar='N', help='print frequency (default: 10)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')
parser.add_argument('--seed', default=None, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--gpu', default=0, type=int,
                    help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')
parser.add_argument('--alpha', default=0.7, type=float, help='interpolation strength (uniform=1., ERM=0.)')

best_acc1 = 0
args = parser.parse_args()
base_learning_rate = args.base_lr * args.batch_size / 256.
# base_learning_rate *= torch.cuda.device_count()

def main(eps, att_type):
    args.eps = eps
    args.type = att_type
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    #print(args.distributed)

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    args.gpu = gpu

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    # create model
    if args.pretrained:
        print("=> using pre-trained model '{}'".format(args.arch))
        model = models.__dict__[args.arch](pretrained=True)
    else:
        print("=> creating model '{}'".format(args.arch))
        model = models.__dict__[args.arch]()

    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int(args.workers / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.arch.startswith('alexnet') or args.arch.startswith('vgg'):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()

    # define loss function (criterion) and optimizer
    cel = nn.CrossEntropyLoss()
    criterion = lambda pred, target, lam: (-F.log_softmax(pred, dim=1) * torch.zeros(pred.size()).cuda().scatter_(1, target.data.view(-1, 1), lam.view(-1, 1))).sum(dim=1).mean()
    parameters_bias = [p[1] for p in model.named_parameters() if 'bias' in p[0]]
    parameters_scale = [p[1] for p in model.named_parameters() if 'scale' in p[0]]
    parameters_others = [p[1] for p in model.named_parameters() if not ('bias' in p[0] or 'scale' in p[0])]
    optimizer = torch.optim.SGD(
        [{'params': parameters_bias, 'lr': args.base_lr/10.},
        {'params': parameters_scale, 'lr': args.base_lr/10.},
        {'params': parameters_others}],
        lr=base_learning_rate,
        momentum=args.momentum,
        weight_decay=args.weight_decay)

    # optionally resume from a checkpoint

    #print(model)

    args.resume = './checkpoint/' + 'fixup_resnet101_best.pth.tar'

    if (1):
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            checkpoint = torch.load(args.resume)
            args.start_epoch = checkpoint['epoch']
            best_acc1 = checkpoint['best_acc1']
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint['state_module_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    # Data loading code
    traindir = os.path.join('./imagenet/imagenet-data/', 'train')
    valdir = os.path.join('./imagenet/imagenet-data/', 'validation')
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    train_dataset = datasets.ImageFolder(
        traindir,
        transforms.Compose([
            transforms.RandomResizedCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize,
        ]))

    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
        num_workers=args.workers, pin_memory=True, sampler=train_sampler)

    val_loader = torch.utils.data.DataLoader(
        datasets.ImageFolder(valdir, transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ])),
        batch_size=args.batch_size, shuffle=False,
        num_workers=args.workers, pin_memory=True)

    if (1):
        validate(val_loader, model, criterion, args)
        return

    sgdr = CosineAnnealingLR(optimizer, args.epochs, eta_min=0, last_epoch=-1)
    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, model, criterion, optimizer, epoch, args)

        # evaluate on validation set
        acc1 = validate(val_loader, model, cel, args)

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)

        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0):
            save_checkpoint({
                'epoch': epoch + 1,
                'arch': args.arch,
                'state_dict': model.state_dict(),
                'state_module_dict': model.module.state_dict(),
                'best_acc1': best_acc1,
                'optimizer' : optimizer.state_dict(),
            }, is_best, args.arch, epoch)

def pgd_radial_movement_alt(model, X, y, epsilon, ls, alpha=0.01, num_iter=40):
    tempinit = alpha*(torch.zeros_like(X).sign())
    advx = (X + tempinit).clone().detach()
    advxalt = (X + tempinit).clone().detach()
    i = 0
    with torch.no_grad():
        origlat = model.firstlayerlatent(X)
        altoriglat = model.difflayerlatent(X)
    for t in range(num_iter):
        model.zero_grad()
        advx.requires_grad = True
        getstuff = model.difflayerlatent(advx)
        #loss = torch.norm(getstuff - altoriglat)
        if(t<=20):
            loss = -torch.norm(getstuff)
            deflator = 40
        else:
            loss = -(torch.dot(getstuff.flatten(),altoriglat.flatten())) / torch.norm(getstuff)
            #loss = -(torch.dot(getstuff.flatten(),altoriglat.flatten())) + 0.5*torch.dot(getstuff.flatten(),getstuff.flatten())
            deflator = 2
        grad = torch.autograd.grad(loss, advx,retain_graph=False, create_graph=False)[0]

        model.zero_grad()
        advxalt.requires_grad = True
        getstuffalt = model.firstlayerlatent(advxalt)
        #lossalt = torch.norm(getstuffalt - origlat)
        if(t<=20):
            lossalt = -torch.norm(getstuffalt)
        else:
            lossalt = -(torch.dot(getstuffalt.flatten(),origlat.flatten())) / torch.norm(getstuffalt)
            #lossalt = -(torch.dot(getstuffalt.flatten(),origlat.flatten())) + 0.5*torch.dot(getstuffalt.flatten(),getstuffalt.flatten())
        gradalt = torch.autograd.grad(lossalt, advxalt,retain_graph=False, create_graph=False)[0]

        advx = advx.detach() + (alpha/deflator)*(grad.sign() + gradalt.sign())
        #advx = advx.detach() + (alpha/deflator)*((grad + gradalt + gradalt_new).sign())
        delta = torch.clamp(advx - X, min=-epsilon, max=epsilon)
        advx = (X + delta).detach()
        advxalt = (X + delta).detach()
    #return torch.clamp(X + delta.detach(),0,1)
    return X + delta.detach()

def train(train_loader, model, criterion, optimizer, epoch, args):
    batch_time = AverageMeter()
    data_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()

    # switch to train mode
    model.train()

    end = time.time()
    for i, (inputs, targets) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        inputs = inputs.cuda(args.gpu, non_blocking=True)
        targets = targets.cuda(args.gpu, non_blocking=True)

        inputs, targets_a, targets_b, lam = mixup_data(inputs, targets, args.alpha, use_cuda=True)

        # compute output
        output = model(inputs)
        loss_func = mixup_criterion(targets_a, targets_b, lam)
        loss = loss_func(criterion, output)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, targets, topk=(1, 5))
        losses.update(loss.item(), inputs.size(0))
        top1.update(acc1[0], inputs.size(0))
        top5.update(acc5[0], inputs.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % 1 == 0:
            print('Epoch: [{0}][{1}/{2}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                  'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                  'Acc@1 {top1.val:.3f} ({top1.avg:.3f})\t'
                  'Acc@5 {top5.val:.3f} ({top5.avg:.3f})'.format(
                   epoch, i, len(train_loader), batch_time=batch_time,
                   data_time=data_time, loss=losses, top1=top1, top5=top5))


def fgsm(model, X, y, epsilon, ls):
    model.zero_grad()
    delta = torch.zeros_like(X, requires_grad=True)
    getstuff = model(X+delta)
    loss = ls(getstuff, y)
    loss.backward()
    return (X + epsilon * delta.grad.detach().sign())

def pgd_alt(model, X, y, epsilon, ls, alpha=0.01, num_iter=40):
    advx = X.clone().detach()
    for t in range(num_iter):
        advx.requires_grad = True
        model.zero_grad()
        getstuff = model(advx)
        loss = ls(getstuff, y)
        #loss.backward()
        grad = torch.autograd.grad(loss, advx,retain_graph=False, create_graph=False)[0]
        advx = advx.detach() + alpha*(grad.sign())
        delta = torch.clamp(advx - X, min=-epsilon, max=epsilon)
        advx = (X + delta).detach()
    #return torch.clamp(X + delta.detach(),0,1)
    return X + delta.detach()

def randomnoise(model, X, y, epsilon, ls):
    model.zero_grad()
    delta = torch.zeros_like(X, requires_grad=True)
    getstuff = model(X+delta)
    loss = ls(getstuff, y)
    loss.backward()
    return (X + epsilon * ((torch.randn_like(delta.grad.detach())).sign()))

def validate(val_loader, model, criterion, args):
    batch_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()

    # switch to evaluate mode
    model.eval()

    end = time.time()
    for i, (input, target) in enumerate(val_loader):
        if args.gpu is not None:
            input = input.cuda(args.gpu, non_blocking=True)
        target = target.cuda(args.gpu, non_blocking=True)

        # compute output
        cel = nn.CrossEntropyLoss()
        #input = fgsm(model, input, target, 0.1, cel)
        if args.type=='pgd':
            input = pgd_alt(model, input, target, args.eps, cel, alpha=0.01, num_iter=40)
        elif args.type=='fgsm':
            input = fgsm(model, input, target, args.eps, cel)
        else:
            input = randomnoise(model, input, target, args.eps, cel)

        #input = pgd_radial_movement_alt(model, input, target, args.eps, cel, alpha=0.01, num_iter=40)

        output = model(input)
        #loss = criterion(output, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        #losses.update(loss.item(), input.size(0))
        top1.update(acc1[0], input.size(0))
        top5.update(acc5[0], input.size(0))

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            print('Test: [{0}/{1}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Acc@1 {top1.val:.3f} ({top1.avg:.3f})\t'
                  'Acc@5 {top5.val:.3f} ({top5.avg:.3f})'.format(
                   i, len(val_loader), batch_time=batch_time,   top1=top1, top5=top5))

    print(' * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'
          .format(top1=top1, top5=top5))

    return top1.avg


def save_checkpoint(state, is_best, suffix, suffixepoch):
    torch.save(state, './checkpoint/'+str(suffix)+'_'+"current"+'.pth.tar')
    if is_best:
        shutil.copyfile('./checkpoint/'+str(suffix)+'_'+"current"+'.pth.tar', './checkpoint/'+str(suffix)+'_'+'best.pth.tar')


class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self):
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count


def adjust_learning_rate(optimizer, epoch, args):
    # """Sets the learning rate to the initial LR decayed by 10 every 30 epochs"""
    # lr = args.base_lr * (0.1 ** (epoch // 30))
    #     for param_group in optimizer.param_groups:
    #     param_group['lr'] = lr

    for param_group in optimizer.param_groups:
        if param_group['initial_lr'] == base_learning_rate:
            print("adjust non-scalar lr.")
            lr = base_learning_rate * (0.1 ** (epoch // 30))
            param_group['lr'] = lr
        else:
            print("adjust scalar lr.")
            scalar_lr = param_group['initial_lr'] * (0.1 ** (epoch // 30))
            param_group['lr'] = scalar_lr


def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].view(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res


if __name__ == '__main__':
    for eps in [0.03,0.06,0.1]:
        for typeofatt in ['random','fgsm','pgd']:
            main(eps,typeofatt)
