############################################################################################
# app_witness_synthetic.R
#
# Generates a synthetic study for evaluating the Witness Protection Program methodology.
#
# Code by
#
#  - Ricardo Silva (ricardo@stats.ucl.ac.uk)
#
# Current version: 17/04/2014
# First version: 17/04/2014

source("witness.R")

############################################################################################
# app_witness_synthetic::
#
# Generates a synthetic study in low-dimensional problems using default relaxation metrics.
# THIS ASSUMES THE PROBLEM IS SMALL ENOUGH SO THAT EXACT THE CURRENT EXHAUSTIVE SEARCH
# PROCEDURE CAN BE USED AND THE EXACT POPULATION EFFECTS CALCULATED.
#
# * Input
#
# - bias_thres: between 0 and 1, a threshold of bias for the full-conditioning backdoor adjustment,
#                so that we accept a synthetic case
# - p: number of background variables that don't hurt by conditioning
# - q: number of background variables that hurt by conditioning
# - par_max: maximum number of parents
# - no_sol: if TRUE, generates a problem in which no witness can be found (in the population)
# - epsilons: parameters of relaxation. See 'witness.R'
# - num_S: number of simulations
# - sample_size: sample size
# - num_monte_carlo: number of Monte Carlo samples
# - input_problems: pre-determined input problems, if desired (output of app_witness_generate_problem_batch).
#                   Parameters bias_thres, p, q, par_max, num_S are ignored if this is given. "no_sol",
#                   however, is used.
#
# * Output:
#
# - problems: the respective problems
# - effects: population effects
# - v: bounds found by witness procedure
# - s: summary of those bounds
# - score_tightest: 1, if tighest bound of the solution includes the true effect. 0, otherwise
# - score_error: if true effect "e" is not in the tightest bound [u, l], this is e - l if e < u, or
#                e - u, if e > u
# - score_loosest: 1, if loosest composite bound (the maximum of the upper bounds joined with the
#                  minimum of the lower bounds) of the solution includes the true effect. 0, otherwise
# - score_trivial: 1, if solution returned found no witness. SO this is good or bad depending on
#                  the type of simulation performed. Notice that even for the "no_sol" problems we
#                  compute the other metrics, since they might still be approximately correct
# - solvable: copy information from "no_sol" indicating whether this is a set of solvable instances

app_witness_synthetic <- function(bias_thres, p, q, par_max, no_sol, epsilons, num_S, sample_size, num_monte_carlo, input_problems)
{
  problems <- list()
  effects <- list()
  effects_backdoor <- list()
  v <- list()
  s <- list()

  if (!missing(input_problems)) num_S <- length(input_problems)
    
  score_tightest <- rep(0, num_S)
  score_highest  <- rep(0, num_S)
  score_error    <- rep(0, num_S)
  score_loosest  <- rep(0, num_S)
  score_trivial  <- rep(0, num_S)

  for (i in 1:num_S) {

    cat("SIMULATION ", i," [SOLVABLE = ", no_sol == FALSE, "]\n", sep = "")

    if (missing(input_problems)) {
      app_instance <- app_witness_infer_synthetic(bias_thres, p, q, par_max, no_sol, epsilons, sample_size, num_monte_carlo)
    } else {
      app_instance <- app_witness_infer_synthetic(bias_thres, p, q, par_max, no_sol, epsilons, sample_size, num_monte_carlo, input_problems[[i]])
    }

    problems[[i]]         <- app_instance$problems
    effects[[i]]          <- app_instance$effects
    effects_backdoor[[i]] <- app_instance$effects_backdoor
    v[[i]]                <- app_instance$v
    s[[i]]                <- app_instance$s

    ### Evaluate

    evals <- app_witness_score(s[[i]]$tightest_bound, s[[i]]$highest_bound, s[[i]]$min_l, s[[i]]$max_u, effects[[i]]$effect_real)

    score_tightest[i] <- evals$score_tightest
    score_highest[i]  <- evals$score_highest
    score_error[i]    <- evals$score_error
    score_loosest[i]  <- evals$score_loosest
    score_trivial[i]  <- evals$score_trivial

  }


  return(list(problems = problems, effects = effects, effects_backdoor = effects_backdoor, v = v, s = s,
              score_tightest = score_tightest, score_highest = score_highest, score_error = score_error,
              score_loosest = score_loosest, score_trivial = score_trivial,
              solvable = no_sol == FALSE))
}

############################################################################################
# app_witness_infer_synthetic::
#
# Do a run of the WPP algorithm and returns a series of results.
# THIS ASSUMES THE PROBLEM IS SMALL ENOUGH SO THAT EXACT SOLUTIONS CAN BE FOUND.
#
# * Input
#
# - bias_thresh: between 0 and 1, a threshold of bias for the full-conditioning backdoor adjustment,
#                so that we accept a synthetic case
# - p: number of background variables that don't hurt by conditioning
# - q: number of background variables that hurt by conditioning
# - par_max: maximum number of parents
# - no_sol: if TRUE, generates a problem in which no witness can be found (in the population)
# - epsilons: parameters of relaxation
# - sample_size: sample size
# - num_monte_carlo: number of Monte Carlo samples
#
# * Output:
#
# - problem: the respective problem
# - effects: population effects
# - effects_backdoor: population effects given by suggested backdoor adjustments
# - v: bounds found by witness procedure
# - s: summary of those bounds

app_witness_infer_synthetic <- function(bias_thres, p, q, par_max, no_sol, epsilons, sample_size, num_monte_carlo, problem)
{
  if (missing(problem)) {
    cat("Generating case...")
    i <- 1
    while (TRUE) {
      problem <- simulate_witness_model(p, q, par_max, sample_size, no_sol)
      effects <- bindag_causal_effect(problem)
      out <- abs(effects[[1]] - effects[[2]])
      if (out > bias_thres) break()
      printCount(i)
      i <- i + 1
    }
    cat("\n")
  } else {
    effects <- bindag_causal_effect(problem)
  }

  v <- witness_bound_search(problem, epsilons, num_monte_carlo)

  effects_backdoor <- rep(0, length(v$w_list$witness))
  for (j in seq_along(v$w_list$witness)) effects_backdoor[j] <- bindag_causal_effect_backdoor_S(problem, v$w_list$Z[[j]])$ACE

  s <- witness_summarize_bounds(v, problem, epsilons)
  problem$counts <- NULL # To save space
  problem$probs <- NULL # To save space
  return(list(problem = problem, effects = effects, effects_backdoor = effects_backdoor, v = v, s = s))
}

############################################################################################
# app_witness_score::
#
# Scores a solution of witness_search against the known true ACE of a problem.
#
# * Input
#
# - tighest_bound: 2 dimensional vector, indicates the lower and upper bound of the tightest bound
# - highest_bound: 2 dimensional vector, indicates the lower and upper bound of the bound associated with best fit
# - min_l: minimum of all lower bounds of the solution
# - max_u: maximum of all upper bounds of the solution
# - effect_real: real ACE of the problem
#
# * Output:
#
# - score_tightest: 1, if tighest bound of the solution includes the true effect. 0, otherwise
# - score_highest: 1, if best scoring bound of the solution includes the true effect. 0, otherwise
# - score_error: distance from the real ACE to the closest point in the bounded interval. That is,
#                if true effect "e" is not in the tightest bound [u, l], this is e - l if e < u, or
#                e - u, if e > u
# - score_loosest: 1, if loosest composite bound (the maximum of the upper bounds joined with the
#                  minimum of the lower bounds) of the solution includes the true effect. 0, otherwise
# - score_trivial: 1, if solution returned found no witness. SO this is good or bad depending on
#                  the type of simulation performed. Notice that even for the "no_sol" problems we
#                  compute the other metrics, since they might still be approximately correct

app_witness_score <- function(tightest_bound, highest_bound, min_l, max_u, effect_real)
{
  score_tightest <- 0
  score_highest <- 0
  score_error <- 0
  score_loosest <- 0
  score_trivial <- 0

  if (tightest_bound[2] - tightest_bound[1] < 2) { # Non-trivial [-1, 1] bounds

    ## First score: does the tighest bound cover the causal effect?
    if (effect_real >= tightest_bound[1] && effect_real <= tightest_bound[2]) {
      score_tightest <- 1
    }

    ## Second score: does the highest (scoring) bound cover the causal effect?
    if (effect_real >= highest_bound[1] && effect_real <= highest_bound[2]) {
      score_highest <- 1
    } else { # Measure distance, if not
      if (effect_real < highest_bound[1])
        score_error <- effect_real - highest_bound[1]
      if (effect_real > highest_bound[2])
        score_error <- effect_real - highest_bound[2]
    }

    ## Second score: does the most relaxed bound cover the causal effect?
    if (effect_real >= min_l && effect_real <= max_u)
      score_loosest <- 1

  } else { # Indicate a trivial solution was found
    score_trivial <- 1
  }

  return(list(score_tightest = score_tightest, score_highest = score_highest, score_error = score_error,
              score_loosest = score_loosest, score_trivial = score_trivial))
}

############################################################################################
# app_witness_summarize_run::
#
# Summarizes the output of a run.
#
# Input:
#
# - results: the list of results obtained from a call to 'app_witness_synthetic'.
#
# Output: a list of statistics summarizing the quality of the results;
#
# - bias_naive1: bias of the (population) unadjusted P(Y = 1 | X = 1) - P(Y = 1 | X = 0)
# - bias_naive2: bias of the (population) full-conditioning backdoor adjustment
# - bias_backdoor: bias of the (population) faithfulness-chosen backdoor adjustment
# - bias_highest_bound: bias of the (posterior expected) highest-scoring bound
# - score_tightest, score_highest, score_loosest, score_error: as in 'app_witness_score'
# - tightest_bound_width, hightest_bound_width, loosest_bound_width: width of the bounded
#                    intervals, as defined in 'app_witness_score'
# - correct_trivial: array of integers;  1 if problem is not solvable and we 
#                    correctly identified it, 0 if problem is solvable but we stated
#                    otherwise, -1 if problem is not solvable but we stated the opposite
# - ace_reported: array of binary integers, 1 indicates whether we reported a non-vacuous bound

app_witness_summarize_run <- function(results)
{
  num_r <- length(results$effects)
  ace_reported <- which(results$score_trivial == 0)

  score_tightest <- rep(NA, num_r)
  score_tightest[ace_reported] <- results$score_tightest[ace_reported]
  score_highest <- rep(NA, num_r)
  score_highest[ace_reported] <- results$score_highest[ace_reported]
  score_loosest <- rep(NA, num_r)
  score_loosest[ace_reported] <- results$score_loosest[ace_reported]
  correct_trivial <- results$score_trivial == 1 - results$solvable

  bias_naive1   <- rep(0, num_r)
  bias_naive2   <- rep(0, num_r)
  bias_backdoor <- rep(NA, num_r)
  tightest_bound_width <- rep(NA, num_r)
  highest_bound_width <- rep(NA, num_r)
  loosest_bound_width <- rep(NA, num_r)
  bias_highest_bound <- rep(NA, num_r)

  for (i in 1:num_r) {

    bias_naive1[i] <- abs(results$effects[[i]]$effect_real - results$effects[[i]]$effect_naive)
    bias_naive2[i] <- abs(results$effects[[i]]$effect_real - results$effects[[i]]$effect_naive2)

    if (results$score_trivial[i] == 0) {
       bias_backdoor[i] <- mean(abs(results$effects_backdoor[[i]] - results$effects[[i]]$effect_real))
       tightest_bound_width[i] <- results$s[[i]]$tightest_bound[2] - results$s[[i]]$tightest_bound[1]
       highest_bound_width[i] <- results$s[[i]]$highest_bound[2] - results$s[[i]]$highest_bound[1]
       loosest_bound_width[i] <- results$s[[i]]$max_u - results$s[[i]]$min_l
       if (!is.na(results$s[[i]]$highest_bound[1])) {
         if (results$effects[[i]]$effect_real < results$s[[i]]$highest_bound[1]) {
           bias_highest_bound[i] <- results$effects[[i]]$effect_real - results$s[[i]]$highest_bound[1]
         } else if (results$effects[[i]]$effect_real > results$s[[i]]$highest_bound[2]) {
           bias_highest_bound[i] <- results$effects[[i]]$effect_real - results$s[[i]]$highest_bound[2]
         } else {
           bias_highest_bound[i] <- 0
         }
       } else {
         cat("NA result in iteration", i, "\n")
         bias_highest_bound[i] <- NA
       }
    }

  }

  return(list(bias_naive1 = bias_naive1, bias_naive2 = bias_naive2, bias_backdoor = bias_backdoor,
              bias_highest_bound = bias_highest_bound,               
              score_tightest = score_tightest, score_highest = score_highest, score_loosest = score_loosest,
              score_error = results$score_error,
              tightest_bound_width = tightest_bound_width,
              highest_bound_width = highest_bound_width,
              loosest_bound_width = loosest_bound_width,
              correct_trivial = correct_trivial, ace_reported = ace_reported))
}

############################################################################################
# app_witness_diagnose::
#
# Perform diagnostics on a particular output of a series of synthetic studies.
#
# Input:
#
# - results: the outcome of the call to "app_witness_synthetic"
# - idx: particular instance of results to be analyzed
# - verbose: if TRUE, prints out model parameters (warning: might take much space)

app_witness_diagnose <- function(results, idx, verbose = FALSE)
{
  n_bounds <- nrow(results$v[[idx]]$bounds)
  if (n_bounds > 0) {
    cat("Bounds:\n\n")
    for (i in 1:n_bounds)
       cat("  [", results$v[[idx]]$bounds[i, 1], ", ", results$v[[idx]]$bounds[i, 2], "]\n", sep = "")
    cat("\n")
  }

  cat("Effects:\n\n")
  cat("  Real   :", results$effects[[idx]]$effect_real, "\n")
  cat("  Naive 1:", results$effects[[idx]]$effect_naive, "\n")
  cat("  Naive 2:", results$effects[[idx]]$effect_naive2, "\n\n")

  cat("Graph:\n\n")
  witness_print_graph(results$problems[[idx]])

  if (verbose) {
    cat("Model:\n\n")
    witness_print_model(results$problems[[idx]])
  }
}

############################################################################################
# app_witness_generate_problem_batch::
#
# Generates a batch of synthetic problems.
#
# Input:
#
# - results: the list of results obtained from a call to 'app_witness_synthetic'.
#
# Output: a list of statistics summarizing the quality of the results;
#
# - bias_thres: between 0 and 1, a threshold of bias for the full-conditioning backdoor adjustment,
#              so that we accept a synthetic case
# - p: number of background variables that don't hurt by conditioning
# - q: number of background variables that hurt by conditioning
# - par_max: maximum number of parents
# - no_sol: if TRUE, generates a problem in which no witness can be found (in the population)
# - sample_size: sample size
#
# Output:
#
# - problems: list of problems

app_witness_generate_problem_batch <- function(batch_size, bias_thres, p, q, par_max, sample_size, no_sol)  
{
  problems <- list()
  for (k in 1:batch_size) {
    cat(k, ": Generating case...")
    i <- 1
    while (TRUE) {
      problem <- simulate_witness_model(p, q, par_max, sample_size, no_sol)
      effects <- bindag_causal_effect(problem)
      out <- abs(effects[[1]] - effects[[2]])
      if (out > bias_thres) break()
      printCount(i)
      i <- i + 1
    }
    cat("\n")
    problems[[k]] <- problem
  }
  return(problems)
}

app_witness_generate_problem_batch_example <- function()
{
  p <- 4
  q <- 4
  par_max <- 3
  sample_size <- 5000
  hardness_factor <- 0.1  
  no_sol <- FALSE
  batch_size <- 100
  problems <- app_witness_generate_problem_batch(batch_size, hardness_factor, p, q, par_max, sample_size, no_sol)  
  return(problems)
}

############################################################################################
# app_witness_synthetic_default_run::
#
# An example of usage of app_witness_synthetic.

app_witness_synthetic_default_run <- function(file_name = "app_witness_synthetic_test.RData")
{
  epsilons <- c(0.2, 0.2, 0.2, 0.2, 1, 1)
  p <- 4                  # Maximum number of covariates which might be enough for correct ACE backdoor adjustment
  q <- 4                  # Minimum number of covariates which will introduce bias if conditioned on
  par_max <- 3            # Maximum number of parents per node
  num_S <- 100            # Number of synthetic cases
  sample_size <- 5000     # Number of simulated data points per case
  num_monte_carlo <- 1000 # Number of samples used in rejection sampler Monte Carlo procedure
  hardness_factor <- 0.1  # Degree of bias for the full-set backdoor estimator

  # Solvable instances: there is at least one witness
  result_solvable_1 <- app_witness_synthetic(hardness_factor, p, q, par_max, FALSE, epsilons, num_S, sample_size, num_monte_carlo)
  solvable_stats_1 <- app_witness_summarize_run(result_solvable_1)
  result_solvable_2 <- app_witness_synthetic(0, p, q, par_max, FALSE, epsilons, num_S, sample_size, num_monte_carlo)
  solvable_stats_2 <- app_witness_summarize_run(result_solvable_2)
  
  # 'Unsolvable' instances: there is no choice of provable witness when using exact independence
  result_nsolvable_1 <- app_witness_synthetic(hardness_factor, p, q, par_max, TRUE, epsilons, num_S, sample_size, num_monte_carlo)
  nsolvable_stats_1 <- app_witness_summarize_run(result_nsolvable_1)
  result_nsolvable_2 <- app_witness_synthetic(0, p, q, par_max, TRUE, epsilons, num_S, sample_size, num_monte_carlo)
  nsolvable_stats_2 <- app_witness_summarize_run(result_nsolvable_2)

  cat("DONE, saving results\n")
  save(result_solvable_1, solvable_stats_1, result_solvable_2,
       solvable_stats_2, result_nsolvable_1, nsolvable_stats_1,
       result_nsolvable_2, nsolvable_stats_2, file = file_name)
}

############################################################################################
# app_witness_synthetic_default_run2::
#
# An example of usage of app_witness_synthetic, here assuming problems have been generated
# independently.
#
# Input:
#
# - problems: output of 'app_witness_generate_problem_batch'
# - file_name: name of the file used to generate output

app_witness_synthetic_default_run2 <- function(file_name, problems)
{
  epsilons <- c(0.2, 0.2, 0.2, 0.2, 1, 1)
  no_sol <- FALSE
  num_monte_carlo <- 1000 # Number of samples used in rejection sampler Monte Carlo procedure
  
  result <- app_witness_synthetic(-1, -1, -1, -1, no_sol, epsilons, num_S, -1, num_monte_carlo, problems)
  result_stats <- app_witness_summarize_run(result)
  
  cat("DONE, saving results\n")
  save(result, result_stats, file = file_name)
}

############################################################################################
# app_witness_synthetic_default_run3::
#
# An example of usage of app_witness_synthetic, here assuming problems have been generated
# independently. But the function is self-contained.
#
# Input:
#
# - problems: output of 'app_witness_generate_problem_batch'
# - file_name: name of the file used to generate output

app_witness_synthetic_default_run3 <- function(file_name)
{
  epsilons_list <- list()
  for (i in 1:6) {
    epsilons_list[[i]] <- c(0.05 * i, 0.05 * i, 0.05 * i, 0.05 * i, 1, 1)
  }
  
  p <- 4                  # Maximum number of covariates which might be enough for correct ACE backdoor adjustment
  q <- 4                  # Minimum number of covariates which will introduce bias if conditioned on
  par_max <- 3            # Maximum number of parents per node
  num_S <- 100            # Number of synthetic cases 
  sample_size <- 5000     # Number of simulated data points per case
  num_monte_carlo <- 1000 # Number of samples used in rejection sampler Monte Carlo procedure
  hardness_factor <- 0.1  # Degree of bias for the full-set backdoor estimator

  # Generate problems
  
  problems_solvable_hard <- app_witness_generate_problem_batch(num_S, hardness_factor, p, q, par_max, sample_size, FALSE)  
  problems_solvable_easy <- app_witness_generate_problem_batch(num_S, 0, p, q, par_max, sample_size, FALSE)  
  problems_nsolvable_hard <- app_witness_generate_problem_batch(num_S, hardness_factor, p, q, par_max, sample_size, TRUE)  
  problems_nsolvable_easy <- app_witness_generate_problem_batch(num_S, 0, p, q, par_max, sample_size, TRUE)  
  
  cat('Saving problems\n')
  save(problems_solvable_hard, problems_solvable_easy, problems_nsolvable_hard, problems_nsolvable_easy,
       file = paste(file_name, "_problems.RData", sep = ""))
  
  # Solvable, hard
    
  result_solvable_hard <- list()
  result_stats_solvable_hard <- list()
  for (i in 1:length(epsilons_list)) {
    cat("###### SOLVABLE, HARD, Case", i, "out of", length(epsilons_list), "\n")
    result_solvable_hard[[i]] <- app_witness_synthetic(-1, -1, -1, -1, FALSE, epsilons_list[[i]], -1, -1, num_monte_carlo, problems_solvable_hard)
    result_stats_solvable_hard[[i]] <- app_witness_summarize_run(result_solvable_hard[[i]])
  }

  # Solvable, easy
    
  result_solvable_easy <- list()
  result_stats_solvable_easy <- list()
  for (i in 1:length(epsilons_list)) {
    cat("###### SOLVABLE, EASY, Case", i, "out of", length(epsilons_list), "\n")
    result_solvable_easy[[i]] <- app_witness_synthetic(-1, -1, -1, -1, FALSE, epsilons_list[[i]], -1, -1, num_monte_carlo, problems_solvable_easy)
    result_stats_solvable_easy[[i]] <- app_witness_summarize_run(result_solvable_easy[[i]])
  }

  # Not solvable, hard
    
  result_nsolvable_hard <- list()
  result_stats_nsolvable_hard <- list()
  for (i in 1:length(epsilons_list)) {
    cat("###### NOT SOLVABLE, HARD, Case", i, "out of", length(epsilons_list), "\n")
    result_nsolvable_hard[[i]] <- app_witness_synthetic(-1, -1, -1, -1, TRUE, epsilons_list[[i]], -1, -1, num_monte_carlo, problems_nsolvable_hard)
    result_stats_nsolvable_hard[[i]] <- app_witness_summarize_run(result_nsolvable_hard[[i]])
  }
  
  # Not solvable, easy
    
  result_nsolvable_easy <- list()
  result_stats_nsolvable_easy <- list()
  for (i in 1:length(epsilons_list)) {
    cat("###### NOT SOLVABLE, EASY, Case", i, "out of", length(epsilons_list), "\n")
    result_nsolvable_easy[[i]] <- app_witness_synthetic(-1, -1, -1, -1, TRUE, epsilons_list[[i]], -1, -1, num_monte_carlo, problems_nsolvable_easy)
    result_stats_nsolvable_easy[[i]] <- app_witness_summarize_run(result_nsolvable_easy[[i]])
  }

  cat("DONE, saving results\n")
  save(result_solvable_hard, result_stats_solvable_hard, 
       result_nsolvable_hard, result_stats_nsolvable_hard,
       result_solvable_easy, result_stats_solvable_easy, 
       result_nsolvable_easy, result_stats_nsolvable_easy,
       file = paste(file_name, "_results.RData"))
}

############################################################################################
# app_witness_single_run::
#
# Even simpler, does one run. This forces the random problem to be "hard". A good function
# to play with, possibly useful for debugging extensions of the procedure.
#
# Output:
#
# - problem: synthetic problem
# - effects: true ACE
# - effects_backdoor: estimated ACE for every possible witness/background set found
# - v: all choices of witness and bounds
# - s: summary of bounds
# - true_witness: all possible valid witnessess, as given by populatio model

app_witness_single_run <- function()
{
  epsilons <- c(0.2, 0.2, 0.2, 0.2, 0.9, 1.1)
  p <- 4
  q <- 4
  par_max <- 3
  sample_size <- 1000
  num_monte_carlo <- 1000
  hardness_factor <- 0.1

  N <- 1000; i <- 1
  while (TRUE) {
    problem <- simulate_witness_model(p, q, par_max, sample_size)
    effects <- bindag_causal_effect(problem)
    out <- effects[[1]] - effects[[2]]
    if (abs(out) > hardness_factor) break()
    printPercentage(i, N)
    i <- i + 1
  }

  v <- witness_bound_search(problem, epsilons, num_monte_carlo, verbose = TRUE)
  s <- witness_summarize_bounds(v, problem, epsilons)
  true_witness <- witness_search(problem, TRUE, TRUE, TRUE)
  effects_backdoor <- rep(0, length(v$w_list$witness))
  if (length(v$w_list$witness) > 0) {
     for (i in 1:length(v$w_list$witness))
       effects_backdoor[i] <- bindag_causal_effect_backdoor_S(problem, v$w_list$Z[[i]])
  }

  return(list(problem = problem, effects = effects, effects_backdoor = effects_backdoor,
              v = v, s = s, true_witness = true_witness))
}

############################################################################################
# app_plot_summary_witness::
#
# Plots and prints output generated by app_witness_summarize_run.
#
# Input:
#
# - result_stats: output of 'app_witness_summarize_run'
# - hardness_factor: minimum full-set backdoor adjustment bias used when generating the
#                    synthetic cases

app_plot_summary_witness <- function(result_stats)
{
  tail_bias <- 0.1
  x11(); par(mfrow = c(2, 3))
  m_1 <- round(mean(result_stats$bias_naive1) * 100) /100
  e_1 <- mean(result_stats$bias_naive1 > tail_bias)
  hist(result_stats$bias_naive1, xlim = c(0, 1), xlab = "Bias", main = paste("Full adjustment, mean =", m_1))
  m_2 <- round(mean(result_stats$bias_naive2) * 100) /100
  e_2 <- mean(result_stats$bias_naive2 > tail_bias)
  hist(result_stats$bias_naive2, xlim = c(0, 1), xlab = "Bias", main = paste("No adjustment, mean =", m_2))
  m_3 <- round(mean(result_stats$bias_backdoor, na.rm = TRUE) * 100) / 100
  e_3 <- mean(result_stats$bias_backdoor > tail_bias, na.rm = TRUE)
  hist(result_stats$bias_backdoor, xlim = c(0, 1), xlab = "Bias", main = paste("Faithfulness adj., mean =", m_3))
  m_4 <- round(mean(result_stats$score_highest, na.rm = TRUE) * 100) / 100
  hist(result_stats$highest_bound_width, xlim = c(0, 1), xlab = "Width", main = paste("Highest, coverage =", m_4))
  m_5 <- round(mean(result_stats$score_loosest, na.rm = TRUE) * 100) / 100
  hist(result_stats$loosest_bound_width, xlim = c(0, 1), xlab = "Width", main = paste("Loosest, coverage =", m_5))
  m_6 <- round(mean(abs(result_stats$bias_highest_bound), na.rm = TRUE) * 100) /100
  hist(abs(result_stats$bias_highest_bound), xlim = c(0, 1), xlab = "Bias", main = paste("Highest bound bias, mean =", m_6))
  e_4 <- mean(abs(result_stats$bias_highest_bound) > tail_bias, na.rm = TRUE)
  cat("ACE reported: ", 100 * length(result_stats$ace_reported) / length(result_stats$bias_naive1), "%\n", sep = "")
  cat("Width excess: ", mean(abs(result_stats$highest_bound_width) > 2 * m_1, na.rm = TRUE), "\n", sep = "")
  cat("Median width: ", median(abs(result_stats$highest_bound_width), na.rm = TRUE), "\n", sep = "")
  cat("Tail bias [0.1]:", c(e_1, e_2, e_3, e_4), "\n")
}

############################################################################################
# app_plot_summary_witness_latexify::
#
# Produce a line of LaTeX code summarizing results. 

# Input:
#
# - result_stats: a list of output of 'app_witness_summarize_run', each entry of the list
#                 correspond to different setups. First dimension represents different 
#                 configurations for the P(U | W) factor, second dimension represents
#                 different values for the other parameters (assuming all equal)
#                 
#
# Output. A total of 10 columns, as follows:
#
#  1. Proportion of times WPP found a solution
#  2. Average error for naive estimator 1 (NE1), which conditions on all covariates
#  3. Tail error at 0.1 for NE1 (see WPP paper)
#  4. Average error for naive estimator 2 (NE2), which conditions on no covariates
#  5. Tail error at 0.1 for NE2
#  6. Average error for back-door adjustment using the same admissible set as WPP
#  7. Tail error at 0.1 for the above
#  8. Average error for WPP
#  9. Tail error at 0.1 for WPP
# 10. Width of WPP interval
#
# Everything up to 2 digits of precision.

app_summary_witness_latexify <- function(epsilons, result_stats)
{
  num_columns <- length(result_stats)
  num_rows    <- length(result_stats[[1]])
    
  tail_bias <- 0.1
  
  m_1 <- mean(result_stats[[1]][[1]]$bias_naive1)
  e_1 <- mean(result_stats[[1]][[1]]$bias_naive1 > tail_bias)    
  m_2 <- mean(result_stats[[1]][[1]]$bias_naive2)
  e_2 <- mean(result_stats[[1]][[1]]$bias_naive2 > tail_bias)

  cat(sprintf("$%1.2f$ $%1.2f$ $%1.2f$ $%1.2f$\n", m_1, e_1, m_2, e_2))
  
  for (j in 1:num_rows) {
    
    m_3 <- mean(result_stats[[1]][[j]]$bias_backdoor, na.rm = TRUE)
    e_3 <- mean(result_stats[[1]][[j]]$bias_backdoor > tail_bias, na.rm = TRUE)
    
    ACE_reported <- length(result_stats[[1]][[j]]$ace_reported) / length(result_stats[[1]][[i]]$bias_naive1)
    
    line <- c(epsilons[j], ACE_reported, m_3, e_3)

    for (i in 1:num_columns) {
      m_4 <- mean(abs(result_stats[[i]][[j]]$bias_highest_bound), na.rm = TRUE)  
      e_4 <- mean(abs(result_stats[[i]][[j]]$bias_highest_bound) > tail_bias, na.rm = TRUE)
      wpp_width <- median(abs(result_stats[[i]][[j]]$highest_bound_width), na.rm = TRUE)
      line <- c(line, m_4, e_4, wpp_width)
    }
    
    format_m <- "$%1.2f$"; cat(sprintf(format_m, line[1]))
    for (k in 2:length(line)) cat(sprintf(" & $%1.2f$", line[k]))
    cat(sprintf("\\\\ \n"))
    
  }
  
  # Example of application, using the output of app_witness_synthetic_default_run3:
  #
  # load("JMLR_synth_21_10_2014_beta_1_0_results.RData")
  # rs_hard <- result_stats_solvable_hard
  # rs_easy <- result_stats_solvable_easy
  # rn_hard <- result_stats_nsolvable_hard
  # rn_easy <- result_stats_nsolvable_easy
  #
  # load("JMLR_synth_21_10_2014_beta_1_1_results.RData")
  # rs_hard <- list(rs_hard, result_stats_solvable_hard)
  # rs_easy <- list(rs_easy, result_stats_solvable_easy)
  # rn_hard <- list(rn_hard, result_stats_nsolvable_hard)
  # rn_easy <- list(rn_easy, result_stats_nsolvable_easy)
  #
  # app_summary_witness_latexify(0.05 * (1:length(result_stats_solvable_hard)), rs_hard)
}
