############################################################################################
# LPP_tools.R
# 
# Basic methods for finding ACE bounds using linear programs. The Bayesian method developed
# in 'bayesian.R' shares a similar set of functions performed using analytical bounds.
#
# Code by 
#
#  - Ricardo Silva (ricardo@stats.ucl.ac.uk)
#  - Robin Evans (robin.evans@stats.ox.ac.uk)
#
# Current version: 15/08/2014
# First version: 07/02/2014

#library(linprog)
library(rcdd)

##################################################################################################
# get_V_eta_star::
#
# Get the feasible region for the local constraints on eta_xw^\star in terms of its extreme points.
# Basically, the extreme points corresponding to
#
# |eta_x0^star - eta_x1^star| <= epsilon[1]
# |eta_x0^star - P(Y = 1 | X = x, W = 0)| <= epsilon[2]
# |eta_x1^star - P(Y = 1 | X = x, W = 1)| <= epsilon[3]
# 0 <= eta_xw^star <= 1
#
# where Px[1] = P(Y = 1 | X = x, W = 0), Px[2] = P(Y = 1 | X = x, W = 1)
#
# We use rcdd here out of laziness, but it can be easily done analytically.

get_V_eta_star <- function(epsilon, Px)
{
  A <- rbind(c(1, -1), c(-1, 1), c(1, 0), c(-1, 0), c(0, 1), c(0, -1), c(1, 0), c(-1, 0), c(0, 1), c(0, -1))
  b <- c(epsilon[1], epsilon[1], 
         epsilon[2] + Px[1], epsilon[2] - Px[1],
         epsilon[3] + Px[2], epsilon[3] - Px[2],
         1, 0, 1, 0)
  H <- makeH(A, b, rep(0, ncol(A)), 0)
  H <- H[-1, ]
  V <- scdd(H)
  V <- V$output
  return(V[, 3:ncol(V), drop = FALSE])
}

#################################################################################################
# build_table_parameters::
#
# Build the mapping between the eta^star/delta^star parameters into the zeta^star/eta^star space.
#
# 

build_table_parameters <- function(eta_space_0, eta_space_1, delta_space_0, delta_space_1, monotonic)
{  
  num_base_pair_eta_0 <- nrow(eta_space_0)
  num_base_pair_eta_1 <- nrow(eta_space_1)
  base_pair_delta <- rbind(c(delta_space_0[1], delta_space_1[1]), 
                           c(delta_space_0[1], delta_space_1[2]), 
                           c(delta_space_0[2], delta_space_1[1]), 
                           c(delta_space_0[2], delta_space_1[2]));
  num_base_pair_delta <- nrow(base_pair_delta)
  
  # Build T1: the extreme points in eta^star/delta^star space
  
  T1 <- matrix(rep(0, num_base_pair_eta_0 * num_base_pair_eta_1 * num_base_pair_delta * 6), ncol = 6)
  t1_row <- 1;
  for (i in 1:num_base_pair_eta_0)
    for (j in 1:num_base_pair_eta_1)
      for (k in 1:num_base_pair_delta) {
        T1[t1_row, ] <- c(eta_space_0[i, ], eta_space_1[j, ], base_pair_delta[k, ])
        t1_row <- t1_row + 1
      }
  if (monotonic) {
    remove_rows <- which(T1[, 6] < T1[, 5])
    if (length(remove_rows) > 0) T1 <- T1[-remove_rows, ]
  }
  
  # Now, map to T2: the extreme points in zeta^star/eta^star space
  
  T2 <- matrix(rep(0, nrow(T1) * 12), ncol = 12);
  
  T2[, 1] <- (1 - T1[, 1]) * (1 - T1[, 5])
  T2[, 2] <- (1 - T1[, 3]) * T1[, 5]
  T2[, 3] <- T1[, 1] * (1 - T1[, 5])
  T2[, 4] <- T1[, 3] * T1[, 5]
  T2[, 5] <- (1 - T1[, 2]) * (1 - T1[, 6])
  T2[, 6] <- (1 - T1[, 4]) * T1[, 6]
  T2[, 7] <- T1[, 2] * (1 - T1[, 6])
  T2[, 8] <- T1[, 4] * T1[, 6]
  T2[, 9:12] <- T1[, 1:4]
  
  return(list(T1 = T1, T2 = T2))
}

# Simple case, straightforward IV model

build_table_parameters_simple <- function()
{
  
  base_pair <- rbind(c(0, 0), c(0, 1), c(1, 0), c(1, 1))
  num_base_pair <- nrow(base_pair)
  
  # Build T1: the extreme points in eta^star space
  
  T1 <- matrix(rep(0, num_base_pair * num_base_pair * 4), ncol = 4)
  t1_row <- 1;
  for (i in 1:num_base_pair)
    for (j in 1:num_base_pair) {
        T1[t1_row, ] = c(base_pair[i, ], base_pair[j, ])
        t1_row <- t1_row + 1
      }
  
  # Now, map to T2: the extreme points in zeta^star/eta^star space
  
  T2 <- matrix(rep(0, nrow(T1) * 10), ncol = 10);
  
  for (i in 1:nrow(T2)) {  
    T2[i, 1] <- (1 - T1[i, 1]) * (1 - T1[i, 3])
    T2[i, 2] <- (1 - T1[i, 2]) * T1[i, 3]
    T2[i, 3] <- T1[i, 1] * (1 - T1[i, 3])
    T2[i, 4] <- T1[i, 2] * T1[i, 3]
    T2[i, 5] <- (1 - T1[i, 1]) * (1 - T1[i, 4])
    T2[i, 6] <- (1 - T1[i, 2]) * T1[i, 4]
    T2[i, 7] <- T1[i, 1] * (1 - T1[i, 4])
    T2[i, 8] <- T1[i, 2] * T1[i, 4]
    T2[i, 9:10] <- T1[i, 1:2]
  }
  
  return(list(T1 = T1, T2 = T2))
}

########################################################################################################
# build_base_inequalities_extended::
#
# In the folowing case, we assume the range of the basic eta x delta space if the unit hypercube. We
# we then map them not onlt to zeta and eta, but also delta

build_base_inequalities_extended <- function()
{  
  base_pair <- rbind(c(0, 0), c(0, 1), c(1, 0), c(1, 1))
  eta_space_0 <- base_pair
  eta_space_1 <- base_pair
  delta_space <- base_pair
  
  # Build T1: the extreme points in eta^star/delta^star space
  
  T1 <- matrix(rep(0, 4 * 4 * 4 * 6), ncol = 6)
  t1_row <- 1;
  for (i in 1:4)
    for (j in 1:4)
      for (k in 1:4) {
        T1[t1_row, ] = c(eta_space_0[i, ], eta_space_1[j, ], delta_space[k, ])
        t1_row <- t1_row + 1
      }
  
  # Now, map to T2: the extreme points in zeta^star/eta^star space
  
  T2 <- matrix(rep(0, nrow(T1) * 14), ncol = 14);
  
  T2[, 1] <- (1 - T1[, 1]) * (1 - T1[, 5])
  T2[, 2] <- (1 - T1[, 3]) * T1[, 5]
  T2[, 3] <- T1[, 1] * (1 - T1[, 5])
  T2[, 4] <- T1[, 3] * T1[, 5]
  T2[, 5] <- (1 - T1[, 2]) * (1 - T1[, 6])
  T2[, 6] <- (1 - T1[, 4]) * T1[, 6]
  T2[, 7] <- T1[, 2] * (1 - T1[, 6])
  T2[, 8] <- T1[, 4] * T1[, 6]
  T2[, 9:14] <- T1
  
  # Now, transform it in a system of inequalities
  
  V <- makeV(T2)
  pre_H <- scdd(V)
  H_matrix <- pre_H$output
  ineq_idx <- which(H_matrix[,1] == 0)
  A_ineq <- -H_matrix[ineq_idx, 3:ncol(H_matrix)]
  b_ineq <- H_matrix[ineq_idx, 2]
  
  # Remove rows which make no reference to eta and delta
  
  sel_rows <- which(rowSums(abs(A_ineq[, 9:14])) > 0)
  A_ineq <- A_ineq[sel_rows, ]
  b_ineq <- b_ineq[sel_rows]
  
  return(list(A = A_ineq, b = b_ineq))
}

#################################################################################################
# build_lpp_scdd::
#
# Build LPP solution using scdd as follows (example):
#
# > T <- build_table_parameters(epsilon, c(0, 1), c(0, 1))
# > lpp_ace <- build_lpp_scdd(T$T2, LPP_SCALE, model$P_ZETA)
#
# 'scale' should be set in a way to make all coefficients reasonably representable with
# not many digits, or representation can explode (or maybe I don't know how
# to properly use scdd)
# 
# P_W and P_ZETA are vectors which respectively represent the  distributions P(W = w) and 
# P(Y = y, X = x | W = w) as in
#
# P_W = c(P(W = 0), P(W = 1))
# P_ZETA = c(P(Y = 0, X = 0 | W = 0), P(01 | 0), P(10 | 0), P(11 | 0), P(00 | 1), P(01 | 1) etc.)

build_lpp_scdd <- function(M, P_W, P_ZETA, beta_lower, beta_upper)
{
  OMEGA_POS.00 <- 9

  ETA_POS.00 <- 13
  ETA_POS.01 <- 14
  ETA_POS.10 <- 15
  ETA_POS.11 <- 16
  
  NUM_VARS <- 16
  NUM_RELAX_CONSTRAINTS <- 40
  
  # Basic setup 
  
  V <- makeV(M)
  pre_H <- c()
  try(pre_H <- scdd(V), silent = TRUE)
  if (length(pre_H) == 0) return(list())  
  H_matrix <- pre_H$output
  ineq_idx <- which(H_matrix[,1] == 0)
  eq_idx <- which(H_matrix[,1] == 1)
  if (length(ineq_idx) == 1) {
    A_ineq0 <- matrix(c(-H_matrix[ineq_idx, 3:ncol(H_matrix)], rep(0, 4)), nrow = 1)
  } else {
    A_ineq0 <- cbind(-H_matrix[ineq_idx, 3:ncol(H_matrix)], matrix(rep(0, length(ineq_idx) * 4), ncol = 4))  
  }
  b_ineq0 <- H_matrix[ineq_idx, 2]
    
  A_eq <- matrix(0, ncol = ncol(H_matrix) + 2, nrow = 2)
  A_eq[1, 1:4] <- 1 # Sum k_{yx.w0} == 1
  A_eq[2, 5:8] <- 1 # Sum k_{yx.w1} == 1 
  b_eq <- c(1, 1)
  
  # Now add the constraints related to the randomization relaxation
  
  A <- matrix(rep(0, NUM_RELAX_CONSTRAINTS * NUM_VARS), ncol = NUM_VARS)
  b <- rep(0, NUM_RELAX_CONSTRAINTS)
  pc <- 1
  
  # This relates kappa to the observables
  
  for (w in 0:1)
    for (x in 0:1)
      for (y in 0:1) {
        var_pos <- 1 + w * 4  + y * 2 + x * 1;
        A[pc, var_pos] <- 1
        b[pc] <-  P_ZETA[var_pos] / beta_lower
        pc <- pc + 1
        A[pc, var_pos] <- -1
        b[pc] <- -P_ZETA[var_pos] / beta_upper
        pc <- pc + 1
      }
  
  # This relates eta_xw to omega_xw
  
  for (w in 0:1)
    for (x in 0:1) {
      var_pos_eta <- ETA_POS.00 + w * 2 + x * 1;
      var_pos_omega <- OMEGA_POS.00 + w * 2 + x * 1;
      A[pc, var_pos_eta] <- 1
      A[pc, var_pos_omega] <- -beta_upper 
      pc <- pc + 1
      A[pc, var_pos_eta] <- -1
      A[pc, var_pos_omega] <- beta_lower
      pc <- pc + 1
    }
  
  # 0 <= eta_xw <= 1
  
  for (i in 0:3) {
    A[pc, ETA_POS.00 + i] <-  1; b[pc] <- 1; pc <- pc + 1
    A[pc, ETA_POS.00 + i] <- -1;             pc <- pc + 1
  }
  
  # 0 <= omega_xw <= 1
  
  for (i in 0:3) {
    A[pc, OMEGA_POS.00 + i] <-  1; b[pc] <- 1; pc <- pc + 1
    A[pc, OMEGA_POS.00 + i] <- -1;             pc <- pc + 1
  }
  
  A_ineq <- rbind(A_ineq0, A)
  b_ineq <- c(b_ineq0, b)  
  
  # Build objective function
  
  C <- rep(0, NUM_VARS)
  C[ETA_POS.00] <- -P_W[1]
  C[ETA_POS.01] <- -P_W[2]
  C[ETA_POS.10] <-  P_W[1]
  C[ETA_POS.11] <-  P_W[2]
  
  # Constraint encoding
    
  constr_type <- c(rep("<=", nrow(A_ineq)), rep("=", nrow(A_eq)))
  
  # Done, return information
  
  return(list(A = rbind(A_ineq, A_eq), b = c(b_ineq, b_eq), C = C, constr_type = constr_type))
  
}

#################################################################################################
# build_lpp_iv_scdd::
#
# Similar, but for the basic IV model {W -> X, X -> Y, X <- U -> Y}. For illustration purposes
# mostly. See analytical_iv for more information.

build_lpp_iv_scdd <- function(M, P_W, P_ZETA)
{
  # Basic setup 
  
  V <- makeV(M)
  pre_H <- scdd(V)
  H_matrix <- pre_H$output
  ineq_idx <- which(H_matrix[,1] == 0)
  eq_idx <- which(H_matrix[,1] == 1)
  A_ineq <- -H_matrix[ineq_idx, 3:ncol(H_matrix)]
  b_ineq <- H_matrix[ineq_idx, 2]
  A_eq0 <- -H_matrix[eq_idx, 3:ncol(H_matrix)]
  b_eq0 <- H_matrix[eq_idx, 2]
  
  # Introduce data: add values as equality constraints
  
  A_eq <- matrix(rep(0, 8 * 10), ncol = 10)
  b_eq <- rep(0, 8)
  constr_type_eq <- rep("=", 8)
  for (i in 1:8) {
    A_eq[i, i] <- 1
    b_eq[i] <- P_ZETA[i]
  }
  A_eq <- rbind(A_eq0, A_eq)
  b_eq <- c(b_eq0, b_eq)    
  
  # Objective function
  
  C <- rep(0, 10)
  C[9] <- -1
  C[10] <- 1  
  
  # Constraint encoding
  
  constr_type <- c(rep("<=", nrow(A_ineq)), rep("=", nrow(A_eq)))
  
  # Done, return information
  
  return(list(A = rbind(A_ineq, A_eq), b = c(b_ineq, b_eq), C = C, constr_type = constr_type))
  
}

################################################################################################
# analytical_iv::
#
# This uses the analytical solution for the basic IV model {W -> X, X -> Y, X <- U -> Y}, 
# U latent, ACE given by P(Y = 1 | do(X = 1)) - P(Y = 1 | do(X = 0)).
#
# * Input:
#
# - P_ZETA: 8-dimensional vector of probabilitys, where P_ZETA[1:4] is P(YX | W = 0), and
#           entries 1, 2, 3, 4 correspond to YX = {00, 01, 10, 11}. Analogously,
#           P_ZETA[1:4] is P(YX | W = 1).
#
# * Output:
#
# - bottom, upper: the bounds on the ACE

analytical_iv <- function(P_ZETA)
{
   w_0_upper <- min(1 - P_ZETA[1], 
                    1 - P_ZETA[5], 
                    P_ZETA[2] + P_ZETA[3] + P_ZETA[7] + P_ZETA[8],
                    P_ZETA[3] + P_ZETA[4] + P_ZETA[6] + P_ZETA[7])
   w_0_bottom <- max(P_ZETA[7],
                     P_ZETA[3],
                     P_ZETA[3] + P_ZETA[4] - P_ZETA[5] - P_ZETA[8],
                     -P_ZETA[1] - P_ZETA[4] + P_ZETA[7] + P_ZETA[8])
   w_1_upper <- min(1 - P_ZETA[6],
                    1 - P_ZETA[2],
                    P_ZETA[3] + P_ZETA[4] + P_ZETA[5] + P_ZETA[8],
                    P_ZETA[1] + P_ZETA[4] + P_ZETA[7] + P_ZETA[8])
   w_1_bottom <- max(P_ZETA[8],
                     P_ZETA[4],
                     -P_ZETA[2] - P_ZETA[3] + P_ZETA[7] + P_ZETA[8],
                     P_ZETA[3] + P_ZETA[4] - P_ZETA[6] - P_ZETA[7])
  return(list(upper = w_1_upper - w_0_bottom, bottom = w_1_bottom - w_0_upper))
}

