/**
 * Description: This file implements the EP updates of the horseshoe bayesian model for feature selection 
 *             
 * Author: Daniel Hernández Lobato
 *
 * Year: 2013
*/


#include <R.h>
#include <Rinternals.h>
#include <Rdefines.h>
#include <Rmath.h>
#include <R_ext/Rdynload.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <gsl/gsl_integration.h>
#include <R_ext/Applic.h>

typedef struct {
	double mWOld;
	double vWOld;
	double vUOld;
	double vVOld;
	} Params;

/* Auxiliary function to get the list element named str, or return NULL */

SEXP getListElementNu(SEXP list, const char *str) {

	SEXP elmt = R_NilValue, names = getAttrib(list, R_NamesSymbol);
	int i;

	for (i = 0; i < length(list); i++)
		if(strcmp(CHAR(STRING_ELT(names, i)), str) == 0) {
			elmt = VECTOR_ELT(list, i);
			break;
		}
	
	return elmt;
}

/**
 * Function which computes the normalization constant.
 *
 */

void computeNormalizationConstantAndDerviativesInternal(double mWOld, double vWOld, double vUOld, double vVOld, 
	double *Z, double *dZdmWOld, double *dZdvWOld, double *dZdvUOld, double *dZdvVOld, int nSplits) {

	int i;
	double maxb, minb, hb, xb, tan_value, value_tmp, value_tmp_2;
	double unNormdensityValue, weight, factor;

	maxb = 1;
	minb = 0;
	hb = (maxb - minb) / (nSplits * 4);
	
	*Z = *dZdmWOld = *dZdvWOld = *dZdvUOld = *dZdvVOld = 0;
	xb = minb;

	weight = 7 * hb * 2.0 / 45;
	tan_value = tan(PI * (xb - 0.5));
	unNormdensityValue = dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
	*Z += unNormdensityValue * weight;
	value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
	*dZdmWOld += - value_tmp * unNormdensityValue * weight;
	value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));
	*dZdvWOld +=  value_tmp_2 * unNormdensityValue * weight;
	*dZdvUOld +=  tan_value * tan_value / vVOld * value_tmp_2 * unNormdensityValue * weight;

	xb += hb;

	for (i = 0 ; i < nSplits ; i++) {

		weight = 32 * hb * 2.0 / 45;
		tan_value = tan(PI * (xb - 0.5));
		unNormdensityValue = dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
		*Z += unNormdensityValue * weight;
		value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
		*dZdmWOld += - value_tmp * unNormdensityValue * weight;
		value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));
		*dZdvWOld +=  value_tmp_2 * unNormdensityValue * weight;
		*dZdvUOld +=  tan_value * tan_value / vVOld * value_tmp_2 * unNormdensityValue * weight;
		xb += hb;

		weight = 12 * hb * 2.0 / 45;
		tan_value = tan(PI * (xb - 0.5));
		unNormdensityValue = dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
		*Z += unNormdensityValue * weight;
		value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
		*dZdmWOld += - value_tmp * unNormdensityValue * weight;
		value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));
		*dZdvWOld +=  value_tmp_2 * unNormdensityValue * weight;
		*dZdvUOld +=  tan_value * tan_value / vVOld * value_tmp_2 * unNormdensityValue * weight;
		xb += hb;

		weight = 32 * hb * 2.0 / 45;
		tan_value = tan(PI * (xb - 0.5));
		unNormdensityValue = dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
		*Z += unNormdensityValue * weight;
		value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
		*dZdmWOld += - value_tmp * unNormdensityValue * weight;
		value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));
		*dZdvWOld +=  value_tmp_2 * unNormdensityValue * weight;
		*dZdvUOld +=  tan_value * tan_value / vVOld * value_tmp_2 * unNormdensityValue * weight;
		xb += hb;

		if (i != nSplits - 1)
			factor = 2;
		else
			factor = 1;

		weight = 7 * hb * 2.0 / 45;
		tan_value = tan(PI * (xb - 0.5));
		unNormdensityValue = dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
		*Z += factor * unNormdensityValue * weight;
		value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
		*dZdmWOld += - factor * value_tmp * unNormdensityValue * weight;
		value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));
		*dZdvWOld +=  factor * value_tmp_2 * unNormdensityValue * weight;
		*dZdvUOld +=  factor * tan_value * tan_value / vVOld * value_tmp_2 * unNormdensityValue * weight;
		xb += hb;
	}

	*dZdvVOld = - (*dZdvUOld) * vUOld / vVOld;
}


/**
 * Function which refines the likelihood factors.
 *
 */

SEXP computeNormalizationConstantAndMoments(SEXP R_m) {

	/* Auxiliary variables */

	int i;

	/* Local points to the variables stored in R_m */

	double *mWOld;
	double *vWOld;
	double *vUOld;
	double *vVOld;
	double *Z;
	double *dZdmWOld;
	double *dZdvWOld;
	double *dZdvUOld;
	double *dZdvVOld;
	int *computed;
	int d;

	/* We access the elements from the list */

	mWOld = NUMERIC_POINTER(getListElementNu(R_m, "mWOld"));
	vWOld = NUMERIC_POINTER(getListElementNu(R_m, "vWOld"));
	vUOld = NUMERIC_POINTER(getListElementNu(R_m, "vUOld"));
	vVOld = NUMERIC_POINTER(getListElementNu(R_m, "vVOld"));
	dZdmWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdmWOld"));
	dZdvWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvWOld"));
	dZdvUOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvUOld"));
	dZdvVOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvVOld"));
	Z = NUMERIC_POINTER(getListElementNu(R_m, "Z"));
	computed = INTEGER_POINTER(getListElementNu(R_m, "computed"));
	d = *INTEGER_POINTER(getListElementNu(R_m, "d"));

	/* Main loop */

	for (i = 0 ; i < d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			computeNormalizationConstantAndDerviativesInternal(mWOld[ i ], vWOld[ i ], vUOld[ i ], vVOld[ i ], 
				&Z[ i ], &dZdmWOld[ i ], &dZdvWOld[ i ], &dZdvUOld[ i ], &dZdvVOld[ i ], 500);

			computed[ i ] = 1;
		}
	}

	return R_m;
}


/**
 * Function which refines the likelihood factors.
 *
 */

void computeNormalizationConstantAndMomentsLowLevel(double *mWOld, double *vWOld, 
	double *vUOld, double *vVOld, double *Z, double *dZdmWOld, double *dZdvWOld, double *dZdvUOld, double *dZdvVOld,
	int *computed, int *d) {

	/* Auxiliary variables */

	int i;

	/* Local points to the variables stored in R_m */

	/* We access the elements from the list */
	/* Main loop */

	for (i = 0 ; i < *d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			computeNormalizationConstantAndDerviativesInternal(mWOld[ i ], vWOld[ i ], vUOld[ i ], vVOld[ i ], 
				&Z[ i ], &dZdmWOld[ i ], &dZdvWOld[ i ], &dZdvUOld[ i ], &dZdvVOld[ i ], 500);

			computed[ i ] = 1;
		}
	}
}

double f_z(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = tan(PI * (x - 0.5));

	return dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
}

double f_dZdmWOld(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = tan(PI * (x - 0.5));

	return - mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value) *
			dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
}

double f_dZdvWOld(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, value_tmp, value_tmp_2;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = tan(PI * (x - 0.5));
	value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
	value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));

	return value_tmp_2 * dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
}

double f_dZdvUOld(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, value_tmp, value_tmp_2;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = tan(PI * (x - 0.5));
	value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
	value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));

	return tan_value * tan_value / vVOld * value_tmp_2 * dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value)) * (1.0 + tan_value * tan_value) * PI;
}


/**
 * Function which refines the likelihood factors.
 *
 */

SEXP computeNormalizationConstantAndMomentsGSL(SEXP R_m) {

	/* Auxiliary variables */


	/* Local points to the variables stored in R_m */

	double *mWOld;
	double *vWOld;
	double *vUOld;
	double *vVOld;
	double *Z;
	double *dZdmWOld;
	double *dZdvWOld;
	double *dZdvUOld;
	double *dZdvVOld;
	int *computed;
	double error;
	Params params;
	gsl_function My_function;
	gsl_integration_workspace *work_ptr = gsl_integration_workspace_alloc (1e4);
	int d, i;

	/* We access the elements from the list */

	mWOld = NUMERIC_POINTER(getListElementNu(R_m, "mWOld"));
	vWOld = NUMERIC_POINTER(getListElementNu(R_m, "vWOld"));
	vUOld = NUMERIC_POINTER(getListElementNu(R_m, "vUOld"));
	vVOld = NUMERIC_POINTER(getListElementNu(R_m, "vVOld"));
	dZdmWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdmWOld"));
	dZdvWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvWOld"));
	dZdvUOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvUOld"));
	dZdvVOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvVOld"));
	Z = NUMERIC_POINTER(getListElementNu(R_m, "Z"));
	computed = INTEGER_POINTER(getListElementNu(R_m, "computed"));
	d = *INTEGER_POINTER(getListElementNu(R_m, "d"));

	/* Auxiliary variables */

	/* Local points to the variables stored in R_m */

	/* We access the elements from the list */
	/* Main loop */

	for (i = 0 ; i < d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			params.mWOld = mWOld[ i ];
			params.vWOld = vWOld[ i ];
			params.vUOld = vUOld[ i ];
			params.vVOld = vVOld[ i ];
			
			My_function.params = (void *) &params;

			My_function.function = &f_z;
			gsl_integration_qags (&My_function, 0.0, 1.0, 1e-12, 1e-12, 1e4, work_ptr, &Z[ i ], &error);

			My_function.function = &f_dZdmWOld;
			gsl_integration_qags (&My_function, 0.0, 1.0, 1e-12, 1e-12, 1e4, work_ptr, &dZdmWOld[ i ], &error);

			My_function.function = &f_dZdvWOld;
			gsl_integration_qags (&My_function, 0.0, 1.0, 1e-12, 1e-12, 1e4, work_ptr, &dZdvWOld[ i ], &error);

			My_function.function = &f_dZdvUOld;
			gsl_integration_qags (&My_function, 0.0, 1.0, 1e-12, 1e-12, 1e4, work_ptr, &dZdvUOld[ i ], &error);

			dZdvVOld[ i ] = - dZdvUOld[ i ] * vUOld[ i ] / vVOld[ i ];

			computed[ i ] = 1;
		}
	}

	gsl_integration_workspace_free(work_ptr);

	return R_m;
}


double f_z_no_transform(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;

	return dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 1.0 / (PI * (1.0 + tan_value * tan_value));
}

double f_dZdmWOld_no_transform(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;

	return - mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value) *
			dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value));
}

double f_dZdvWOld_no_transform(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, value_tmp, value_tmp_2;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;
	value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
	value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));

	return value_tmp_2 * dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value));
}

double f_dZdvUOld_no_transform(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, value_tmp, value_tmp_2;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;
	value_tmp = mWOld / (vWOld + vUOld / vVOld * tan_value * tan_value);
	value_tmp_2 = 0.5 * (value_tmp * value_tmp - 1.0 / (vWOld + vUOld / vVOld * tan_value * tan_value));

	return tan_value * tan_value / vVOld * value_tmp_2 * dnorm(mWOld, 0, sqrt(vWOld + vUOld / vVOld * tan_value * tan_value), 0) * 
			1.0 / (PI * (1.0 + tan_value * tan_value));
}


/**
 * Function which refines the likelihood factors.
 *
 */

SEXP computeNormalizationConstantAndMomentsGSLnoTransform(SEXP R_m) {

	/* Auxiliary variables */


	/* Local points to the variables stored in R_m */

	double *mWOld;
	double *vWOld;
	double *vUOld;
	double *vVOld;
	double *Z;
	double *dZdmWOld;
	double *dZdvWOld;
	double *dZdvUOld;
	double *dZdvVOld;
	int *computed;
	double error;
	Params params;
	gsl_function My_function;
	gsl_integration_workspace *work_ptr = gsl_integration_workspace_alloc (1e4);
	int d, i;

	/* We access the elements from the list */

	mWOld = NUMERIC_POINTER(getListElementNu(R_m, "mWOld"));
	vWOld = NUMERIC_POINTER(getListElementNu(R_m, "vWOld"));
	vUOld = NUMERIC_POINTER(getListElementNu(R_m, "vUOld"));
	vVOld = NUMERIC_POINTER(getListElementNu(R_m, "vVOld"));
	dZdmWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdmWOld"));
	dZdvWOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvWOld"));
	dZdvUOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvUOld"));
	dZdvVOld = NUMERIC_POINTER(getListElementNu(R_m, "dZdvVOld"));
	Z = NUMERIC_POINTER(getListElementNu(R_m, "Z"));
	computed = INTEGER_POINTER(getListElementNu(R_m, "computed"));
	d = *INTEGER_POINTER(getListElementNu(R_m, "d"));

	/* Auxiliary variables */

	/* Local points to the variables stored in R_m */

	/* We access the elements from the list */
	/* Main loop */

	for (i = 0 ; i < d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			params.mWOld = mWOld[ i ];
			params.vWOld = vWOld[ i ];
			params.vUOld = vUOld[ i ];
			params.vVOld = vVOld[ i ];
			
			My_function.params = (void *) &params;

			My_function.function = &f_z_no_transform;
			gsl_integration_qagi (&My_function, 1e-12, 1e-12, 1e4, work_ptr, &Z[ i ], &error);

			My_function.function = &f_dZdmWOld_no_transform;
			gsl_integration_qagi (&My_function, 1e-12, 1e-12, 1e4, work_ptr, &dZdmWOld[ i ], &error);

			My_function.function = &f_dZdvWOld_no_transform;
			gsl_integration_qagi (&My_function, 1e-12, 1e-12, 1e4, work_ptr, &dZdvWOld[ i ], &error);

			My_function.function = &f_dZdvUOld_no_transform;
			gsl_integration_qagi (&My_function, 1e-12, 1e-12, 1e4, work_ptr, &dZdvUOld[ i ], &error);

			dZdvVOld[ i ] = - dZdvUOld[ i ] * vUOld[ i ] / vVOld[ i ];

			computed[ i ] = 1;
		}
	}

	gsl_integration_workspace_free(work_ptr);

	return R_m;
}


double f_z_same_R_code(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;

	return dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value));
}

double f_dZdvUOld_same_R_code(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, tmp_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;
	tmp_value = 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value));

	return dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * - tmp_value * tmp_value * PI / vVOld;
}

double f_Ew2_same_R_code(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, tmp_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;
	tmp_value = (1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld)) * mWOld / vWOld;

	return dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value)) * 
		(tmp_value * tmp_value + 1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld));
}

double f_Ew_same_R_code(double x, void *par) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
		
	params = (Params *) par;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	tan_value = x;

	return dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value)) * 
		1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld)  * mWOld / vWOld;
}


/**
 * Function which refines the likelihood factors.
 *
 */

SEXP computeNormalizationConstantAndMomentsGSLnoTransformSameRCode(SEXP R_m) {

	/* Auxiliary variables */


	/* Local points to the variables stored in R_m */

	double *mWOld;
	double *vWOld;
	double *vUOld;
	double *vVOld;
	double *Z;
	double *Ew;
	double *Ew2;
	double *dlogZdvUOld;
	int *computed;
	double error;
	Params params;
	gsl_function My_function;
	gsl_integration_workspace *work_ptr = gsl_integration_workspace_alloc (1e4);
	int d, i;

	/* We access the elements from the list */

	mWOld = NUMERIC_POINTER(getListElementNu(R_m, "mWOld"));
	vWOld = NUMERIC_POINTER(getListElementNu(R_m, "vWOld"));
	vUOld = NUMERIC_POINTER(getListElementNu(R_m, "vUOld"));
	vVOld = NUMERIC_POINTER(getListElementNu(R_m, "vVOld"));
	Ew = NUMERIC_POINTER(getListElementNu(R_m, "Ew"));
	Ew2 = NUMERIC_POINTER(getListElementNu(R_m, "Ew2"));
	dlogZdvUOld = NUMERIC_POINTER(getListElementNu(R_m, "dlogZdvUOld"));
	Z = NUMERIC_POINTER(getListElementNu(R_m, "Z"));
	computed = INTEGER_POINTER(getListElementNu(R_m, "computed"));
	d = *INTEGER_POINTER(getListElementNu(R_m, "d"));

	/* Auxiliary variables */

	/* Local points to the variables stored in R_m */

	/* We access the elements from the list */
	/* Main loop */

	for (i = 0 ; i < d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			params.mWOld = mWOld[ i ];
			params.vWOld = vWOld[ i ];
			params.vUOld = vUOld[ i ];
			params.vVOld = vVOld[ i ];
			
			My_function.params = (void *) &params;

			My_function.function = &f_z_same_R_code;
			gsl_integration_qagi (&My_function, 1e-10, 1e-10, 1e4, work_ptr, &Z[ i ], &error);
			Z[ i ] *= sqrt(vUOld[ i ] / vVOld[ i ]); 

			My_function.function = &f_dZdvUOld_same_R_code;
			gsl_integration_qagi (&My_function, 1e-10, 1e-10, 1e4, work_ptr, &dlogZdvUOld[ i ], &error);
			dlogZdvUOld[ i ] *= sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ];
			dlogZdvUOld[ i ] += 0.5 / vUOld[ i ];

			My_function.function = &f_Ew2_same_R_code;
			gsl_integration_qagi (&My_function, 1e-10, 1e-10, 1e4, work_ptr, &Ew2[ i ], &error);
			Ew2[ i ] *= sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ];

			My_function.function = &f_Ew_same_R_code;
			gsl_integration_qagi (&My_function, 1e-10, 1e-10, 1e4, work_ptr, &Ew[ i ], &error);
			Ew[ i ] *= sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ];

			computed[ i ] = 1;
		}
	}

	gsl_integration_workspace_free(work_ptr);

	return R_m;
}


/**
 This is the code used to call the R integrate function which is the one that works
 */

void f_z_same_R_code_internal_call(double *x, int n, void *ex) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
	int i;
		
	params = (Params *) ex;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	for (i = 0; i < n; i++) {
		tan_value = x[ i ];
		x[ i ] = dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value));
	}
}

void f_dZdvUOld_same_R_code_internal_call(double *x, int n, void *ex) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, tmp_value;
	int i;
		
	params = (Params *) ex;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	
	for (i = 0; i < n; i++) {
		tan_value = x[ i ];
		tmp_value = 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value));
		x[ i ] = dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * - tmp_value * tmp_value * PI / vVOld;
	}
}

void f_Ew2_same_R_code_internal_call(double *x, int n, void *ex) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value, tmp_value;
	int i;
		
	params = (Params *) ex;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	for (i = 0 ; i < n; i++) {
		tan_value = x[ i ];
		tmp_value = (1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld)) * mWOld / vWOld;
		x[ i ] = dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value)) * 
			(tmp_value * tmp_value + 1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld));
	}
}

void f_Ew_same_R_code_internal_call(double *x, int n, void *ex) {
	
	Params *params;
	double mWOld, vWOld, vVOld, vUOld, tan_value;
	int i;
		
	params = (Params *) ex;
	mWOld = params->mWOld;
	vWOld = params->vWOld;
	vVOld = params->vVOld;
	vUOld = params->vUOld;

	for (i = 0 ; i < n; i++) {
		tan_value = x[ i ];
		x[ i ] = dnorm(mWOld, 0, sqrt(vWOld + tan_value * tan_value), 0) * 1.0 / (PI * (vUOld / vVOld + tan_value * tan_value)) * 
		1.0 / (1.0 / (tan_value * tan_value) + 1.0 / vWOld)  * mWOld / vWOld;
	}
}

/**
 * Function which refines the likelihood factors.
 *
 */

SEXP computeNormalizationConstantAndMomentsGSLnoTransformSameRCodeInternalCall(SEXP R_m) {

	/* Auxiliary variables */

	/* Local points to the variables stored in R_m */

	double *mWOld;
	double *vWOld;
	double *vUOld;
	double *vVOld;
	double *Z;
	double *Ew;
	double *Ew2;
	double *dlogZdvUOld;
	int *computed;
	double error;
	Params params;
	int d, i;

	double bound, epsabs, epsrel, result, abserr, *work;
	int inf, neval, ier, limit, lenw, last, *iwork;

	/* We access the elements from the list */

	mWOld = NUMERIC_POINTER(getListElementNu(R_m, "mWOld"));
	vWOld = NUMERIC_POINTER(getListElementNu(R_m, "vWOld"));
	vUOld = NUMERIC_POINTER(getListElementNu(R_m, "vUOld"));
	vVOld = NUMERIC_POINTER(getListElementNu(R_m, "vVOld"));
	Ew = NUMERIC_POINTER(getListElementNu(R_m, "Ew"));
	Ew2 = NUMERIC_POINTER(getListElementNu(R_m, "Ew2"));
	dlogZdvUOld = NUMERIC_POINTER(getListElementNu(R_m, "dlogZdvUOld"));
	Z = NUMERIC_POINTER(getListElementNu(R_m, "Z"));
	computed = INTEGER_POINTER(getListElementNu(R_m, "computed"));
	d = *INTEGER_POINTER(getListElementNu(R_m, "d"));


	/* Auxiliary variables */

	/* Local points to the variables stored in R_m */

	/* We access the elements from the list */
	/* Main loop */

	for (i = 0 ; i < d ; i++) {

		computed[ i ] = 0;

		if (vWOld[ i ] > 0 && vVOld[ i ]  > 0 && vUOld[ i ] > 0) {

			/* We compute the normalization contant */

			params.mWOld = mWOld[ i ];
			params.vWOld = vWOld[ i ];
			params.vUOld = vUOld[ i ];
			params.vVOld = vVOld[ i ];

			bound = 0;
			inf = 1;
			epsabs = 1e-12;
			epsrel = 1e-12;
			limit = 1000;
			lenw = 4 * limit;
			iwork = (int *) malloc(limit * sizeof(int));
			work = (double *) malloc(lenw * sizeof(double));

			Rdqagi(f_z_same_R_code_internal_call, (void *) &params, &bound, &inf,
				&epsabs, &epsrel, &result, &abserr, &neval, &ier, &limit, &lenw, &last, iwork, work);
			Z[ i ] = 2 * result * sqrt(vUOld[ i ] / vVOld[ i ]); 

			Rdqagi(f_dZdvUOld_same_R_code_internal_call, (void *) &params, &bound, &inf,
				&epsabs, &epsrel, &result, &abserr, &neval, &ier, &limit, &lenw, &last, iwork, work);
			dlogZdvUOld[ i ] = 2 * result * sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ] + 0.5 / vUOld[ i ];

			Rdqagi(f_Ew2_same_R_code_internal_call, (void *) &params, &bound, &inf,
				&epsabs, &epsrel, &result, &abserr, &neval, &ier, &limit, &lenw, &last, iwork, work);
			Ew2[ i ] = 2 * result * sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ];

			Rdqagi(f_Ew_same_R_code_internal_call, (void *) &params, &bound, &inf,
				&epsabs, &epsrel, &result, &abserr, &neval, &ier, &limit, &lenw, &last, iwork, work);
			Ew[ i ] = 2 * result * sqrt(vUOld[ i ] / vVOld[ i ]) / Z[ i ];

			free(iwork);
			free(work);
			
			computed[ i ] = 1;
		}
	}

	return R_m;
}


