%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% get_entropy_sets:
%
% Cache the calculation of conditional entropies for all possible subsets
% of parents.
%
% Input:
%
% - L, S: parameters of the underlying probit SEM
% - entropy_order: order to be followed
%
% Output:
%
% - entropy_sets: cell structure listing relevant entropy set information
%
% Created by: Ricardo Silva, London, 05/05/2011
% University College London
% Current version: 05/05/2011

function entropy_sets = get_entropy_sets(L, S, entropy_order)

[num_y num_x] = size(L); num_x = num_x - 1;
entropy_sets = cell(num_y, 1);

m_all = L(:, end);
V_all = L(:, 1:num_x) * S * L(:, 1:num_x)' + eye(num_y);

for y = 1:num_y

  % Preliminaries
  
  entropy_sets{y}.parents = entropy_order.parents{y};  

  % First, compute contingency table for all combinations of values of y and
  % its parents
  
  y_set = [y entropy_sets{y}.parents];
  m = m_all(y_set);
  V = V_all(y_set, y_set);
  a = zeros(length(y_set), 1); b = zeros(length(y_set), 1);

  set_size = length(y_set);
  num_comb_v = 2^set_size;
  table = zeros(num_comb_v, 1);
  set_value = zeros(set_size, 1);
  
  for i = 1:num_comb_v
    a(set_value == 0) = -Inf; a(set_value == 1) = -m(set_value == 1);
    b(set_value == 0) = -m(set_value == 0); b(set_value == 1) = Inf;
    table(i) = qscmvnv(5000, V, a, eye(set_size), b);
    set_value = advance_bits(set_value);  
  end
  
  % Given the contingency table, now calculate the corresponding entropy
  % terms for every possible subset of parents
  
  num_p = length(entropy_sets{y}.parents);
  num_comb_p = 2^num_p;  
  p_sel = zeros(num_p, 1);
  
  entropy_sets{y}.value = zeros(num_comb_p, 1);
  
  for i = 1:num_comb_p % All subsets of parents
    
    % Preliminaries    
    
    y_subset      = [y entropy_sets{y}.parents(p_sel == 1)];
    y_subset_bin  = [true (p_sel == 1)'];
    y_subset_nbin = [false (p_sel == 0)'];
    subset_size   = length(y_subset);    
    num_comb_sel  = 2^subset_size;
    
    nonsel_set = entropy_sets{y}.parents(p_sel == 0);
    num_comb_nonsel = 2^length(nonsel_set);
    
    % Marginalize the excluded parents
    
    marginals = zeros(num_comb_sel, 1);
    subset_value = zeros(subset_size, 1);
    for j = 1:num_comb_sel      
      nonsel_value = zeros(length(nonsel_set), 1);
      for k = 1:num_comb_nonsel
        full_value = zeros(1, set_size);
        full_value(y_subset_bin)  = subset_value;
        full_value(y_subset_nbin) = nonsel_value;
        marginals(j) = marginals(j) + table(parent_entry(full_value));
        nonsel_value = advance_bits(nonsel_value);
      end
      subset_value = advance_bits(subset_value);
    end
    
    % Calculate conditional entropy: iterate through all values
    
    num_p_sel = 2^sum(p_sel);
    p_value = zeros(1, sum(p_sel));
    for j = 1:num_p_sel
       idx = zeros(1, subset_size);
       idx(1) = 1; idx(2:end) = p_value;
       marg_value_1 = marginals(parent_entry(idx));  
       idx(1) = 0;
       marg_value_2 = marginals(parent_entry(idx));  
       sum_marg = marg_value_1 + marg_value_2;
       entropy_sets{y}.value(i) = entropy_sets{y}.value(i) ...
            - marg_value_1 * (log(marg_value_1) - log(sum_marg)) ...
            - marg_value_2 * (log(marg_value_2) - log(sum_marg));
       p_value = advance_bits(p_value);
    end
    
    % Advance
    
    p_sel = advance_bits(p_sel);
    
  end
    
end
