function setup = setup_redwood()

setup.num_bins = 25;

[setup.X setup.Y] = get_redwood_data(setup.num_bins);

setup.runs           = 10;
setup.iterations     = 5000;
setup.burn           = 1000;
setup.ess_iterations = 10;
setup.max_ls         = 5.0;
setup.min_ls         = 1/setup.num_bins;
setup.max_gain       = 10.0;
setup.min_gain       = 1e-5;
setup.min_gpmean     = -10;
setup.max_gpmean     = 10;
setup.max_aux_std    = 200;

jitter    = 1e-6;
gpml_covs = {'covSum', {'covSEard', 'covNoise'}};

setup.slice_width     = 10;
setup.llh_fn          = @redwood_llh;
setup.cov_fn          = @(theta) feval(gpml_covs{:}, [theta ; 0 ; log(jitter)], setup.X);
setup.theta_log_prior = @(theta) log(1.0*all((theta>log(setup.min_ls)) & (theta<log(setup.max_ls))));
setup.aux_noise_fn    = @aux_noise;
setup.aux_taylor_fn   = @aux_taylor;
setup.update_gain     = @update_gain;
setup.update_mean     = @update_mean;

  function llh = redwood_llh(ff, gain, gp_mean)
    means = exp(gain*ff + gp_mean);
    llh   = sum( setup.Y .* log(means) - means - gammaln(setup.Y+1) );
  end

  function [std gg] = aux_noise(theta, K, gain, gp_mean)
    [std gg] = poiss_aux(setup.Y, 2*log(gain), gp_mean);
    std = std/gain;
    std(std > setup.max_aux_std) = setup.max_aux_std;
    gg  = (gg-gp_mean)/gain;
  end

  function [std gg] = aux_taylor(theta, K, gain, gp_mean)
    [std gg] = poiss_aux_fixed(setup.Y);
    std = std/gain;
    std(std > setup.max_aux_std) = setup.max_aux_std;
    gg  = (gg-gp_mean)/gain;
  end

  function [gain llh] = update_gain(gain, ff, cur_mean, cur_llh)
    
    % Slice sample
    particle = struct('pos', gain, 'ff', ff, 'mean', cur_mean);
    particle = gain_slice_fn(particle, -Inf);
    particle = slice_sweep(particle, @gain_slice_fn, 1, 0);
    gain     = particle.pos;
    llh      = particle.Lpstar;
  end

  function [new_mean llh] = update_mean(cur_mean, ff, gain, cur_llh)
    
    % Slice sample
    particle = struct('pos', cur_mean, 'ff', ff, 'gain', gain);
    particle = mean_slice_fn(particle, -Inf);
    particle = slice_sweep(particle, @mean_slice_fn, 1, 0);
    new_mean = particle.pos;
    llh      = particle.Lpstar;
  end

  function pp = gain_slice_fn(pp, Lpstar_min)
    
    gain = pp.pos;
    
    if (gain < setup.min_gain) || (gain > setup.max_gain)
      pp.Lpstar   = -Inf;
      pp.on_slice = false;
      return;
    end
    
    pp.Lpstar   = redwood_llh(pp.ff, gain, pp.mean);
    pp.on_slice = (pp.Lpstar >= Lpstar_min);
  end

  function pp = mean_slice_fn(pp, Lpstar_min)
    
    new_mean = pp.pos;
    
    if (new_mean < setup.min_gpmean) || (new_mean > setup.max_gpmean)
      pp.Lpstar   = -Inf;
      pp.on_slice = false;
      return;
    end
    
    pp.Lpstar   = redwood_llh(pp.ff, pp.gain, new_mean);
    pp.on_slice = (pp.Lpstar >= Lpstar_min);
  end

end