% Copyright (C) 2009  Arno Onken
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% -------------------------------------------------------------------------
% Cumulative distribution function of an empirical distribution over
% integer values.
%
% Arguments:
%  x      - n-by-d matrix of random samples
%  lambda - d-by-y matrix of PMF values; lambda(d, 1) is for the zero
%           values; sum (lambda(k, :)) == 1 sould hold for any k in
%           {1, ..., d}.
%
% Returns:
%  p      - n-by-d vector of cumulative probabilties
% -------------------------------------------------------------------------
function p = discempcdf (x, lambda)

    % Check consistency of arguments
    if (nargin ~= 2)
        error ('discempcdf: usage p = discempcdf (x, lambda)');
    end

    if (size (lambda, 1) ~= size (x, 2))
        error ('discempcdf: dimensions of lambda and x not consistent');
    end

    % Bring x into the right format
    x = floor (x);
    x(x > length (lambda)) = length (lambda);
    x(x < 0) = 0;

    % Cumulative sum for cumulative probabilities
    lambda = cumsum (lambda, 2);

    p = zeros (size (x));
    for j = 1:size (x, 2)
        p(:, j) = lambda(j, x(:, j) + 1);
    end

end