# Geometric Manifold Component Estimator (GEOMANCER)

This package provides an implementation of the Geometric Manifold Component
Estimator, or GEOMANCER, as supplementary material for "Disentangling by
Subspace Diffusion". The code provided can run all the experiments in Fig. 5,
as well as the first two columns of Table 1, and the evaluation metric used
in Table 1. We do not include the experiments with scaled or linearly distorted
data, nor do we include experiments with Laplacian Eigenmaps embeddings,
beta-VAE embeddings, or direct from pixels. We hope that this is enough for the
reviewers to resolve any questions about implementation details of GEOMANCER,
as well as verify the results on synthetic manifolds for themselves.


## Installation

To install the package locally, just run `pip install -e .` from this folder.

## Example

To run, simply load or generate an array of data, and call the `fit` function:

```
import numpy as np
import geomancer

# Generate data from a product of two spheres
data = []
for i in range(2):
  foo = np.random.randn(1000, 3)
  data.append(foo / np.linalg.norm(foo, axis=1, keepdims=True))
data = np.concatenate(data, axis=1)

# Run GEOMANCER. The underlying manifold is 4-dimensional.
components, spectrum = geomancer.fit(data, 4)
```

If ground truth information about the tangent spaces is available in a space
that is aligned with the data, then the performance can be evaluated using the
`eval_aligned` function. If ground truth data is only available in an unaligned
space, for instance if the embedding used to generate the data is not the same
as the space in which the data is observed, then the `eval_unaligned` function
can be used, which requires both the data and disentangled tangent vectors in
the ground truth space. Examples of both evaluation metrics are given in the
demo in `train.py`.


## Demo on Synthetic Manifolds

The file `train.py` runs GEOMANCER on a product of manifolds that can be
specified by the user. The number of data points to train on is given by the
`--npts` flag, while the specification of the manifold is given by the
`--specification` flag. The `--rotate` flag specifies whether a random rotation
should be applied to the data. If false, `eval_aligned` will be used to evaluate
the result. If true, `eval_unaligned` will be used to evaluate the result.

For instance, to run on the product of the sphere in 2 and 4 dimensions and the
special orthogonal group in 3 dimensions, run:

```
python3 train.py --specification='S^2','S^4','SO(3)'
```

This passes a list of strings as the manifold specification flag.

The demo plots 3 different outputs:
1. The eigenvalue spectrum of the 2nd-order graph connection Laplacian. This
should have a large gap in the spectrum at the eigenvalue equal to the number
of submanifolds.
2. The basis vectors for each disentangled subspace around one point.
3. The ground truth basis vectors for the disentangled subspaces at the same
point. If `--rotate=False`, and GEOMANCER has sufficient data, each basis matrix
should span the same subspace as the results in the second plot.
