# For evaluation only, a nest installation is not required. For generation of data,
# a nest version with changes by the authors is required.
try:
    import nest
except:
    nest = None    
import numpy as np

def readout_rate(multimeter, KernelTime, T, idx, printselection=True):
    '''
    Readout the current rate of the recorded neurons.

    Parameters
    ----------
    multimeter : 
        Multimeter, created by nest. Measures the rate of a number of neurons.

    KernelTime : float
        Current time at begin of simulation.

    T : float
        Timepoint when to readout the rate after start of simulation.

    idx : integer
        Index which neuron to readout the rate from.
        Be careful if you measure the rates of random neuron indices.

    Returns
    -------
    rate : float or array
        Rates of neuron idx at time T.
    '''

    #dictionary: contains senders, times, rates
    data=nest.GetStatus(multimeter)[0]['events']
    dt=nest.GetStatus(multimeter, 'interval')[0]

    #Find the index in the list of senders and write their indices
    sender_indices=data['senders']
    senders=np.where(np.abs(sender_indices-idx)<1)
    
    #Find the indices, where T is closest to
    times=data['times']
    temp_time_idx=np.argmin(np.abs(times[senders]-(T+KernelTime)))
    correct_index=senders[0][temp_time_idx]

    #Tell the chosen time, if the readout time is not allowed

    #Check, if T is allowed
    if printselection:
        close_to_T=False
        for time in data['times']:
            if np.abs(time-(T+KernelTime))<dt/10:
                close_to_T=True
        if not(close_to_T):
            print('There is no timepoint measured close to ', (T+KernelTime), ' ms.')
            print('Chose ', times[correct_index], ' instead.')

    #read out the rate at those timepoints
    all_rates=data['rate']
    rate=all_rates[np.array(correct_index)]

    return rate
