# For evaluation only, a nest installation is not required. For generation of data,
# a nest version with changes by the authors is required.
try:
    import nest
except:
    nest = None    
import numpy as np
import sys

'''
This function creates a network given the specified parameters and sets up the Kernel.
'''


def create_network(N, N_recorded, neuron_dict, multi_dict, neuron_type='lin_rate_ipn', input_type='same', W=None, input_vector=None, neuron_connect_g=0., dt=0.1, setKernel=True):
    '''
    Setup your network: Create nodes and a multimeter and connect them.
    
    Parameters
    ----------
    N : integer
        Size of the network.
    
    N_recorded : integer
        Number of neurons recorded by the multimeter.

    neuron_dict : dictionary
        Dictionary containing the information for neuron creation.
        Contents depend on neuron type.

    multi_dict : dictionary
        Dictionary containing the information for multimeter creation.

    neuron_type : string
        Type of the neurons to be created. Default: linear rate neuron as in
        (d/dt + 1) y(t) = W y(t) + u x(t)

    input_type : string
        'same' or 'different'. Will all neurons get the same or different input?

    W : matrix NxN
        Connectivity matrix. Not to be combined with neuron_connect_g.

    neuron_connect_g : float
        Connection strength of the neurons.

    dt : float
        Resolution of simulation.
        
    setKernel : boolean
        At the first, and only the first, instantiation of nest, the Kernel has
        to be set. Set to False at the second created network and all following.
    
    Returns
    -------
    multimeter : 
        Multimeter created by nest, measuring N_recorded neurons.

    neurons :
        Neurons created by nest.

    rate_generator :
        Step_rate_generator or list of such, depending on input_type
    '''
    
    ############
    #Set Kernel#
    ############

    if setKernel:
        nest.SetKernelStatus({'resolution':dt, 'use_wfr':False, 'print_time':False, 'overwrite_files':True, 'data_path':'./data'})

    ########
    #Create#
    ########
    
    #Create Neurons
    #Ensures working with an older version of nest
    if 'mu' in neuron_dict:
        neuron_dict['mean'] = neuron_dict.pop('mu')
        neuron_dict['std'] = neuron_dict.pop('sigma')
    
    nest.SetDefaults(neuron_type, neuron_dict)
    neurons=nest.Create(neuron_type, N, params=neuron_dict)
    #Reminder concerning the use of polynomial_rate_ipn together with self-written step_rate_generator.
    #As implemented now, x(t) passes through the non-lnearity Phi. Therefore, stimuli x'(t) have to be
    #passed such that the Phi(x'(t)) = x(t). This is possible only for relatively small x(t), therefore
    #they have to be preprocessed accordingly. This is due to the way nest is built.
    if neuron_type=='polynomial_rate_ipn':
        if neuron_dict['poly_coeffs'][2]!=0:
            print('Please remember to adapt the stimulus strength to compare simulation with theory')

    #Create Multimeter
    multimeter=nest.Create('multimeter')
    nest.SetStatus(multimeter, multi_dict)

    #Create rate generator
    if input_type=='same':
        rate_generator=nest.Create('step_rate_generator')
    elif input_type=='different':
        rate_generator=[]
        for i in range(N):
            rate_generator.append(nest.Create('step_rate_generator'))
    else:
          sys.exit('The input_type must be either "same" or "different".')
    
    #########
    #Connect#
    #########

    y=np.random.choice(N, size=N_recorded, replace=False)
    if input_vector is None:
        input_vector=np.ones(N)
    for i in range(N):
        for j in range(N):
            if W is None:
                nest.Connect([neurons[j]], [neurons[i]], 'one_to_one', {'weight':np.random.normal(0, neuron_connect_g/np.sqrt(N)), 'model':'rate_connection_delayed', 'delay':dt})
            else:
                nest.Connect([neurons[j]], [neurons[i]], 'one_to_one', {'weight':W[i][j], 'model':'rate_connection_delayed', 'delay':dt})
        if input_type=='different':
            nest.Connect(rate_generator[i], [neurons[i]], 'one_to_one', {'weight':input_vector[i],'model':'rate_connection_delayed', 'delay':dt})
        if input_type=='same':
            nest.Connect(rate_generator, [neurons[i]], 'one_to_one', {'weight':input_vector[i],'model':'rate_connection_delayed', 'delay':dt})
    nest.Connect(multimeter, [neurons[y_i] for y_i in y])
    
    return multimeter, neurons, rate_generator


