# Copyright 2020 Erik Härkönen. All rights reserved.
# This file is licensed to you under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License. You may obtain a copy
# of the License at http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software distributed under
# the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
# OF ANY KIND, either express or implied. See the License for the specific language
# governing permissions and limitations under the License.

# Patch for broken CTRL+C handler
# https://github.com/ContinuumIO/anaconda-issues/issues/905
#
# This code has been modified for GSP applications by XXX.
#

import os
os.environ['FOR_DISABLE_CONSOLE_CTRL_HANDLER'] = '1'

import torch, json, numpy as np
from types import SimpleNamespace
import matplotlib.pyplot as plt
from pathlib import Path
from os import makedirs
from PIL import Image
from netdissect import proggan, nethook, easydict, zdataset
from netdissect.modelconfig import create_instrumented_model
from estimators import get_estimator
from models import get_instrumented_model
from scipy.cluster.vq import kmeans
import re
import sys
import datetime
import argparse
from tqdm import trange
from config import Config
from decomposition import get_random_dirs, get_or_compute, get_max_batch_size, SEED_VISUALIZATION
from utils import pad_frames

def x_closest(p):
    distances = np.sqrt(np.sum((X - p)**2, axis=-1))
    idx = np.argmin(distances)
    return distances[idx], X[idx]

def make_gif(imgs, duration_secs, outname):
    head, *tail = [Image.fromarray((x * 255).astype(np.uint8)) for x in imgs]
    ms_per_frame = 1000 * duration_secs / instances
    head.save(outname, format='GIF', append_images=tail, save_all=True, duration=ms_per_frame, loop=0)

def make_mp4(imgs, duration_secs, outname, scale=None):
    import shutil
    import subprocess as sp

    FFMPEG_BIN = shutil.which("ffmpeg")
    assert FFMPEG_BIN is not None, 'ffmpeg not found, install with "conda install -c conda-forge ffmpeg"'
    assert len(imgs[0].shape) == 3, 'Invalid shape of frame data'

    resolution = imgs[0].shape[0:2]
    fps = int(len(imgs) / duration_secs)

    if scale is None:
        scale_str = f'{resolution[0]}x{resolution[1]}'
    else:
        scale_str = scale

    command = [
        FFMPEG_BIN,
        '-y', # overwrite output file
        '-f', 'rawvideo',
        '-vcodec','rawvideo',
        '-s', f'{resolution[0]}x{resolution[1]}',
        '-pix_fmt', 'rgb24',
        '-r', f'{fps}',
        '-i', '-', # imput from pipe
        '-an', # no audio
        '-c:v', 'libx264',
        '-preset', 'slow',
        '-crf', '17',
        '-g', '5',
        '-s', scale_str, # size of one frame
        str(Path(outname).with_suffix('.mp4'))
        # str(outname)
    ]

    frame_data = np.concatenate([(x * 255).astype(np.uint8).reshape(-1) for x in imgs])


    with sp.Popen(command, stdin=sp.PIPE, stdout=sp.PIPE, stderr=sp.PIPE) as p:
        ret = p.communicate(frame_data.tobytes())
        if p.returncode != 0:
            print(ret[1].decode("utf-8"))
            raise sp.CalledProcessError(p.returncode, command)

def make_grid(
        latent,
        lat_mean,
        lat_comp,
        lat_stdev,
        act_mean,
        act_comp,
        act_stdev,
        scale=1,
        n_rows=10,
        n_cols=5,
        make_plots=True,
        edit_type='latent'
        ):
    from notebooks.notebook_utils import create_strip_centered

    inst.remove_edits()
    x_range = np.linspace(-scale, scale, n_cols, dtype=np.float32) # scale in sigmas

    rows = []
    for r in range(n_rows):
        curr_row = []
        out_batch = create_strip_centered(inst, edit_type, layer_key, [latent],
            act_comp[r], lat_comp[r], act_stdev[r], lat_stdev[r], act_mean, lat_mean, scale, 0, -1, n_cols)[0]
        for i, img in enumerate(out_batch):
            curr_row.append(('c{}_{:.2f}'.format(r, x_range[i]), img))

        rows.append(curr_row[:n_cols])

    inst.remove_edits()

    if make_plots:
        # If more rows than columns, make several blocks side by side
        n_blocks = 2 if n_rows > n_cols else 1

        for r, data in enumerate(rows):
            # Add white borders
            imgs = pad_frames([img for _, img in data])

            coord = ((r * n_blocks) % n_rows) + ((r * n_blocks) // n_rows)
            plt.subplot(n_rows//n_blocks, n_blocks, 1 + coord)
            plt.imshow(np.hstack(imgs))

            # Custom x-axis labels
            W = imgs[0].shape[1] # image width
            P = imgs[1].shape[1] # padding width
            locs = [(0.5*W + i*(W+P)) for i in range(n_cols)]
            plt.xticks(locs, ["{:.2f}".format(v) for v in x_range])
            plt.yticks([])
            plt.ylabel(f'C{r}')

        plt.tight_layout()
        plt.subplots_adjust(top=0.96) # make room for suptitle

    return [img for row in rows for img in row]


######################
### Visualize results
######################

def visualize_component(
        pc_values=[0 for _ in range(70)],
        pc_to_manipulate=0,
        sigma_range=2,
        n_frames=150,
        output_file="test.mp4",
        scale="320x320",
        resample=False,
        seed=0,
        truncation=0.9
    ):
    """
    Visualises a principal component of StyleGAN2 on the face class.

    Parameters
    ----------

    pc_values
        List of length 70, providing the values for each of the 70 principal components.

    pc_to_manipulate
        Integer between 0 and 69, identifying which principal component should be manipulated

    sigma_range
        Positive number, identifying the number of standard deviations that should be explored
        either side of the zero value of the principal component.

    n_frames
        Number of frames in the video to be created.

    output_file
        File to which the output video should be saved.
    """
    global max_batch, sample_shape, feature_shape, inst, args, layer_key, model

    args = Config().from_dict({
        "model": "StyleGAN2",
        "layer": "style",
        "output_class": "ffhq",
        "make_video": True,
        "batch_mode": True,
        "batch_size": 10000,
        "use_w": True,
        "estimator": "ipca",
        "sparsity": 1.0,
        "components": 80,
        "n": 300000,
        "sigma": 2.0,
        "inputs": None,
        "seed": seed
    })

    t_start = datetime.datetime.now()
    timestamp = lambda : datetime.datetime.now().strftime("%d.%m %H:%M")
    print(f'[{timestamp()}] {args.model}, {args.layer}, {args.estimator}')

    # Ensure reproducibility
    torch.manual_seed(0) # also sets cuda seeds
    np.random.seed(0)

    # Speed up backend
    torch.backends.cudnn.benchmark = True
    torch.autograd.set_grad_enabled(False)

    has_gpu = torch.cuda.is_available()
    device = torch.device('cuda' if has_gpu else 'cpu')
    layer_key = args.layer
    layer_name = layer_key #layer_key.lower().split('.')[-1]

    basedir = Path(__file__).parent.resolve()
    outdir = basedir / 'out'

    # Load model
    inst = get_instrumented_model(args.model, args.output_class, layer_key, device, use_w=args.use_w)
    model = inst.model
    feature_shape = inst.feature_shape[layer_key]
    latent_shape = model.get_latent_shape()
    print('Feature shape:', feature_shape)

    # Layout of activations
    if len(feature_shape) != 4: # non-spatial
        axis_mask = np.ones(len(feature_shape), dtype=np.int32)
    else:
        axis_mask = np.array([0, 1, 1, 1]) # only batch fixed => whole activation volume used

    # Shape of sample passed to PCA
    sample_shape = feature_shape*axis_mask
    sample_shape[sample_shape == 0] = 1

    # Load or compute components
    dump_name = get_or_compute(args, inst)
    data = np.load(dump_name, allow_pickle=False) # does not contain object arrays
    X_comp = data['act_comp']
    X_global_mean = data['act_mean']
    X_stdev = data['act_stdev']
    X_var_ratio = data['var_ratio']
    X_stdev_random = data['random_stdevs']
    Z_global_mean = data['lat_mean']
    Z_comp = data['lat_comp']
    Z_stdev = data['lat_stdev']
    n_comp = X_comp.shape[0]
    data.close()

    # Transfer components to device
    tensors = SimpleNamespace(
        X_comp = torch.from_numpy(X_comp).to(device).float(), #-1, 1, C, H, W
        X_global_mean = torch.from_numpy(X_global_mean).to(device).float(), # 1, C, H, W
        X_stdev = torch.from_numpy(X_stdev).to(device).float(),
        Z_comp = torch.from_numpy(Z_comp).to(device).float(),
        Z_stdev = torch.from_numpy(Z_stdev).to(device).float(),
        Z_global_mean = torch.from_numpy(Z_global_mean).to(device).float()
        # Z_location = torch.from_numpy(Z_location).to(device).float()
    )

    transformer = get_estimator(args.estimator, n_comp, args.sparsity)
    tr_param_str = transformer.get_param_str()

    # Compute max batch size given VRAM usage
    max_batch = args.batch_size or (get_max_batch_size(inst, device) if has_gpu else 1)
    print('Batch size:', max_batch)

    def show():
        if args.batch_mode:
            plt.close('all')
        else:
            plt.show()

    print(f'[{timestamp()}] Creating visualizations')

    # Ensure visualization gets new samples
    torch.manual_seed(SEED_VISUALIZATION)
    np.random.seed(SEED_VISUALIZATION)

    # Make output directories
    est_id = f'spca_{args.sparsity}' if args.estimator == 'spca' else args.estimator
    outdir_comp = outdir/model.name/layer_key.lower()/est_id/'comp'
    outdir_inst = outdir/model.name/layer_key.lower()/est_id/'inst'
    outdir_summ = outdir/model.name/layer_key.lower()/est_id/'summ'
    makedirs(outdir_comp, exist_ok=True)
    makedirs(outdir_inst, exist_ok=True)
    makedirs(outdir_summ, exist_ok=True)

    # Measure component sparsity (!= activation sparsity)
    sparsity = np.mean(X_comp == 0) # percentage of zero values in components
    print(f'Sparsity: {sparsity:.2f}')

    def get_edit_name(mode):
        if mode == 'activation':
            is_stylegan = 'StyleGAN' in args.model
            is_w = layer_key in ['style', 'g_mapping']
            return 'W' if (is_stylegan and is_w) else 'ACT'
        elif mode == 'latent':
            return model.latent_space_name()
        elif mode == 'both':
            return 'BOTH'
        else:
            raise RuntimeError(f'Unknown edit mode {mode}')

    # Only visualize applicable edit modes
    if args.use_w and layer_key in ['style', 'g_mapping']:
        edit_modes = ['latent'] # activation edit is the same
    else:
        edit_modes = ['activation', 'latent']

    c = pc_to_manipulate

    # This is just a random sample
    # latent = model.sample_latent(n_samples=1)

    # The original script bases the videos on the mean latent (Z) location.
    # Instead, we base the video on the location provided in the function call.
    # This location was specified in terms of principal component values,
    # we need to convert it to the z basis.
    latent = get_z_from_pc_values(pc_values, tensors.Z_global_mean, tensors.Z_comp, tensors.Z_stdev, seed, model, resample, truncation)

    for edit_mode in edit_modes:
        # frames = make_grid(tensors.Z_location,
        frames = make_grid(latent, tensors.Z_global_mean, tensors.Z_comp[c:c+1, :, :], tensors.Z_stdev[c:c+1], tensors.X_global_mean,
            tensors.X_comp[c:c+1, :, :], tensors.X_stdev[c:c+1], n_rows=1, n_cols=n_frames, scale=sigma_range, make_plots=False, edit_type=edit_mode)
        plt.close('all')

        frames = [x for _, x in frames]
        # frames = frames + frames[::-1]
        make_mp4(frames, 5, output_file, scale=scale)

    print('Done in', datetime.datetime.now() - t_start)

def get_z_from_pc_values(pc_values, z_global_mean, z_components, z_sds, seed, model, resample, truncation):
    if resample:
        z = model.sample_latent(n_samples=1, seed=seed, truncation=truncation)
    else:
        z = z_global_mean.clone().detach()
    for pc_value, z_component, z_sd in zip(pc_values, z_components, z_sds):
        z = z + z_component * pc_value * z_sd
    return z
