#include <stdlib.h>
#include <iostream>
#include <vector>
#include <assert.h>
#include <limits>
#include <fstream>
#include <list>
#include <map>
#include <random>
#include <string.h>

#include "InfInt/InfInt.h"

size_t H; // Number of hypotheses
size_t Q; // Number of queries
size_t N; // Number of thresholds
size_t U; // Maximum utility
size_t D; // Maximum distance
size_t S; // Step size for thresholds

using namespace std;

// pow for integers
// O(exp)
int power(const int &base, const int &exp)
{
  int ans = 1;
  for (int i = 0; i < exp; i++) {
    ans *= base;
  }
  return ans;
}

// used for doubly truncated hypothesis and distance function (F and G)
InfInt truncate(const InfInt &a, const InfInt &b, const InfInt &c)
{
  return max(min(a, b), c) - c;
}

// returns smallest integer i such that i * step >= v
InfInt inline ceil(long double v, long double step)
{
  return  InfInt((long long) (ceil(v / step) + 0.1));
}

// returns largest integer i such that i * step < v
InfInt inline floor(long double v, long double step)
{
  return InfInt((long long) (floor(v / step) + 0.1));
}


class Hypothesis
{
public:
  vector<int> response_to_query;
  vector<int> utility;
  vector<int> distance;

  Hypothesis()
  {
  }

  Hypothesis(vector<int> response_to_query, vector<int> utility, vector<int> distance)
  {
    this->response_to_query = response_to_query;
    this->utility = utility;
    this->distance = distance;
  }

  // Utility function
  // Sum of utilities from queries that this hypothesis is interested in
  InfInt F(const vector<int> &query, const vector<int> &response) const
  {
    long long ans = 0;
    for (size_t i = 0; i < query.size(); i++) {
      if (response_to_query[query[i]] == 1) {
        ans += utility[query[i]];
      }
    }

    return InfInt(ans);
  }

  // Distance function
  // Sum of distances from queries that have different responses
  InfInt G(const vector<int> &query, const vector<int> &response) const
  {
    long long ans = 0;
    for (size_t i = 0; i < query.size(); i++) {
      if (response_to_query[query[i]] != response[i]) {
        ans += distance[query[i]];
      }
    }

    return InfInt(ans);
  }
};

class Dataset
{
public:
  vector<Hypothesis> hypothesis;

  Dataset()
  {
  }

  Dataset(vector<Hypothesis> hypothesis)
  {
    this->hypothesis = hypothesis;
  }

  InfInt inline F(const int &h, const vector<int> &query, const vector<int> &response) const
  {
    return hypothesis[h].F(query, response);
  }
  
  InfInt inline G(const int &h, const vector<int> &query, const vector<int> &response) const
  {
    return hypothesis[h].G(query, response);
  }

  // Definition 4.3
  // This expects alpha and kappa to be padded with zeros (convert from zero-indexing to one-indexing used in paper)
  virtual InfInt F_bar(const vector<int> &query, const vector<int> &response, const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    const size_t H = hypothesis.size();

    InfInt factor = 1;
    for (size_t i = 1; i <= N; i++) {
      factor *= (kappa[i] - kappa[i - 1]);
    }
    // f[i] = \prod_{n=1}^{N}(\kappa_n - \kappa_{n - 1})
    vector<InfInt> f(N + 1, factor);
    for (size_t i = 1; i <= N; i++) {
      f[i] /= (kappa[i] - kappa[i - 1]);
    }

    InfInt ans = 0;
    for (size_t h = 0; h < H; h++) {
      InfInt Fh = F(h, query, response);
      InfInt Gh = G(h, query, response);
      for (size_t i = 1; i <= N; i++) {
        ans += f[i] * (((kappa[i] - kappa[i - 1]) - truncate(Gh, kappa[i], kappa[i - 1])) * truncate(Fh, alpha[i], alpha[i + 1]) + truncate(Gh, kappa[i], kappa[i - 1]) * (alpha[i] - alpha[i + 1]));
      }
    }
    return ans;
  }

  virtual InfInt Fb_max(const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    InfInt m = InfInt(hypothesis.size()) * alpha[1];
    for (size_t i = 1; i <= N; i++) {
      m *= (kappa[i] - kappa[i - 1]);
    }
    return m;
  }
};

class DatasetAlternative : public Dataset
{
public:
  DatasetAlternative(vector<Hypothesis> hypothesis)
  : Dataset(hypothesis)
  {
  }

  // Definition D.1: Alternative definition
  InfInt F_bar(const vector<int> &query, const vector<int> &response, const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    const size_t H = hypothesis.size();

    InfInt ans = 0;
    for (size_t h = 0; h < H; h++) {
      InfInt Fh = F(h, query, response);
      InfInt Gh = G(h, query, response);
      for (size_t i = 1; i <= N; i++) {
        ans += (((kappa[i] - kappa[i - 1]) - truncate(Gh, kappa[i], kappa[i - 1])) * truncate(Fh, alpha[i], alpha[i + 1]) + truncate(Gh, kappa[i], kappa[i - 1]) * (alpha[i] - alpha[i + 1]));
      }
    }
    return ans;
  }

  InfInt Fb_max(const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    InfInt m = 0;
    for (size_t i = 1; i <= N; i++) {
      m += (alpha[i] - alpha[i + 1]) * (kappa[i] - kappa[i - 1]);
    }
    return m * hypothesis.size();
  }
};

class DatasetCircuit : public Dataset
{
public:
  DatasetCircuit(vector<Hypothesis> hypothesis)
  : Dataset(hypothesis)
  {
  }

  // Definition D.6: Direct Reduction of a Monotone Boolean Circuit of Constraints
  InfInt F_bar(const vector<int> &query, const vector<int> &response, const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    const size_t H = hypothesis.size();

    InfInt ans = 0;
    for (size_t h = 0; h < H; h++) {
      InfInt Fh = F(h, query, response);
      InfInt Gh = G(h, query, response);
      for (size_t i = 1; i <= N; i++) {
        ans += ((kappa[i] - min(kappa[i], Gh)) * min(alpha[i], Fh) + min(kappa[i], Gh) * alpha[i]);
      }
    }
    return ans;
  }
  InfInt Fb_max(const vector<InfInt> &alpha, const vector<InfInt> &kappa, const size_t &N) const
  {
    InfInt m = 0;
    for (size_t i = 1; i <= N; i++) {
      m += alpha[i] * kappa[i];
    }
    return m * hypothesis.size();
  }
};

// Structure to hold the results of the greedy algorithm (Algorithm 1)
class GreedyResult
{
public:
  vector<int> query;
  vector<int> response;
  long double cost;

  GreedyResult()
  {
  }

  GreedyResult(vector<int> query, vector<int> response, long double cost)
  {
    this->query = query;
    this->response = response;
    this->cost = cost;
  }
};

// Runs Algorithm 1
// alpha is assumed to have one zero padded at the front and back
// kappa is assumed to have one zero padded at the front
// (the zero at the front shifts from zero-indexing to the one-indexing used
//  in the paper, and alpha_{N+1} is used, so the extra zero at the back is
//  needed)
// Only uses unit cost of queries
GreedyResult greedy(const Dataset &dataset, vector<InfInt> alpha, vector<InfInt> kappa, const Hypothesis &target, const int &Q, const int &R, vector<int> query = vector<int>(), vector<int> response = vector<int>())
{
  long double c = 0;

  vector<bool> used(Q, false);
  for (size_t i = 0; i < query.size(); i++) {
    used[query[i]] = true;
  }

  const size_t N = alpha.size() - 2;
  assert(kappa.size() - 1 == N);

  InfInt Fb_max = dataset.Fb_max(alpha, kappa, N);

  InfInt Fb = dataset.F_bar(query, response, alpha, kappa, N);
  while (Fb < Fb_max) {
    int best_q = -1;
    InfInt change = -1;
    for (int i = 0; i < Q; i++) {
      if (used[i]) {
        continue;
      }

      vector<int> q_prime = query;
      vector<int> r_prime = response;
  
      q_prime.push_back(i);
      r_prime.push_back(-1);

      r_prime[response.size()] = 0;
      InfInt ch = dataset.F_bar(q_prime, r_prime, alpha, kappa, N) - Fb;
      for (int j = 1; j < R; j++) {
        r_prime[response.size()] = j;
        ch = min(ch, dataset.F_bar(q_prime, r_prime, alpha, kappa, N) - Fb);
      }

      if (ch > change) {
        best_q = i;
        change = ch;
      }
    }

    if (best_q == -1) {
      return GreedyResult(query, response, -1);
    }

    used[best_q] = true;
    query.push_back(best_q);
    response.push_back(target.response_to_query[best_q]);
    Fb = dataset.F_bar(query, response, alpha, kappa, N);

    c++;
  }

  return GreedyResult(query, response, c);
}

// Structure to hold the results of a sequence of greedy algorithms
// This is used when satisfying thresholds one at a time
class GreedySequenceResult
{
public:
  vector<GreedyResult> gr;
  long double cost;

  GreedySequenceResult()
  {
  }

  GreedySequenceResult(vector<GreedyResult> gr)
  {
    this->gr = gr;
    cost = 0;
    for (size_t i = 0; i < gr.size(); i++) {
      if (gr[i].cost == -1) { // one of the thresholds could not be completed
        cost = -1;
        break;
      }
      cost += gr[i].cost;
    }
  }
};


// Satisfies thresholds one at a time
// perm specifies the order that the thresholds are done in
GreedySequenceResult greedy_sequence(const Dataset &dataset, vector<InfInt> alpha, vector<InfInt> kappa, const Hypothesis &target, const int &Q, const int &R, vector<size_t> perm)
{
  vector<GreedyResult> gr;
  vector<int> query;
  vector<int> response;
  for (size_t i = 0; i < perm.size(); i++) {
    vector<InfInt> a;
    vector<InfInt> k;

    a.push_back(0);
    k.push_back(0);
    a.push_back(alpha[perm[i] + 1]);
    k.push_back(kappa[perm[i] + 1]);
    a.push_back(0);

    gr.push_back(greedy(dataset, a, k, target, Q, R, query, response));
    query = gr[i].query;
    response = gr[i].response;
  }

  return GreedySequenceResult(gr);
}

int main(int argc, char *argv[])
{
  if (argc != 8) {
    cout << "Usage: " << argv[0] << " <|H|> <Queries> <Thresholds> <Maximum Utility> <Maximum Distance> <Step Size> <Seed>\n";
    return 0;
  }

  H = atoi(argv[1]); // Number of hypotheses
  Q = atoi(argv[2]); // Number of queries
  N = atoi(argv[3]); // Number of thresholds
  U = atoi(argv[4]); // Maximum utility
  D = atoi(argv[5]); // Maximum distance
  S = atoi(argv[6]); // Step size for thresholds

  srand(atoi(argv[7])); // Seed random number generator


  vector<Hypothesis> hypothesis;
  vector<int> response_to_query(Q, 0);
  vector<int> utility(Q, 0);
  vector<int> distance(Q, 0);
  for (size_t i = 0; i < H; i++) {
    for (size_t j = 0; j < Q; j++) {
      response_to_query[j] = rand() % 2; // random "yes" or "no" response to each query
      utility[j] = rand() % (U + 1);     // utility added when hypothesis is interested
      distance[j] = rand() % (D + 1);    // distance added when responses do not match
    }
    hypothesis.push_back(Hypothesis(response_to_query, utility, distance));
  }
  const Dataset dataset(hypothesis);
  const DatasetAlternative dataset2(hypothesis);
  const DatasetCircuit dataset3(hypothesis);

  // Thresholds are (1 * S, N * S), (2 * S, (N - 1) * S), ..., (N * S, 1 * S)
  vector<InfInt> alpha;
  vector<InfInt> kappa;
  alpha.push_back(0);
  kappa.push_back(0);
  for (size_t n = 1; n <= N; n++) {
    alpha.push_back(S * (N - n + 1));
    kappa.push_back(S * n);
  }
  alpha.push_back(0);


  for (size_t i = 0; i < Q; i++) {
    response_to_query[i] = rand() % 2;
    utility[i] = rand() % (U + 1);
    distance[i] = rand() % (D + 1);
  }
  const Hypothesis target(response_to_query, utility, distance);
  const int R = 2; // number of responses (just "yes" or "no")

  // Run greedy algorithm
  GreedyResult result = greedy(dataset, alpha, kappa, target, Q, R);
  cout << result.cost << "\n";

  // Run greedy algorithm (alternative)
  GreedyResult result2 = greedy(dataset2, alpha, kappa, target, Q, R);
  cout << result2.cost << "\n";

  // Run greedy algorithm (circuit)
  GreedyResult result3 = greedy(dataset3, alpha, kappa, target, Q, R);
  cout << result3.cost << "\n";

  // Run forwards sequence
  vector<size_t> perm1;
  for (size_t i = 0; i < N; i++) {
    perm1.push_back(i);
  }
  GreedySequenceResult gsr1 = greedy_sequence(dataset, alpha, kappa, target, Q, R, perm1);
  cout << gsr1.cost << "\n";

  // Run backwards sequence
  vector<size_t> perm2;
  for (size_t i = 0; i < N; i++) {
    perm2.push_back(N - i - 1);
  }
  GreedySequenceResult gsr2 = greedy_sequence(dataset, alpha, kappa, target, Q, R, perm2);
  cout << gsr2.cost << "\n";

}

