import sys
sys.path.append(r'./document_generation/')
import torch

device = "cuda" if torch.cuda.is_available() else "cpu"

from simctg import SimCTG
from transformers import AutoTokenizer
# load model and tokenizer
model_path = r'./simctg_wikitext103'
tokenizer = AutoTokenizer.from_pretrained(model_path)
model = SimCTG(model_path, tokenizer.pad_token_id).to(device)
model.eval()

with torch.no_grad():
    # prepare prefix input
    text = r"Butt criticized Donald 's controls in certain situations in the game , as well as the difficulty of some levels and puzzles . Buchanan also criticized the controls , calling"
    tokens = tokenizer.tokenize(text)
    input_ids = tokenizer.convert_tokens_to_ids(tokens)
    input_ids = torch.LongTensor(input_ids).view(1,-1).to(device)

    print ('------ Prefix is ------')
    print (text)
    print ('-----------------------')
    print ('\n')

    print ('------ Contrastive Search Result ------')
    # use contrastive search to generate the result
    beam_width, alpha, decoding_len = 8, 0.6, 128
    output = model.fast_contrastive_search(input_ids, beam_width, alpha, decoding_len)
    print("Output:\n" + 100 * '-')
    print(tokenizer.decode(output))
    print ('-----------------------')
    print ('\n')

    '''
        ------ Contrastive Search Result ------
        Output:
        ----------------------------------------------------------------------------------------------------
        Butt criticized Donald's controls in certain situations in the game, as well as the difficulty of some levels and puzzles. 
        Buchanan also criticized the controls, calling them " unimpressive " and a " nightmare " of an experience to play with 
        players unfamiliar with Tetris. On the other hand, his opinion was shared by other reviewers, and some were critical of 
        the game's technical design for the Wii version of Tetris. In addition, Tintin's review included a quote from Roger Ebert, 
        who said that Tetris was better than the original game due to its simplicity and ease of play. Ebert's comments were 
        included in the game's DVD commentary, released on March 22, 2010. It is unclear if any of the video commentary was taken 
        from the DVD
        -----------------------
    '''

    print ('------ Greedy Search Result ------')
    decoding_len = 128
    output = model.greedy_search(input_ids, decoding_len)
    print("Output:\n" + 100 * '-')
    print(tokenizer.decode(output))
    print ('-----------------------')
    print ('\n')

    '''
        ------ Greedy Search Result ------
        Setting `pad_token_id` to `eos_token_id`:50256 for open-end generation.
        Output:
        ----------------------------------------------------------------------------------------------------
        Butt criticized Donald's controls in certain situations in the game, as well as the difficulty of some levels and puzzles. 
        Buchanan also criticized the controls, calling them " too easy " and " too difficult ". He also criticized the game's 
        graphics, saying that the game's graphics were " too simplistic " and " too simplistic ". He also criticized the game's 
        music, saying that it was " too loud " and " too loud ". He also criticized the game's music, saying that it was " too loud " 
        and " too loud ". He also criticized the game's graphics, saying that the game's graphics were " too simplistic " and 
        " too simplistic ". He also criticized the game's music, saying that the game's graphics were
        -----------------------
    '''

    print ('------ Beam Search Result ------')
    beam_width, decoding_len = 10, 128
    output = model.beam_search(input_ids, beam_width, decoding_len)
    print("Output:\n" + 100 * '-')
    print(tokenizer.decode(output))
    print ('-----------------------')
    print ('\n')

    '''
        ------ Beam Search Result ------
        Setting `pad_token_id` to `eos_token_id`:50256 for open-end generation.
        Output:
        ----------------------------------------------------------------------------------------------------
        Butt criticized Donald's controls in certain situations in the game, as well as the difficulty of some levels and puzzles. 
        Buchanan also criticized the controls, calling them " too easy " and " too difficult ". He also criticized the game's graphics, 
        stating that the game's graphics were " too simplistic " and that the game's graphics were " too simplistic ". He also criticized 
        the game's music, stating that the game's music was " too loud " and that the game's music was " too loud ". He also criticized 
        the game's graphics, stating that the game's graphics were " too simplistic " and that the game's graphics were " too simplistic ". 
        He also criticized the game's music, stating that the game's
        -----------------------
    '''

    print ('------ Nucleus Sampling Result ------')
    nucleus_p, decoding_len = 0.95, 128
    output = model.nucleus_sampling(input_ids, nucleus_p, decoding_len)
    print("Output:\n" + 100 * '-')
    print(tokenizer.decode(output))
    print ('-----------------------')

    '''
        The result of nucleus sampling is different at every run due to its stochasticity.
    '''
