# DenseNet

**DenseNet** is a type of convolutional neural network that utilises dense connections between layers, through [Dense Blocks](http://www.paperswithcode.com/method/dense-block), where we connect *all layers* (with matching feature-map sizes) directly with each other. To preserve the feed-forward nature, each layer obtains additional inputs from all preceding layers and passes on its own feature-maps to all subsequent layers.

The **DenseNet Blur** variant in this collection by Ross Wightman employs [Blur Pooling](http://www.paperswithcode.com/method/blur-pooling)

## How do I use this model on an image?
To load a pretrained model:

```python
import external.timm.pytorch_image_models.timm

model = external.resnet_strikes.pytorch_image_models.timm.create_model('densenet121', pretrained=True)
model.eval()
```

To load and preprocess the image:
```python 
import urllib
from PIL import Image
from external.resnet_strikes.pytorch_image_models.timm.data import resolve_data_config
from external.resnet_strikes.pytorch_image_models.timm.data.transforms_factory import create_transform

config = resolve_data_config({}, model=model)
transform = create_transform(**config)

url, filename = ("https://github.com/pytorch/hub/raw/master/images/dog.jpg", "dog.jpg")
urllib.request.urlretrieve(url, filename)
img = Image.open(filename).convert('RGB')
tensor = transform(img).unsqueeze(0) # transform and add batch dimension
```

To get the model predictions:
```python
import torch
with torch.no_grad():
    out = model(tensor)
probabilities = torch.nn.functional.softmax(out[0], dim=0)
print(probabilities.shape)
# prints: torch.Size([1000])
```

To get the top-5 predictions class names:
```python
# Get imagenet class mappings
url, filename = ("https://raw.githubusercontent.com/pytorch/hub/master/imagenet_classes.txt", "imagenet_classes.txt")
urllib.request.urlretrieve(url, filename) 
with open("imagenet_classes.txt", "r") as f:
    categories = [s.strip() for s in f.readlines()]

# Print top categories per image
top5_prob, top5_catid = torch.topk(probabilities, 5)
for i in range(top5_prob.size(0)):
    print(categories[top5_catid[i]], top5_prob[i].item())
# prints class names and probabilities like:
# [('Samoyed', 0.6425196528434753), ('Pomeranian', 0.04062102362513542), ('keeshond', 0.03186424449086189), ('white wolf', 0.01739676296710968), ('Eskimo dog', 0.011717947199940681)]
```

Replace the model name with the variant you want to use, e.g. `densenet121`. You can find the IDs in the model summaries at the top of this page.

To extract image features with this model, follow the [timm feature extraction examples](https://rwightman.github.io/pytorch-image-models/feature_extraction/), just change the name of the model you want to use.

## How do I finetune this model?
You can finetune any of the pre-trained models just by changing the classifier (the last layer).
```python
model = external.resnet_strikes.pytorch_image_models.timm.create_model('densenet121', pretrained=True, num_classes=NUM_FINETUNE_CLASSES)
```
To finetune on your own dataset, you have to write a training loop or adapt [timm's training
script](https://github.com/rwightman/pytorch-image-models/blob/master/train.py) to use your dataset.

## How do I train this model?

You can follow the [timm recipe scripts](https://rwightman.github.io/pytorch-image-models/scripts/) for training a new model afresh.

## Citation

```BibTeX
@article{DBLP:journals/corr/HuangLW16a,
  author    = {Gao Huang and
               Zhuang Liu and
               Kilian Q. Weinberger},
  title     = {Densely Connected Convolutional Networks},
  journal   = {CoRR},
  volume    = {abs/1608.06993},
  year      = {2016},
  url       = {http://arxiv.org/abs/1608.06993},
  archivePrefix = {arXiv},
  eprint    = {1608.06993},
  timestamp = {Mon, 10 Sep 2018 15:49:32 +0200},
  biburl    = {https://dblp.org/rec/journals/corr/HuangLW16a.bib},
  bibsource = {dblp computer science bibliography, https://dblp.org}
}
```

```
@misc{rw2019timm,
  author = {Ross Wightman},
  title = {PyTorch Image Models},
  year = {2019},
  publisher = {GitHub},
  journal = {GitHub repository},
  doi = {10.5281/zenodo.4414861},
  howpublished = {\url{https://github.com/rwightman/pytorch-image-models}}
}
```

<!--
Type: model-index
Collections:
- Name: DenseNet
  Paper:
    Title: Densely Connected Convolutional Networks
    URL: https://paperswithcode.com/paper/densely-connected-convolutional-networks
Models:
- Name: densenet121
  In Collection: DenseNet
  Metadata:
    FLOPs: 3641843200
    Parameters: 7980000
    File Size: 32376726
    Architecture:
    - 1x1 Convolution
    - Average Pooling
    - Batch Normalization
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Techniques:
    - Kaiming Initialization
    - Nesterov Accelerated Gradient
    - Weight Decay
    Training Data:
    - ImageNet
    ID: densenet121
    LR: 0.1
    Epochs: 90
    Layers: 121
    Dropout: 0.2
    Crop Pct: '0.875'
    Momentum: 0.9
    Batch Size: 256
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L295
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-weights/densenet121_ra-50efcf5c.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 75.56%
      Top 5 Accuracy: 92.65%
- Name: densenet161
  In Collection: DenseNet
  Metadata:
    FLOPs: 9931959264
    Parameters: 28680000
    File Size: 115730790
    Architecture:
    - 1x1 Convolution
    - Average Pooling
    - Batch Normalization
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Techniques:
    - Kaiming Initialization
    - Nesterov Accelerated Gradient
    - Weight Decay
    Training Data:
    - ImageNet
    ID: densenet161
    LR: 0.1
    Epochs: 90
    Layers: 161
    Dropout: 0.2
    Crop Pct: '0.875'
    Momentum: 0.9
    Batch Size: 256
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L347
  Weights: https://download.pytorch.org/models/densenet161-8d451a50.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 77.36%
      Top 5 Accuracy: 93.63%
- Name: densenet169
  In Collection: DenseNet
  Metadata:
    FLOPs: 4316945792
    Parameters: 14150000
    File Size: 57365526
    Architecture:
    - 1x1 Convolution
    - Average Pooling
    - Batch Normalization
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Techniques:
    - Kaiming Initialization
    - Nesterov Accelerated Gradient
    - Weight Decay
    Training Data:
    - ImageNet
    ID: densenet169
    LR: 0.1
    Epochs: 90
    Layers: 169
    Dropout: 0.2
    Crop Pct: '0.875'
    Momentum: 0.9
    Batch Size: 256
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L327
  Weights: https://download.pytorch.org/models/densenet169-b2777c0a.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 75.9%
      Top 5 Accuracy: 93.02%
- Name: densenet201
  In Collection: DenseNet
  Metadata:
    FLOPs: 5514321024
    Parameters: 20010000
    File Size: 81131730
    Architecture:
    - 1x1 Convolution
    - Average Pooling
    - Batch Normalization
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Techniques:
    - Kaiming Initialization
    - Nesterov Accelerated Gradient
    - Weight Decay
    Training Data:
    - ImageNet
    ID: densenet201
    LR: 0.1
    Epochs: 90
    Layers: 201
    Dropout: 0.2
    Crop Pct: '0.875'
    Momentum: 0.9
    Batch Size: 256
    Image Size: '224'
    Weight Decay: 0.0001
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L337
  Weights: https://download.pytorch.org/models/densenet201-c1103571.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 77.29%
      Top 5 Accuracy: 93.48%
- Name: densenetblur121d
  In Collection: DenseNet
  Metadata:
    FLOPs: 3947812864
    Parameters: 8000000
    File Size: 32456500
    Architecture:
    - 1x1 Convolution
    - Batch Normalization
    - Blur Pooling
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Data:
    - ImageNet
    ID: densenetblur121d
    Crop Pct: '0.875'
    Image Size: '224'
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L305
  Weights: https://github.com/rwightman/pytorch-image-models/releases/download/v0.1-weights/densenetblur121d_ra-100dcfbc.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 76.59%
      Top 5 Accuracy: 93.2%
- Name: tv_densenet121
  In Collection: DenseNet
  Metadata:
    FLOPs: 3641843200
    Parameters: 7980000
    File Size: 32342954
    Architecture:
    - 1x1 Convolution
    - Average Pooling
    - Batch Normalization
    - Convolution
    - Dense Block
    - Dense Connections
    - Dropout
    - Max Pooling
    - ReLU
    - Softmax
    Tasks:
    - Image Classification
    Training Techniques:
    - SGD with Momentum
    - Weight Decay
    Training Data:
    - ImageNet
    ID: tv_densenet121
    LR: 0.1
    Epochs: 90
    Crop Pct: '0.875'
    LR Gamma: 0.1
    Momentum: 0.9
    Batch Size: 32
    Image Size: '224'
    LR Step Size: 30
    Weight Decay: 0.0001
    Interpolation: bicubic
  Code: https://github.com/rwightman/pytorch-image-models/blob/d8e69206be253892b2956341fea09fdebfaae4e3/timm/models/densenet.py#L379
  Weights: https://download.pytorch.org/models/densenet121-a639ec97.pth
  Results:
  - Task: Image Classification
    Dataset: ImageNet
    Metrics:
      Top 1 Accuracy: 74.74%
      Top 5 Accuracy: 92.15%
-->