import logging
import logging.config

from .file_ops import ensure_dir_exists
from .arg_parsing import ArgumentRegistry


_logging_format = '%(asctime)s - %(levelname)-8s - %(name)-8s - %(message)s'
_logging_datefmt = '%y-%b-%d_%H:%M:%S'
_logging_choices = ['NOTSET','DEBUG','INFO','WARNING','ERROR','FATAL']


_iter_stride , _def_itrstrd = 'iter_stride', 1
_file_level, _def_fllvl     = 'file_level' , _logging_choices[1]
_console_level, _def_csllvl = 'console_level', _logging_choices[1]
_logfile_name, _def_lgfil   = 'logfile_name', ''


def getname_logging_iter_stride():
    return get_logging_args_registry().get_arg_fullname(_iter_stride)

def getname_logging_file_level():
    return get_logging_args_registry().get_arg_fullname(_file_level)

def getname_logging_console_level():
    return get_logging_args_registry().get_arg_fullname(_console_level)

def get_logging_args_registry( flvl=_def_fllvl, clvl=_def_csllvl):

    _log_arg_reg = ArgumentRegistry('logging') 
    _log_arg_reg.register_str_arg(_file_level,'logging level for file handler',
                                  flvl, _logging_choices)
    _log_arg_reg.register_str_arg(_console_level,'logging level for console handler',
                                  clvl, _logging_choices)
    _log_arg_reg.register_str_arg(_logfile_name,'log filename level for file handler',
                                  _def_lgfil)

    _log_arg_reg.register_int_arg(_iter_stride, 'number of iters to log after', 
                                  _def_itrstrd)

    return _log_arg_reg



def _logging_dict(conloglevel='DEBUG', handlersdict=None,
                  logfilename=None, fileloglevel='DEBUG'):

    retval= {}
    
    retval['version']= 1
    retval['disable_existing_loggers'] = False
    retval['formatters'] = { 
        'standard': { 
            'format': _logging_format,
            'datefmt': _logging_datefmt
        },
    }
            
    retval['handlers'] = { 
        'default': { 
            'level': conloglevel,
            'formatter': 'standard',
            'class': 'logging.StreamHandler',
            'stream'  : 'ext://sys.stdout'
        },
    }
    handlers=['default']

    # calculate the lower of the two logging levles
    clv=_logging_choices.index(conloglevel)
    tlv=clv
    
    if (logfilename is not None) and (len(logfilename.strip())>0):

        # ensure htis logfile's dir exists
        ensure_dir_exists(logfilename) 

        flv=_logging_choices.index(fileloglevel)
        tlv = min(tlv,flv)
        
        retval['handlers'].update({ 
            'logfile' :{
                'level' : fileloglevel,                            
                'class' : 'logging.FileHandler',
                'formatter': 'standard',
                'filename' : logfilename,
                'mode' : 'w'
                #NEEDED only for rotatingfilehandler
                #'maxBytes: 1024 
                # backupCount: 3
            }})
        handlers.append('logfile')
        

    toploglevel = _logging_choices[tlv]

    retval['loggers'] = { 
            # root logger. Note that all children inherit properties 
            # unless specified otw.
        '': { 
            'handlers': handlers,
            'level': toploglevel,
            'propagate': False  # default is True, causes too many mesgs
        }
    }

    if handlersdict is not None:
        retval['loggers'].update(handlersdict)
        
    return retval
    

def configure_logging(handlersdict=None, arg_dict=None, conlvl='DEBUG',
                      logfile=None, fillvl='DEBUG' ):
    '''
    If you are running this file through a the CLI or a GUI/IDE such as spyder, note that
    the logging system gets initialized only ONCE per console you have open. 
    So rerunning the file via runfile(...) or debugfile(...) will not absorb
    any changes to the log parameters; you need to restart the console. Of course,
    if you are running by calling this as a script, no such issue crops up.
    
    the 'morelgdict should have stucture like this:
    # simple level for all xtrainer children                            
    {
    'xtrainer': { 'level': 'DEBUG' },
            # at 'DEBUG' the line search will spill out each step it takes... quite tedius unless
            # debugging
        'xtrainer.optim.line_search': {'level':'WARNING'},
            #'xtrainer.optim.qnhess': {'level':'DEBUG'},
     }       

    '''
    
    if arg_dict is not None:
        _log_arg_reg = get_logging_args_registry()
        fillvl = arg_dict[_log_arg_reg.get_arg_fullname(_file_level)]
        conlvl = arg_dict[_log_arg_reg.get_arg_fullname(_console_level)]
        logfile = arg_dict[_log_arg_reg.get_arg_fullname(_logfile_name)]
    
        
    lgdict = _logging_dict(conlvl, handlersdict, logfile,fillvl)
    #print("lgreg is now: ",lgdict)
    
    logging.config.dictConfig(lgdict)

    #        numeric_level = getattr(logging, args.log.upper(), None)
    #        if not isinstance(numeric_level, int):
    #            raise ValueError('Invalid log level: %s' % args.log)
    #        handlerstreams = []
    #        h1 = logging.StreamHandler(sys.stdout)
    #        handlerstreams.append(h1)
    #        
    #        h2 = logging.FileHandler(logfilename, mode='w', encoding=None, delay=False)
    #        handlerstreams.append(h2)
    #        logging.basicConfig(format=fmt,datefmt='%y-%b-%d_%H:%M:%S',level=numeric_level,handlers=handlerstreams)

