#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Apr 15 00:44:23 2020

Base class for objectives. Abs stands for Abstract.

"""

class AbsObjective(object): 
    '''
    This class sets up the interface that objectives need to satisfy

    Many of the member functions interchange data in objects that can be concretized 
    as needed by subclasses. E.g. one may wish to use pytorch vectors or may stick
    to numpy arrays.
    '''
    
    def initialize(self):
        '''
        Do any initialization stuff here.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")
          
    def get_blank_of_itersize(self):
        ''' Get a blank data array of the same dimn as the iterate. Could be used
        to store a copy or iterate or gradient or any other such.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")

    def copy_iterate_to(self, dest):
        ''' deep copy iterate to iterable dest. Objective may use whatever form it wishes
        to, e.g. numpay array or pytorch vector.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")

    def set_iterate_from(self, itr):
        ''' Deep copy the values in 'itr' to internal representation of the iterate.
        This helps in initialization or in replacing iterates mid algo etc.
        '''        
        raise NotImplementedError("this is an abstract class defining the interface.")

    def evaluate_fn_and_derivatives(self, skip_deriv=False):
        ''' evaluate function value, the gradient, and higher derivatives if available. 
        RETURN float value of objective, but NEEDNOT return gardient or higher derivatives, will be 
         queried for separately'''

        raise NotImplementedError("this is an abstract class defining the interface.")
        
#    def evaluate_gradient(self):
#        raise NotImplementedError("this is an abstract class defining the interface.")
    
    def get_gradient_dot(self, dirn):
        '''
        Get the outcome of a dot product of curr gradient with this dirn.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")
        
        
    def copy_gradient_to(self, dest, mult):
        ''' deep copy gradient to iterable dest, and also apply multiplier 'mult' while
        at it. For e.g., in minimization, user may wish to take negative gradient.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")
        
    def get_hessian_inverse_vector_prod(self, vec, prod):
        ''' get the product of the hessian inverse and the input 'vec'. Copy it to the 
        output 'prod'. This will raise an implementation error if not available.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")
        

    def add_step_along_dirn(self, stlen, dirn):
        '''
        set the internal representation of the iterate to a point that is
        'stlen' along the 'dirn' from curr val.
        '''
        raise NotImplementedError("this is an abstract class defining the interface.")
        
    def set_samples(self, samples):
        '''
        set this new batch of samples in the objective. Called by stochastic optim methods.
        Subclasses interpret as they wish. DEfault impl provided to avoid confusing subclasses
        that don't need this.
        '''
        pass
    
# +++ an implementation of data classes using numpy arrays
import numpy as np
    
class NumpyObjective(AbsObjective):
    '''
    This class uses numpy 1-dimensional arrays to represent all vectors.
    The evaluate* functions are all left unimplemented.
    '''
    def __init__(self, initdimn):
        
        self.dimen = initdimn
        self.iterate = self.get_blank_of_itersize()
        self.gradient = self.get_blank_of_itersize()
        
        
    def get_blank_of_itersize(self):
        return np.zeros((self.dimen,))
    
    def copy_iterate_to(self, dest):
        dest[:] = self.iterate
        
    def copy_gradient_to(self, dest, mult):
        dest[:] = mult * self.gradient

    def set_iterate_from(self, itr):
        self.iterate[:]=itr
        
    def get_gradient_dot(self, dirn):
        return self.gradient.dot(dirn)
    
    def add_step_along_dirn(self, stsz, dirn):
        self.iterate[:] += stsz * dirn
               


# +++ an example implementation 
class DetRosenbrock(NumpyObjective):
    r'''
    This class defines the stochastic rosenbrock function.
    
    The deterministic rosenbrock function is :
          (1.0 - u)**2 + K * (v - u**2)**2
    
    where I call the 'K' parameter the conditioning factor. The larger it is,
    the more pronounced the banana is.
    
    Note: no matther what the direction of optimization is, Newton's method iterations are
    ALWAYS x_{k+1} = x_k  -  (f'(x_k) / f''(x_k)) ,
    So, only change the sign of the gradient when maximizing.
    '''

    def __init__(self,condnum=10.0, init=[0,0], ismx=False):
        '''
        The user needs to wrap any rng in a lambda that returns one sample from
        the distribution. Note that python ranom needs you to enter distribution
        parameters every time a sample is desired. Please handle that outside. 
        E.g. rng = lambda : random.uniform(2.0,3.0)
        this f-object returns a sample uniformly betn 2.0 and 3.0 whenever rng() 
        is called.
        '''
        super(DetRosenbrock, self).__init__(len(init))
        
        self.condition_number=condnum
        self.set_iterate_from(init)
        self.isMinimization = not ismx
        
        self.hessian = np.zeros((2,2))
        
    def initialize(self):
        pass

    def evaluate_fn_and_derivatives(self, skip_deriv=False):
        cterm = self.condition_number
        u , v = self.iterate
        term1=1.0 - u 
        term2=(v - pow(u,2.0))
        fnval=pow(term1, 2.0) +  pow(term2, 2.0) * cterm
        
        if not self.isMinimization: 
            fnval *= -1.

        if not skip_deriv:
            # gradient next
            self.gradient[0] = -2.0*term1
            self.gradient[0] += -4.0* u * term2 * cterm
            self.gradient[1] = 2.0*term2 * cterm
        
            # finally, hessian
            self.hessian[1,1] = 2. * cterm
            self.hessian[0,1] = -2. * cterm * u
            self.hessian[1,0] = self.hessian[0,1]
            self.hessian[0,0] = 2.0 - 4.0*cterm * v +12.0 * cterm * pow(u,2.0) 
        
            if not self.isMinimization: 
                self.hessian *= -1.
                self.gradient *= -1.
            
        return fnval
    
    def get_hessian_inverse_vector_prod(self, vec, prod):
        prod[:]= np.linalg.inv(self.hessian).dot(vec)
    
    
class StochRosenbrock(DetRosenbrock):
    
    r'''
    The stochastic rosenbrock function is :
          (1.0 - u)**2 + K * E * (v - u**2)**2 
    where the additional factor E is provided as a sample, e.g. from a uniform [lo, hi]
    
    The user can control K, lo , hi and the initial point (u_0,v_0) from commandline
    arguments.
    '''
    
    
    def __init__(self,condnum=10.0, init=[0,0],ismx=False):
        '''
        The user needs to wrap any rng in a lambda that returns one sample from
        the distribution. Note that python ranom needs you to enter distribution
        parameters every time a sample is desired. Please handle that outside. 
        E.g. rng = lambda : random.uniform(2.0,3.0)
        this f-object returns a sample uniformly betn 2.0 and 3.0 whenever rng() 
        is called.
        '''
        super(StochRosenbrock, self).__init__(condnum, init, ismx)
        
        self.orig_condition_number=condnum
        self.rngavg=1.0

    def set_samples(self, sampl):
        self.rngavg = np.average(sampl)
        self.condition_number = self.orig_condition_number*self.rngavg

#    def evaluate_fn_and_derivatives(self):
#        u,v = self.iterate
#        term1=1.0- u
#        term2=(v - pow(u,2.0))
#        retval= pow(term1, 2.0) +  pow(term2, 2.0) * cterm
#
#        # gradient
#        self.gradient[0] = -2.0*term1
#        self.gradient[0] += -4.0*u * term2 * cterm
#        self.gradient[1] = 2.0*term2 * cterm
#        
#        if not self.isMinimization: 
#            self.gradient *= -1.
#            retval*=-1.
#
#        return retval
#    
#    def initialize(self):
#        pass
    
        
        
            
