from main import Biggan
from configs import config
import argparse
import torch
from smyrf.torch.utils import color_clusters
import matplotlib.pyplot as plt
import matplotlib.colors as colors
import matplotlib
# Configure matplotlib with Latex font
matplotlib.rcParams['mathtext.fontset'] = 'stix'
matplotlib.rcParams['font.family'] = 'STIXGeneral'

import random
import numpy as np
from categories import indx2category

parser = argparse.ArgumentParser()
parser = argparse.ArgumentParser()
parser.add_argument('--weights_root', default='.')
parser.add_argument('--experiment_name', default='130k')
parser.add_argument('--ema', default=False, action="store_true")
parser.add_argument('--device', default='cuda')
# Good seeds: {200}
parser.add_argument('--seed', type=int)
parser.add_argument('--imagenet_category', default=None)

# Clustering configuration
parser.add_argument('--q_cluster_size', type=int, default=256)

if __name__ == '__main__':
    args = parser.parse_args()
    # parameterize config
    config['experiment_name'] = args.experiment_name
    config['ema'] = args.ema
    config['weights_root'] = args.weights_root
    config['smyrf'] = True
    config['n_hashes'] = 1
    config['q_cluster_size'] = args.q_cluster_size
    config['q_attn_size'] = args.q_cluster_size
    config['k_cluster_size'] = args.q_cluster_size // 4
    config['k_attn_size'] = args.q_cluster_size // 4
    config['return_attn_map'] = True

    print('smyrf biggan...')
    biggan = Biggan(config)
    biggan.load_pretrained()

    # disable smyrf for ground truth images
    print('no smyrf biggan...')
    config['smyrf'] = False
    biggan_no_smyrf = Biggan(config)
    biggan_no_smyrf.load_pretrained()

    # Random sampling
    category2indx = {val: key for key, val in indx2category.items()}
    if args.imagenet_category is not None:
        category = category2indx[args.imagenet_category]
    else:
        category = None

    generator_inputs = biggan.get_random_inputs(bs=1,
                                                target=category,
                                                seed=args.seed)
    print('Sampling...')
    _, (q_pos, k_pos), _ = biggan.sample(generator_inputs, return_attn_map=True)

    out = biggan_no_smyrf.sample(generator_inputs)


    print('Saving image....')
    biggan_no_smyrf.save_images(out[0], '../../visuals/sample_memberships.png')
    print('Saved...')

    q_pos = q_pos.squeeze()
    k_pos = k_pos.squeeze()
    n_clusters = 4096 // args.q_cluster_size
    print(f'Using {n_clusters} clusters...')

    # queries, keys map with annotations
    color_map = color_clusters(q_pos, k_pos, args.q_cluster_size, args.q_cluster_size // 4)

    # 4096
    q_colored = color_map.sum(dim=-1) // (args.q_cluster_size // 4)
    color_map = q_colored.reshape(64, 64)

    print('Plotting...')
    cluster_colors = [x for x in colors.cnames.keys()]
    print(f'Total colors:{len(cluster_colors)}...')
    random.shuffle(cluster_colors)
    cluster_colors.remove('white')
    cmap = colors.ListedColormap(cluster_colors[:n_clusters])


    fig, ax = plt.subplots()
    ax.set_title(f'Clustering memberships for a random image (64x64) generated by a pre-trained BigGAN. \n Total number of clusters: {n_clusters}. \n Each cluster contains {args.q_cluster_size} queries.')
    ax.imshow(color_map.cpu(), cmap=cmap)

    # draw gridlines
    ax.grid(which='major', axis='both', linestyle='-', color='k', linewidth=0.25)
    ax.set_xticks(np.arange(-.5, 64, 1))
    ax.set_yticks(np.arange(-.5, 64, 1))
    ax.set_yticklabels([])
    ax.set_xticklabels([])

    fig.savefig('../../visuals/clusters.png')

    plt.show()
