/*
 * This C++ snipped includes our full implementation of the Link/cut tree (LCT),
 * along with all subroutines necessary for regenerating our training dataset upon them.
 */

#include <cassert>
#include <cstdio>
#include <cstring>
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>

#define MAX_N 100001

using namespace std;

struct lct_node {
  int id;
  double rank;
  int rev;
  int tag;
  lct_node *l, *r, *p, *pp;
};

lct_node** init(int n) {
  lct_node **lct = new lct_node*[n];
  for (int i=0;i<n;i++) {
    lct[i] = new lct_node;
    lct[i] -> id = i;
    lct[i] -> rank = (i * 1.0) / (n * 1.0);
    lct[i] -> rev = 0;
    lct[i] -> tag = 0;
    lct[i] -> l = lct[i] -> r = lct[i] -> p = lct[i] -> pp = NULL;
  }
  return lct;
}

void reset_tags(lct_node **lct, int n) {
  // reinstate all tag values
  for (int i=0;i<n;i++) {
    lct[i] -> tag = 0;
  }
}

void release(lct_node *u) {
  if (u -> rev) {
    // flip left and right children, propagate flip
    lct_node *tmp = u -> l;
    u -> l = u -> r;
    u -> r = tmp;
    if (u -> l != NULL) {
      u -> l -> rev ^= 1;
      u -> l -> tag = 1;
    }
    if (u -> r != NULL) {
      u -> r -> rev ^= 1;
      u -> r -> tag = 1;
    }
    u -> rev = 0;
  }
  u -> tag = 1;
}

void rotate(lct_node *u) {
  lct_node *p = u -> p;
  lct_node *g = p -> p;
  if (p -> l == u) {
    // zig rotation
    p -> l = u -> r;
    if (u -> r != NULL) {
      u -> r -> p = p;
    }
    u -> r = p;
    u -> pp = p -> pp;
    p -> p = u;
    p -> pp = NULL;
  } else if (p -> r == u) {
    // zag rotation
    p -> r = u -> l;
    if (u -> l != NULL) {
      u -> l -> p = p;
    }
    u -> l = p;
    u -> pp = p -> pp;
    p -> p = u;
    p -> pp = NULL;
  } else {
    // should never happen!
    throw "Failed zig rotation!";
  }
  if (g != NULL) {
    // adjust grandparent's child pointer
    if (g -> l == p) {
      g -> l = u;
    } else if (g -> r == p) {
      g -> r = u;
    } else {
      throw "Failed zig rotation!";
    }
  }
  u -> p = g;
}

void splay(lct_node *u) {
  while (u -> p != NULL) {
    lct_node *p = u -> p;
    lct_node *g = p -> p;
    if (g == NULL) {
      release(p);
      release(u);
      // zig rotation
      rotate(u);
    } else {
      release(g);
      release(p);
      release(u);
      if ((g -> l == p) == (p -> l == u)) {
        // zig-zig rotation
        rotate(p);
        rotate(u);
      } else {
        // zig-zag rotation
        rotate(u);
        rotate(u);
      }
    }
  }
  release(u);
}

void expose(lct_node *u) {
  if (u -> p == NULL && u -> pp == NULL) {
    release(u);
    // already the root of the entire tree!
    // cut the right children
    if (u -> r != NULL) {
      u -> r -> pp = u;
      u -> r -> p = NULL;
      u -> r = NULL;
    }
  }
  while (u -> p != NULL || u -> pp != NULL) {
    splay(u);
    lct_node *w = u -> pp;
    // now u is the root of its auxiliary tree
    if (u -> r != NULL) {
      // all deeper (right)-children get cut off to their own aux. tree
      u -> r -> pp = u;
      u -> r -> p = NULL;
      u -> r = NULL;
    }
    if (w != NULL) {
      // attach u onto w's aux. tree
      splay(w);
      if (w -> r != NULL) {
        w -> r -> pp = w;
        w -> r -> p = NULL;
      }
      // wire u to be w's deeper (right) child
      w -> r = u;
      u -> p = w;
      u -> pp = NULL;
    }
  }
}

lct_node* find_root(lct_node *u) {
  // expose the path to u
  expose(u);
  // now the root is the minimum depth of its aux. tree
  lct_node *ret = u;
  while (ret -> l != NULL) {
    ret = ret -> l;
  }
  // re-expose the root to prevent worst-case complexity
  expose(ret);
  return ret;
}

void evert(lct_node *u) {
  expose(u);
  // Remember that u must flip edges to it
  u -> rev ^= 1;
  release(u);
}

void link(lct_node *u, lct_node *v) {
  evert(u);
  expose(v);
  // u can only have right children in its splay-tree, no need to check
  u -> l = v;
  v -> p = u;
  v -> pp = NULL;
}

void cut(lct_node *u) {
  expose(u);
  // u retains all of the deeper nodes and path-parents, so just cut it off
  if (u -> l != NULL) {
    u -> l -> p = NULL;
    u -> l = NULL;
  }
}

void cut(lct_node *u, lct_node *v) {
  // make u the root
  evert(u);
  // then v is guaranteed to be the child
  cut(v);
}

bool conn(lct_node *u, lct_node *v) {
  return (find_root(u) == find_root(v));
}

bool toggle(lct_node *u, lct_node *v) {
  // precondition: arrange u and v by rank
  if (u -> rank > v -> rank) {
    lct_node *tmp = u;
    u = v;
    v = tmp;
  }
  // test if u and v are connected
  bool ret = conn(u, v);
  if (ret) {
    // if they are, cut the edge
    cut(u, v);
  } else {
    // otherwise, link the edge
    link(u, v);
  }
  // report which action was taken
  return ret;
}

lct_node* lca(lct_node *u, lct_node *v) {
  expose(u);
  lct_node *ret = v;
  while (v -> p != NULL || v -> pp != NULL) {
    splay(v);
    lct_node *w = v -> pp;
    // now v is the root of its auxiliary tree
    if (v -> r != NULL) {
      // all deeper (right)-children get cut off to their own aux. tree
      v -> r -> pp = v;
      v -> r -> p = NULL;
      v -> r = NULL;
    }
    if (w != NULL) {
      // attach v onto w's aux. tree
      splay(w);
      ret = w;
      if (w -> r != NULL) {
        w -> r -> pp = w;
        w -> r -> p = NULL;
      }
      // wire u to be w's deeper (right) child
      w -> r = v;
      v -> p = w;
      v -> pp = NULL;
    }
  }

  return ret;
}

vector<vector<int>> lct_state(lct_node **lct, int n) {
  vector<int> rev(n); // to be reverse-propagated?
  vector<int> tag(n); // has this node been "touched"?
  vector<int> ptr(n); // the pointers to imitate
  vector<int> p_type(n); // is the pointer a path-parent? (equiv: is v root of splay tree?)
  vector<int> left(n); // is left child?
  vector<int> right(n); // is right child?
  for (int i=0;i<n;i++) {
    rev[i] = lct[i] -> rev;
    tag[i] = lct[i] -> tag;
    if (lct[i] -> p == NULL && lct[i] -> pp == NULL) {
      ptr[i] = lct[i] -> id;
      p_type[i] = 1;
      left[i] = right[i] = 0;
    } else if (lct[i] -> p == NULL) {
      ptr[i] = lct[i] -> pp -> id;
      p_type[i] = 1;
      left[i] = right[i] = 0;
    } else if (lct[i] -> pp == NULL) {
      ptr[i] = lct[i] -> p -> id;
      p_type[i] = 0;
      if (lct[i] -> p -> l == lct[i]) {
        left[i] = 1;
        right[i] = 0;
      } else if (lct[i] -> p -> r == lct[i]) {
        left[i] = 0;
        right[i] = 1;
      } else {
        throw "Neither L nor R child!";
      }
    } else {
      throw "Both P and PP set!";
    }
  }
  vector<vector<int>> ret = {rev, tag, ptr, p_type, left, right};
  return ret;
}

void print_ptr(lct_node *u) {
  if (u == NULL) {
    cout << "NULL";
  } else {
    cout << u -> id + 1;
  }
}

void print(lct_node *u) {
  cout << "NODE " << u -> id + 1 << endl;
  cout << "REV " << u -> rev << endl;
  cout << "L "; print_ptr(u -> l); cout << endl;
  cout << "R "; print_ptr(u -> r); cout << endl;
  cout << "P "; print_ptr(u -> p); cout << endl;
  cout << "PP "; print_ptr(u -> pp); cout << endl;
}

void print_state(vector<vector<int>> st) {
  for (vector<int> v : st) {
    for (int i : v) {
      cout << i << " ";
    }
    cout << endl;
  }
}

void gen_dataset(char *dname, int nb_graphs, int nb_nodes, int nb_edges) {
  double ****fts = new double***[nb_edges];
  int ***ptrs = new int**[nb_edges];
  double ****ptr_aux = new double***[nb_edges];
  double ***revs = new double**[nb_edges];
  double ***msks = new double**[nb_edges];
  double **ys = new double*[nb_edges];

  for (int e=0;e<nb_edges;e++) {
    fts[e] = new double**[nb_graphs];
    ptrs[e] = new int*[nb_graphs];
    ptr_aux[e] = new double**[nb_graphs];
    revs[e] = new double*[nb_graphs];
    msks[e] = new double*[nb_graphs];
    ys[e] = new double[nb_graphs];

    for (int g=0;g<nb_graphs;g++) {
      fts[e][g] = new double*[nb_nodes];
      ptrs[e][g] = new int[nb_nodes];
      ptr_aux[e][g] = new double*[nb_nodes];
      revs[e][g] = new double[nb_nodes];
      msks[e][g] = new double[nb_nodes];

      for (int u=0;u<nb_nodes;u++) {
        fts[e][g][u] = new double[2];
        ptr_aux[e][g][u] = new double[3];

        fts[e][g][u][0] = 0.0;
        fts[e][g][u][1] = 0.0;

        ptrs[e][g][u] = 0;

        ptr_aux[e][g][u][0] = 0.0;
        ptr_aux[e][g][u][1] = 0.0;
        ptr_aux[e][g][u][2] = 0.0;

        revs[e][g][u] = 0.0;

        msks[e][g][u] = 0.0;
      }
      ys[e][g] = 0.0;
    }
  }

  for (int g=0;g<nb_graphs;g++) {
    lct_node **lct = init(nb_nodes);
    for (int e=0;e<nb_edges;e++) {
      reset_tags(lct, nb_nodes);
      int u = rand() % nb_nodes;
      int v = rand() % nb_nodes;
      while (v == u) {
        v = rand() % nb_nodes;
      }

      if (toggle(lct[u], lct[v])) {
        ys[e][g] = 1.0;
      } else {
        ys[e][g] = 0.0;
      }

      fts[e][g][u][0] = 1.0;
      fts[e][g][v][0] = 1.0;

      vector<vector<int>> cur_state = lct_state(lct, nb_nodes);

      for (int i=0;i<nb_nodes;i++) {
        fts[e][g][i][1] = lct[i] -> rank;
        revs[e][g][i] = cur_state[0][i] * 1.0;
        msks[e][g][i] = cur_state[1][i] * 1.0;
        ptrs[e][g][i] = cur_state[2][i];
        for (int t=0;t<3;t++) {
          ptr_aux[e][g][i][t] = cur_state[3 + t][i] * 1.0;
        }
      }
    }
  }

  char fname[50];
  sprintf(fname, "%s_fts_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream str_fts(fname);
  sprintf(fname, "%s_ptrs_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream ptr_fts(fname);
  sprintf(fname, "%s_ptr_aux_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream aux_fts(fname);
  sprintf(fname, "%s_revs_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream rev_fts(fname);
  sprintf(fname, "%s_msks_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream msk_fts(fname);
  sprintf(fname, "%s_ys_%d_%d_%d.out", dname, nb_graphs, nb_nodes, nb_edges);
  ofstream lbl_fts(fname);

  for (int e=0;e<nb_edges;e++) {
    for (int g=0;g<nb_graphs;g++) {
      for (int u=0;u<nb_nodes;u++) {
        str_fts << fts[e][g][u][0] << " " << fts[e][g][u][1] << endl;
        ptr_fts << ptrs[e][g][u] << " ";
        rev_fts << revs[e][g][u] << " ";
        msk_fts << msks[e][g][u] << " ";
        for (int t=0;t<3;t++) {
          aux_fts << ptr_aux[e][g][u][t] << " ";
        }
        aux_fts << endl;
      }
      str_fts << endl;
      ptr_fts << endl;
      aux_fts << endl;
      rev_fts << endl;
      msk_fts << endl;
      lbl_fts << ys[e][g] << " ";
    }
    str_fts << endl;
    ptr_fts << endl;
    aux_fts << endl;
    rev_fts << endl;
    msk_fts << endl;
    lbl_fts << endl;
  }
}

void lct_oj() { // test script for SPOJ tasks
  int n, m;
  cin >> n >> m;

  lct_node **lct = init(n);

  while (m--) {
    reset_tags(lct, n);
    string cmd;
    cin >> cmd;
    if (cmd == "link" || cmd == "add") {
      int u, v;
      cin >> u >> v;
      link(lct[u - 1], lct[v - 1]);
    } else if (cmd == "cut") {
      int u;
      cin >> u;
      cut(lct[u - 1]);
    } else if (cmd == "rem") {
      int u, v;
      cin >> u >> v;
      cut(lct[u - 1], lct[v - 1]);
    } else if (cmd == "toggle") {
      int u, v;
      cin >> u >> v;
      toggle(lct[u - 1], lct[v - 1]);
    } else if (cmd == "conn") {
      int u, v;
      cin >> u >> v;
      if (conn(lct[u - 1], lct[v - 1])) {
        cout << "YES" << endl;
      } else {
        cout << "NO" << endl;
      }
    } else if (cmd == "lca") {
      int u, v;
      cin >> u >> v;
      cout << lca(lct[u - 1], lct[v - 1]) -> id + 1 << endl;
    } else if (cmd == "expose") {
      int u;
      cin >> u;
      expose(lct[u - 1]);
    } else if (cmd == "splay") {
      int u;
      cin >> u;
      splay(lct[u - 1]);
    } else if (cmd == "rotate") {
      int u;
      cin >> u;
      rotate(lct[u - 1]);
    } else {
      throw "Invalid command!";
    }
  }
}

int main(int argc, char **argv) {
  srand(1337);

  assert(argc % 4 == 1);

  int idd = 1;
  while (idd < argc) {
    char *dname = argv[idd];
    int nb_graphs = atoi(argv[idd + 1]);
    int nb_nodes = atoi(argv[idd + 2]);
    int nb_edges = atoi(argv[idd + 3]);

    gen_dataset(dname, nb_graphs, nb_nodes, nb_edges);

    idd += 4;
  }

  return 0;
}
