#!/usr/bin/env python

# Copyright 2019 The MixMatch Authors.  All rights reserved.

"""Extract and save accuracy to 'stats/accuracy.json'.

The accuracy is extracted from the most recent eventfile.
"""

import glob
import json
import os.path
from absl import app
from absl import flags
import numpy as np
import tensorflow as tf

FLAGS = flags.FLAGS
TAG = 'accuracy'


def summary_dict(accuracies):
    return {
        'last%02d' % x: np.median(accuracies[-x:]) for x in [1, 10, 20, 50]
    }


def main(argv):
    if len(argv) > 2:
        raise app.UsageError('Too many command-line arguments.')
    folder = argv[1]
    matches = sorted(glob.glob(os.path.join(folder, 'tf/events.out.tfevents.*')))
    assert matches, 'No events files found'
    tags = set()
    accuracies = []
    for event_file in matches:
        for e in tf.train.summary_iterator(event_file):
            for v in e.summary.value:
                if v.tag == TAG:
                    accuracies.append(v.simple_value)
                    break
                elif not accuracies:
                    tags.add(v.tag)

    assert accuracies, 'No "accuracy" tag found. Found tags = %s' % tags
    target_dir = os.path.join(folder, 'stats')
    target_file = os.path.join(target_dir, 'accuracy.json')
    if not os.path.isdir(target_dir):
        os.makedirs(target_dir)

    with open(target_file, 'w') as f:
        json.dump(summary_dict(accuracies), f, sort_keys=True, indent=4)
    print('Saved: %s' % target_file)


if __name__ == '__main__':
    app.run(main)
